/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.domain;

import java.util.Vector;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCheckCmd;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;

/**
 * Represents an EPP Domain &lt;check&gt; command, which is used to determine if
 * a domain name is known to the server. The &lt;domain:check&gt; element MUST
 * contain the following child elements:<br>
 * <br>
 *
 * <ul>
 * <li>One or more (up to a maximum of {@code MAX_DOMAINS}) &lt;domain:name&gt;
 * elements that contain the fully qualified name of the queried domains. Use
 * {@code getNames} and {@code setNames} to get and set the elements. Use
 * {@code setName} to set an individual name.</li>
 * </ul>
 *
 * <br>
 * {@code EPPDomainCheckResp} is the concrete {@code EPPReponse} associated with
 * {@code EPPDomainCheckCmd}. <br>
 * <br>
 *
 * @see com.verisign.epp.codec.domain.EPPDomainCheckResp
 */
public class EPPDomainCheckCmd extends EPPCheckCmd {
  /** Maximum number of domains to check at once. */
  public static final int MAX_DOMAINS = 99;

  /**
   * XML local name for {@code EPPDomainCheckCmd}.
   */
  public static final String ELM_LOCALNAME = "check";

  /**
   * XML Element Name of {@code EPPDomainCheckCmd} root element.
   */
  public static final String ELM_NAME = EPPDomainMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

  /** XML Element Name for the {@code names} attribute. */
  private final static String ELM_DOMAIN_NAME = "name";

  /**
   * Domain Names to check. This is a {@code Vector} of {@code String}
   * instances.
   */
  private Vector<String> names;

  /**
   * {@code EPPDomainCheckCmd} default constructor. It will set the names
   * attribute to an empty {@code Vector}.
   */
  public EPPDomainCheckCmd() {
    this.names = new Vector<String>();
  }

  /**
   * {@code EPPDomainCheckCmd} constructor that will check an individual domain
   * name.
   *
   * @param aTransId
   *           Transaction Id associated with command.
   * @param aName
   *           Domain name to check
   */
  public EPPDomainCheckCmd(String aTransId, String aName) {
    super(aTransId);

    this.names = new Vector<String>();
    this.names.addElement(aName);
  }

  /**
   * {@code EPPDomainCheckCmd} constructor that will check a list of domain
   * names.
   *
   * @param aTransId
   *           Transaction Id associated with command.
   * @param someNames
   *           {@code Vector} of domain name {@code String} instances.
   */
  public EPPDomainCheckCmd(String aTransId, Vector<String> someNames) {
    super(aTransId);

    this.names = someNames;
  }


  /**
   * Gets the EPP command Namespace associated with {@code EPPDomainCheckCmd}.
   *
   * @return {@code EPPDomainMapFactory.NS}
   */
  @Override
  public String getNamespace() {
    return EPPDomainMapFactory.NS;
  }

  /**
   * Gets the key for the domain name object, which is the domain name.
   * 
   * @return The domain name if set with a single value; {@code null} otherwise.
   */
  @Override
  public String getKey() {
	  if (this.names != null && this.names.size() == 1) {
		  return (String) this.names.elementAt(0);
	  }
	  else {
		  return null;
	  }
  }
  
  /**
   * Compare an instance of {@code EPPDomainCheckCmd} with this instance.
   *
   * @param aObject
   *           Object to compare with.
   *
   * @return {@code true} if equal; {@code false} otherwise.
   */
  @Override
  public boolean equals(Object aObject) {
    if (!(aObject instanceof EPPDomainCheckCmd)) {
      return false;
    }

    if (!super.equals(aObject)) {
      return false;
    }

    EPPDomainCheckCmd theMap = (EPPDomainCheckCmd) aObject;

    // Domain Names
    if (!EPPUtil.equalVectors(this.names, theMap.names)) {
      return false;
    }

    return true;
  }

  /**
   * Clone {@code EPPDomainCheckCmd}.
   *
   * @return Deep copy clone of {@code EPPDomainCheckCmd}
   *
   * @exception CloneNotSupportedException
   *               standard Object.clone exception
   */
  @Override
  public Object clone() throws CloneNotSupportedException {
    EPPDomainCheckCmd clone = (EPPDomainCheckCmd) super.clone();

    clone.names = (Vector<String>) this.names.clone();

    return clone;
  }

  /**
   * Implementation of {@code Object.toString}, which will result in an
   * indented XML {@code String} representation of the concrete
   * {@code EPPCodecComponent}.
   *
   * @return Indented XML {@code String} if successful; {@code ERROR}
   *         otherwise.
   */
  @Override
  public String toString() {
    return EPPUtil.toString(this);
  }

  /**
   * Sets domain name to check.
   *
   * @param aName
   *           Name to check.
   */
  public void setName(String aName) {
    this.names = new Vector<String>();
    this.names.addElement(aName);
  }

  /**
   * Gets domain names to check.
   *
   * @return Vector of domain name {@code String}'s.
   */
  public Vector<String> getNames() {
    return this.names;
  }

  /**
   * Sets domain names to check.
   *
   * @param someNames
   *           Names to check.
   */
  public void setNames(Vector<String> someNames) {
    this.names = someNames;
  }

  /**
   * Encode a DOM Element tree from the attributes of the
   * {@code EPPDomainCheckCmd} instance.
   *
   * @param aDocument
   *           DOM Document that is being built. Used as an Element factory.
   *
   * @return Element Root DOM Element representing the
   *         {@code EPPDomainCheckCmd} instance.
   *
   * @exception EPPEncodeException
   *               Unable to encode {@code EPPDomainCheckCmd} instance.
   */
  @Override
  protected Element doEncode(Document aDocument) throws EPPEncodeException {
    if (this.names.size() == 0) {
      throw new EPPEncodeException("No domains names specified in EPPDomainCheckCmd");
    }

    if (this.names.size() > MAX_DOMAINS) {
      throw new EPPEncodeException(
            this.names.size() + " domain names is greater than the maximum of " + MAX_DOMAINS);
    }

    Element root = aDocument.createElementNS(EPPDomainMapFactory.NS, ELM_NAME);

    // Names
    EPPUtil.encodeVector(aDocument, root, this.names, EPPDomainMapFactory.NS,
          EPPDomainMapFactory.NS_PREFIX + ":" + ELM_DOMAIN_NAME);

    return root;
  }

  /**
   * Decode the {@code EPPDomainCheckCmd} attributes from the aElement DOM
   * Element tree.
   *
   * @param aElement
   *           Root DOM Element to decode {@code EPPDomainCheckCmd} from.
   *
   * @exception EPPDecodeException
   *               Unable to decode aElement
   */
  @Override
  protected void doDecode(Element aElement) throws EPPDecodeException {
    // Domain Names
    this.names = EPPUtil.decodeVector(aElement, EPPDomainMapFactory.NS, ELM_DOMAIN_NAME);

    if (this.names == null) {
      this.names = new Vector<String>();
    }
  }

}
