/***********************************************************
Copyright (C) 2024 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.interfaces.addlemail;

import java.util.Random;
import java.util.Vector;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.verisign.epp.codec.addlemail.EPPAddlEmail;
import com.verisign.epp.codec.contact.EPPContactAddress;
import com.verisign.epp.codec.contact.EPPContactInfoResp;
import com.verisign.epp.codec.contact.EPPContactPostalDefinition;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPResult;
import com.verisign.epp.interfaces.EPPApplicationSingle;
import com.verisign.epp.interfaces.EPPCommandException;
import com.verisign.epp.interfaces.EPPContact;
import com.verisign.epp.interfaces.EPPSession;
import com.verisign.epp.util.Environment;
import com.verisign.epp.util.TestThread;

import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Is a unit test of the com.verisign.epp.codec.contact package with
 * implementing the Internationalized Email Addresses in the Extensible
 * Provisioning Protocol (EPP).
 */
public class EPPAddlEmailTst {
  /**
   * Handle to the Singleton EPP Application instance
   * ({@code EPPApplicationSingle})
   */
  private static EPPApplicationSingle app = EPPApplicationSingle.getInstance();

  /**
   * Name of configuration file to use for test (default = epp.config).
   */
  private static String configFileName = "epp.config";

  /**
   * Logging category
   */
  private static Logger cat = LoggerFactory.getLogger(EPPAddlEmailTst.class);

  /**
   * EPP Domain associated with test
   */
  private EPPContact contact = null;

  /**
   * EPP Session associated with test
   */
  private EPPSession session = null;

  /**
   * Current test iteration
   */
  private int iteration = 0;

  /**
   * Random instance for the generation of unique objects (hosts, IP addresses,
   * etc.).
   */
  private Random rd = new Random(System.currentTimeMillis());

  /**
   * JUNIT test method to implement the {@code EPPEAITst TestCase}. Each
   * sub-test will be invoked in order to satisfy testing the EPPDomain
   * interface.
   */
  @Test
  public void testContact() {
    int numIterations = 1;

    String iterationsStr = System.getProperty("iterations");

    if (iterationsStr != null) {
      numIterations = Integer.parseInt(iterationsStr);
    }

    for (iteration = 0; (numIterations == 0) || (iteration < numIterations); iteration++) {
      printStart("Test Suite");

      contactCreate();

      contactUpdate();

      contactInfo();

      printEnd("Test Suite");
    }
  }

  /**
   * Unit test of {@code EPPContact.sendCreate} for a contact with an EAI
   * address. There are two create commands sent:
   * <ol>
   * <li>Send a contact create command for a contact with an alternate ASCII email.</li>
   * <li>Send a contact create command for a contact with an alternate primary SMTPUTF8 email.</li>
   * </ol>
   */
  public void contactCreate() {
    printStart("contactCreate");

    EPPResponse response;

    String theName = "AE-1";

    try {
      System.out.println("\n----------------------------------------------------------------");
      System.out.println("contactCreate: Contact create for with ASCII as alternate address for " + theName);
      contact.setTransId("ABC-12345-XYZ");
      contact.setAuthorizationId("ClientXYZ");
      contact.addContactId(theName);
      contact.setVoicePhone("+1.7035555555");
      contact.setVoiceExt("1234");
      contact.setFaxNumber("+1.7035555556");
      contact.setEmail("jdoe@example.com");

      // Streets
      Vector streets = new Vector();
      streets.addElement("123 Example Dr.");
      streets.addElement("Suite 100");

      EPPContactAddress address = new EPPContactAddress();

      address.setStreets(streets);
      address.setCity("Dulles");
      address.setStateProvince("VA");
      address.setPostalCode("20166-6503");
      address.setCountry("US");

      EPPContactPostalDefinition name = new EPPContactPostalDefinition(EPPContactPostalDefinition.ATTR_TYPE_LOC);

      name.setName("John Doe");
      name.setOrg("Example Inc.");
      name.setAddress(address);

      contact.addPostalInfo(name);

      contact.addExtension(new EPPAddlEmail("jdoe-alternate@example.com"));

      response = contact.sendCreate();

      // -- Output all of the response attributes
      System.out.println("contactCreate: Response = [" + response + "]\n\n");
    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    theName = "AE-2";

    try {
      System.out.println("\n----------------------------------------------------------------");
      System.out.println("contactCreate: Contact create with SMTPUTF8 as alternate primary address for " + theName);
      contact.setTransId("ABC-12345-XYZ");
      contact.setAuthorizationId("ClientXYZ");
      contact.addContactId(theName);
      contact.setVoicePhone("+1.7035555555");
      contact.setVoiceExt("1234");
      contact.setFaxNumber("+1.7035555556");
      contact.setEmail("jdoe@example.com");

      // Streets
      Vector streets = new Vector();
      streets.addElement("123 Example Dr.");
      streets.addElement("Suite 100");

      EPPContactAddress address = new EPPContactAddress();

      address.setStreets(streets);
      address.setCity("Dulles");
      address.setStateProvince("VA");
      address.setPostalCode("20166-6503");
      address.setCountry("US");

      EPPContactPostalDefinition name = new EPPContactPostalDefinition(EPPContactPostalDefinition.ATTR_TYPE_INT);

      name.setName("John Doe");
      name.setOrg("Example Inc.");
      name.setAddress(address);

      contact.addPostalInfo(name);
      
      contact.addExtension(new EPPAddlEmail("麥克風@example.com", true));

      response = contact.sendCreate();

      // -- Output all of the response attributes
      System.out.println("contactCreate: Response = [" + response + "]\n\n");
    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    printEnd("contactCreate");
  }

  /**
   * Unit test of {@code EPPContact.sendUpdate} for a contact with an EAI
   * address. There are two update commands sent:
   * <ol>
   * <li>Send a contact update command with an alternate ASCII email.</li>
   * <li>Send a contact update command with an alternate SMTPUTF8 email.</li>
   * <li>Send a contact update command with unset alternate email.</li>
   * </ol>
   */
  public void contactUpdate() {
    printStart("contactUpdate");

    EPPResponse response;

    String theName;

    try {
      theName = "AE-1";
      System.out.println("\n----------------------------------------------------------------");
      System.out.println("\ncontactUpdate: Contact update to set ASCII alternate email for " + theName);
      contact.setTransId("ABC-12345-XYZ");
      contact.addContactId(theName);

      contact.addExtension(new EPPAddlEmail("jdoe-alternate@example.com"));

      response = contact.sendUpdate();

      // -- Output all of the response attributes
      System.out.println("contactUpdate: Response = [" + response + "]\n\n");
    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    try {
      theName = "AE-2";
      System.out.println("\n----------------------------------------------------------------");
      System.out.println("\ncontactUpdate: Contact update to set SMTPUTF8 alternate email for " + theName);
      contact.setTransId("ABC-12345-XYZ");
      contact.addContactId(theName);
      contact.setEmail("jdoe@example.com");

      contact.addExtension(new EPPAddlEmail("麥克風@example.com"));

      response = contact.sendUpdate();

      // -- Output all of the response attributes
      System.out.println("contactUpdate: Response = [" + response + "]\n\n");
    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    try {
      theName = "AE-3";
      System.out.println("\n----------------------------------------------------------------");
      System.out.println("\ncontactUpdate: Contact update to unset alternate email for " + theName);
      contact.setTransId("ABC-12345-XYZ");
      contact.addContactId(theName);

      contact.addExtension(new EPPAddlEmail());

      response = contact.sendUpdate();

      // -- Output all of the response attributes
      System.out.println("contactUpdate: Response = [" + response + "]\n\n");
    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    printEnd("contactUpdate");
  }

  /**
   * Unit test of {@code EPPContact.sendInfo} for a contact with an EAI
   * address. There are two info commands sent:
   * <ol>
   * <li>Send a contact info command for a contact with an alternate ASCII email.</li>
   * <li>Send a contact info command for a contact with an alternate SMTPUTF8 email.</li>
   * <li>Send a contact info command for a contact with an unset alternate email.</li>
   * </ol>
   */
  public void contactInfo() {
    printStart("contactInfo");

    EPPContactInfoResp response;

    String theName;

    // Send a contact info command for a contact with an alternate ASCII email.
    try {
      theName = "AE-1";
      System.out.println("\n----------------------------------------------------------------");
      System.out.println("\ncontactInfo: Contact info for " + theName);
      contact.setTransId("ABC-12345-XYZ");
      contact.addContactId(theName);

      response = contact.sendInfo();

      // -- Output all of the response attributes
      System.out.println("contactInfo: Response = [" + response + "]\n\n");

      // Contact Id
      System.out.println("contactInfo: id = " + response.getId());

      Assert.assertNotNull(response.getEmail());
      Assert.assertTrue(response.hasExtension(EPPAddlEmail.class));

      // Contact E-mail
      System.out.println("contactInfo: email = " + response.getEmail());
      System.out.println("contactInfo: alt email = " + ((EPPAddlEmail) response.getExtension(EPPAddlEmail.class)).getEmail());

    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    // Send a contact info command for a contact with an alternate SMTPUTF8 email.
    try {
      theName = "AE-2";
      System.out.println("\n----------------------------------------------------------------");
      System.out.println("\ncontactInfo: Contact info for " + theName);
      contact.setTransId("ABC-12345-XYZ");
      contact.addContactId(theName);

      response = contact.sendInfo();

      // -- Output all of the response attributes
      System.out.println("contactInfo: Response = [" + response + "]\n\n");

      // Contact Id
      System.out.println("contactInfo: id = " + response.getId());

      Assert.assertNotNull(response.getEmail());
      Assert.assertTrue(response.hasExtension(EPPAddlEmail.class));

      // Contact E-mail
      System.out.println("contactInfo: email = " + response.getEmail());
      System.out.println("contactInfo: alt email = " + ((EPPAddlEmail) response.getExtension(EPPAddlEmail.class)).getEmail());

    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    // Send a contact info command for a contact without an alternate email.
    try {
      theName = "AE-3";

      System.out.println("\n----------------------------------------------------------------");
      System.out.println("\ncontactInfo: Contact info for " + theName);
      contact.setTransId("ABC-12345-XYZ");
      contact.addContactId(theName);

      response = contact.sendInfo();

      // -- Output all of the response attributes
      System.out.println("contactInfo: Response = [" + response + "]\n\n");

      // Contact Id
      System.out.println("contactInfo: id = " + response.getId());

      Assert.assertNotNull(response.getEmail());
      Assert.assertTrue(response.hasExtension(EPPAddlEmail.class));

      // Contact E-mail
      System.out.println("contactInfo: email = " + response.getEmail());
      System.out.println("contactInfo: alt email = " + ((EPPAddlEmail) response.getExtension(EPPAddlEmail.class)).getEmail());

    }
    catch (EPPCommandException e) {
      handleException(e);
    }

    printEnd("contactInfo");
  }

  /**
   * Unit test of {@code EPPSession.initSession}. The session attribute is
   * initialized with the attributes defined in the EPP sample files.
   */
  private void initSession() {
    printStart("initSession");

    // Set attributes for initSession
    session.setTransId("ABC-12345-XYZ");

    session.setVersion("1.0");

    session.setLang("en");

    // Initialize the session
    try {
      session.initSession();
    }

    catch (EPPCommandException e) {
      EPPResponse response = session.getResponse();

      // Is a server specified error?
      if ((response != null) && (!response.isSuccess())) {
        Assert.fail("Server Error : " + response);
      }
      else {
        e.printStackTrace();

        Assert.fail("initSession Error : " + e);
      }
    }

    printEnd("initSession");
  }

  // End EPPEAITst.initSession()

  /**
   * Unit test of {@code EPPSession.endSession}. The session with the EPP
   * Server will be terminated.
   */
  private void endSession() {
    printStart("endSession");

    session.setTransId("ABC-12345-XYZ");

    // End the session
    try {
      session.endSession();
    }

    catch (EPPCommandException e) {
      EPPResponse response = session.getResponse();

      // Is a server specified error?
      if ((response != null) && (!response.isSuccess())) {
        Assert.fail("Server Error : " + response);
      }

      else // Other error
      {
        e.printStackTrace();

        Assert.fail("initSession Error : " + e);
      }
    }

    printEnd("endSession");
  }

  // End EPPEAITst.endSession()

  /**
   * JUNIT {@code setUp} method, which sets the default client Id to
   * "theRegistrar".
   */
  @Before
  public void setUp() {
    try {
      String theSessionClassName = System.getProperty("EPP.SessionClass");

      if (theSessionClassName != null) {
        try {
          Class theSessionClass = Class.forName(theSessionClassName);

          if (!EPPSession.class.isAssignableFrom((theSessionClass))) {
            Assert.fail(theSessionClassName + " is not a subclass of EPPSession");
          }

          session = (EPPSession) theSessionClass.getDeclaredConstructor().newInstance();
        }
        catch (Exception ex) {
          Assert.fail("Exception instantiating EPP.SessionClass value " + theSessionClassName + ": " + ex);
        }
      }
      else {
        session = new EPPSession();
      }

      session.setClientID(Environment.getProperty("EPP.Test.clientId", "ClientX"));
      session.setPassword(Environment.getProperty("EPP.Test.password", "foo-BAR2"));
    }

    catch (Exception e) {
      e.printStackTrace();

      Assert.fail("Error initializing the session: " + e);
    }

    initSession();

    // System.out.println("out init");
    contact = new EPPContact(session);
  }

  /**
   * JUNIT {@code tearDown}, which currently does nothing.
   */
  @After
  public void tearDown() {
    endSession();
  }

  /**
   * JUNIT {@code suite} static method, which returns the tests associated with
   * {@code EPPEAITst}.
   */
  @BeforeClass
  public static void beforeClass() {
    String theConfigFileName = System.getProperty("EPP.ConfigFile");
    if (theConfigFileName != null)
      configFileName = theConfigFileName;

    try {
      app.initialize(configFileName);
    }
    catch (EPPCommandException e) {
      e.printStackTrace();

      Assert.fail("Error initializing the EPP Application: " + e);
    }
  }

  /**
   * Handle an {@code EPPCommandException}, which can be either a server
   * generated error or a general exception. If the exception was caused by a
   * server error, "Server Error :&lt;Response XML&gt;" will be specified. If
   * the exception was caused by a general algorithm error, "General Error
   * :&lt;Exception Description&gt;" will be specified.
   *
   * @param aException
   *           Exception thrown during test
   */
  public void handleException(Exception aException) {
    EPPResponse theResponse = null;
    if (aException instanceof EPPCommandException) {
      theResponse = ((EPPCommandException) aException).getResponse();
    }

    aException.printStackTrace();

    // Is a server specified error?
    if ((theResponse != null) && (!theResponse.isSuccess())) {
      Assert.fail("Server Error : " + theResponse);
    }

    else {
      Assert.fail("General Error : " + aException);
    }
  }

  /**
   * Makes a unique contact name using the current time.
   *
   * @return Unique contact name {@code String}
   */
  public String makeContactName() {
    long tm = System.currentTimeMillis();

    return new String("AE-" + String.valueOf(tm + rd.nextInt(5)).substring(7));
  }

  /**
   * Print the start of a test with the {@code Thread} name if the current
   * thread is a {@code TestThread}.
   *
   * @param aTest
   *           name for the test
   */
  private void printStart(String aTest) {
    if (Thread.currentThread() instanceof TestThread) {
      System.out.print(Thread.currentThread().getName() + ", iteration " + iteration + ": ");

      cat.info(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aTest + " Start");
    }

    System.out.println("Start of " + aTest);

    System.out.println("****************************************************************\n");
  }

  /**
   * Print the end of a test with the {@code Thread} name if the current thread
   * is a {@code TestThread}.
   *
   * @param aTest
   *           name for the test
   */
  private void printEnd(String aTest) {
    System.out.println("****************************************************************");

    if (Thread.currentThread() instanceof TestThread) {
      System.out.print(Thread.currentThread().getName() + ", iteration " + iteration + ": ");

      cat.info(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aTest + " End");
    }

    System.out.println("End of " + aTest);

    System.out.println("\n");
  }

  /**
   * Print message
   *
   * @param aMsg
   *           message to print
   */
  private void printMsg(String aMsg) {
    if (Thread.currentThread() instanceof TestThread) {
      System.out.println(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);

      cat.info(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);
    }

    else {
      System.out.println(aMsg);

      cat.info(aMsg);
    }
  }

  /**
   * Print error message
   *
   * @param aMsg
   *           errpr message to print
   */
  private void printError(String aMsg) {
    if (Thread.currentThread() instanceof TestThread) {
      System.err.println(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);

      cat.error(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);
    }

    else {
      System.err.println(aMsg);

      cat.error(aMsg);
    }
  }

}
