/***********************************************************
Copyright (C) 2024 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.addlemail;

import java.util.LinkedHashSet;
import java.util.Set;

import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPExtFactory;
import com.verisign.epp.codec.gen.EPPProtocolExtension;
import com.verisign.epp.codec.gen.EPPService;
import com.verisign.epp.codec.gen.EPPUtil;

/**
 * Factory for implementation of the Additional Email Extension. The Additional
 * Email Extension only supports command, response extensions with
 * {@code createExtension}, but currently does not support protocol extensions
 * with {@code createProtocolExtension}. Calling {@code createProtocolExtension}
 * will result in an exception.
 */
public class EPPAddlEmailExtFactory extends EPPExtFactory {

  /** XML Namespace */
  public static final String NS = "urn:ietf:params:xml:ns:epp:addlEmail-1.0";

  /** XML Namespace Prefix */
  public static final String NS_PREFIX = "addlEmail";

  /**
   * Service object associated with {@code EPPEAIExtFactory}. The service
   * object is used when creating the Greeting or the Login.
   */
  private EPPService service;

  /**
   * Default constructor for {@code EPPEAIExtFactory}.
   */
  public EPPAddlEmailExtFactory() {
    this.service = new EPPService(null, NS);
    this.service.setServiceType(EPPService.EXT_SERVICE);
  }

  /**
   * Creates a concrete {@code EPPCodecComponent} from an XML element that
   * represents an EPP extension.
   *
   * @param aExtensionElm
   *           Extension XML element.
   *
   * @return Concrete {@code EPPCodecComponent} associated with the extension
   *         XML element.
   *
   * @exception EPPCodecException
   *               Error creating the concrete {@code EPPCodecComponent}
   */
  @Override
  public EPPCodecComponent createExtension(Element aExtensionElm) throws EPPCodecException {
    String name = aExtensionElm.getLocalName();

    if (!aExtensionElm.getNamespaceURI().equals(NS)) {
      throw new EPPCodecException("Invalid extension type " + name);
    }

    if (name.equals(EPPUtil.getLocalName(EPPAddlEmail.ELM_NAME))) {
      return new EPPAddlEmail();
    }
    else {
      throw new EPPCodecException("Invalid extension element " + name);
    }
  }

  /**
   * Creates a concrete {@code EPPProtocolExtension} from an XML element that
   * represents an EPP protocol extension.
   *
   * @param aExtensionElm
   *           extension XML element.
   *
   * @return Concrete {@code EPPProtocolExtension} associated with the
   *         extension XML element.
   *
   * @exception EPPCodecException
   *               Error creating the concrete {@code EPPProtocolExtension}
   */
  @Override
  public EPPProtocolExtension createProtocolExtension(Element aExtensionElm) throws EPPCodecException {
    throw new EPPCodecException("EPPEAIExtFactory.createProtocolExtension: Protocol extensions not supported");
  }

  /**
   * Gets the service information associated with the concrete
   * {@code EPPExtFactory}. The service information is used by
   * {@code EPPFactory} for extracting the XML namespace associated with the
   * extension factory.
   *
   * @return service description associated with the concrete
   *         {@code EPPExtFactory}.
   */
  @Override
  public EPPService getService() {
    return this.service;
  }

  /**
   * Gets the list of XML schemas that need to be pre-loaded into the XML
   * Parser.
   *
   * @return Empty {@code Set} since factory is only used as a marker for the
   *         extension services.
   *
   *
   * @see com.verisign.epp.codec.gen.EPPMapFactory#getXmlSchemas()
   */
  @Override
  public Set getXmlSchemas() {
    Set theSchemas = new LinkedHashSet();
    theSchemas.add("addlemail-1.0.xsd");
    return theSchemas;
  }

}
