/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPDeleteCmd;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * Represents an EPP Registry &lt;delete&gt; command that allows a client to
 * delete a registry object. The EPP &lt;delete&gt; command provides a transform
 * operation that allows a client to delete a registry object. In addition to
 * the standard EPP command elements, the &lt;delete&gt; command MUST contain a
 * "registry:delete" element that identifies the registry namespace and the
 * location of the registry schema. <br>
 * <br>
 *
 * <ul>
 * <li>A &lt;registry:name&gt; element that contains the fully qualified
 * registry name of the object to be deleted. Use {@code getName} and
 * {@code setName} to get and set the element.</li>
 * </ul>
 *
 * <br>
 * {@code EPPReponse} is the response associated with
 * {@code EPPRegistryDeleteCmd}. <br>
 * <br>
 *
 * @see com.verisign.epp.codec.gen.EPPResponse
 */
public class EPPRegistryDeleteCmd extends EPPDeleteCmd {

	/**
	 * Logger
	 */
	private static Logger cat =  LoggerFactory.getLogger(EPPRegistryDeleteCmd.class);

	/**
	 * XML local name for {@code EPPRegistryDeleteCmd}.
	 */
	public static final String ELM_LOCALNAME = "delete";

	/**
	 * XML root tag for {@code EPPRegistryDeleteCmd}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/** 
	 * Registry Name of zone to delete. 
	 */
	private EPPRegistryZoneName name;

	/**
	 * {@code EPPRegistryDeleteCmd} default constructor. The name is initialized
	 * {@code name} to {@code null}. The {@code name} must be set before invoking
	 * {@code #encode(Document)}.
	 */
	public EPPRegistryDeleteCmd() {
		this.name = null;
	}

	/**
	 * {@code EPPRegistryDeleteCmd} constructor that takes the registry aLabel
	 * name as an argument.
	 *
	 * @param aTransId
	 *           transaction Id associated with command.
	 * @param aName
	 *           fully qualified aLabel name of zone object to delete.
	 */
	public EPPRegistryDeleteCmd(String aTransId, String aName) {
		super(aTransId);

		this.setName(aName);
	}

	/**
	 * {@code EPPRegistryDeleteCmd} constructor that takes the registry name as
	 * an argument.
	 *
	 * @param aTransId
	 *           transaction Id associated with command.
	 * @param aName
	 *           fully qualified name of zone object to delete.
	 */
	public EPPRegistryDeleteCmd(String aTransId, EPPRegistryZoneName aName) {
		super(aTransId);

		this.setName(aName);
	}

	/**
	 * Gets the EPP command Namespace associated with
	 * {@code EPPRegistryDeleteCmd}.
	 *
	 * @return {@code EPPRegistryMapFactory.NS}
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

	/**
	 * Gets the zone name.
	 *
	 * @return Zone name if set; {@code null} otherwise.
	 */
	public EPPRegistryZoneName getName() {
		return this.name;
	}

	/**
	 * Sets the zone name
	 *
	 * @param aName
	 *           Zone name
	 */
	public void setName(EPPRegistryZoneName aName) {
		this.name = aName;
	}

	/**
	 * Sets the aLabel zone name
	 *
	 * @param aName
	 *           aLabel zone name
	 */
	public void setName(String aName) {
		this.name = new EPPRegistryZoneName(aName, EPPRegistryZoneName.Form.aLabel);
	}

	/**
	 * Compare an instance of {@code EPPRegistryDeleteCmd} with this instance.
	 *
	 * @param aObject
	 *           Object to compare with
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryDeleteCmd)) {
			return false;
		}

		if (!super.equals(aObject)) {
			return false;
		}

		EPPRegistryDeleteCmd theComp = (EPPRegistryDeleteCmd) aObject;

		// Name
		if (!EqualityUtil.equals(this.name, theComp.name)) {
			cat.error("EPPRegistryDeleteCmd.equals(): name not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone {@code EPPRegistryDeleteCmd}.
	 *
	 * @return clone of {@code EPPRegistryDeleteCmd}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPRegistryDeleteCmd clone = (EPPRegistryDeleteCmd) super.clone();

		// Name
		if (this.name != null) {
			clone.name = (EPPRegistryZoneName) this.name.clone();
		}

		return clone;
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryDeleteCmd} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Root DOM Element representing the {@code EPPRegistryDeleteCmd}
	 *         instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPRegistryDeleteCmd} instance.
	 */
	@Override
	protected Element doEncode(Document aDocument) throws EPPEncodeException {
		if (this.name == null) {
			throw new EPPEncodeException("name required attribute is not set");
		}

		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

		// Name
		EPPUtil.encodeComp(aDocument, root, this.name);

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryDeleteCmd} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryDeleteCmd} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	protected void doDecode(Element aElement) throws EPPDecodeException {
		// Name
		this.name = (EPPRegistryZoneName) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS,
		      EPPRegistryZoneName.ELM_ZONE_NAME, EPPRegistryZoneName.class);
	}
}
