/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.org;

import java.util.Date;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPTransId;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * Represents an EPP Internationalized Domain Name (IDN) Table &lt;info&gt;
 * response, which support three different forms:<br>
 * <ul>
 * <li>Domain Info Form - Used to validate the domain name code points against
 * the IDN Tables and IDN Policies, and to return the matching IDN Table
 * meta-data.</li>
 * <li>Table Info Form - Used to retrieve information associated with an IDN
 * Table object.</li>
 * <li>List Info Form - Used to retrieve the list of IDN Tables supported by the
 * server.</li>
 * </ul>
 * 
 * <br>
 * <code>EPPOrgInfoCmd</code> is the concrete <code>EPPCommand</code> associated
 * with <code>EPPOrgInfoCmd</code>.
 * 
 * @see com.verisign.epp.codec.org.EPPOrgCreateResp
 */
public class EPPOrgCreateResp extends EPPResponse {

	/**
	 * Category for logging
	 */
	    private static Logger cat = LoggerFactory.getLogger(EPPOrgCreateResp.class);
	      

	/**
	 * XML local name for <code>EPPOrgCreateResp</code>.
	 */
	public static final String ELM_LOCALNAME = "creData";

	/**
	 * XML root tag for <code>EPPOrgCreateResp</code>.
	 */
	public static final String ELM_NAME = EPPOrgMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML local name for the org identifier element
	 */
	private static final String ELM_ORG_ID = "id";

	/**
	 * XML local name for the created date (<code>crDate</code>) element.
	 */
	private final static String ELM_CREATED_DATE = "crDate";

	/**
	 * The org identifier of the org of a sponsoring registrar.
	 */
	private String orgId;

	/**
	 * Date and time the org was created.
	 */
	private Date createdDate;

	/**
	 * <code>EPPOrgCreateResp</code> default constructor.
	 */
	public EPPOrgCreateResp() {
	}

	/**
	 * <code>EPPOrgCreateResp</code> constructor that only takes the transaction
	 * identifier.
	 * 
	 * @param aTransId
	 *           Transaction Id associated with response.
	 */
	public EPPOrgCreateResp(EPPTransId aTransId) {
		super(aTransId);
	}

	/**
	 * <code>EPPOrgCreateResp</code> constructor that all attributes as
	 * parameters.
	 * 
	 * @param aTransId
	 *           Transaction Id associated with command. Set to <code>null</code>
	 *           if a client transaction identifier is not desired.
	 * @param aOrgId
	 *           Org identifier
	 * @param aCreatedDate
	 *           Org created date
	 * 
	 */
	public EPPOrgCreateResp(EPPTransId aTransId, String aOrgId, Date aCreatedDate) {
		super(aTransId);
		this.orgId = aOrgId;
		this.createdDate = aCreatedDate;
	}

	/**
	 * Gets the the server-unique identifier of the organization object.
	 * 
	 * @return The the server-unique identifier of the organization object if
	 *         defined;<code>null</code> otherwise.
	 */
	public String getOrgId() {
		return this.orgId;
	}

	/**
	 * Sets the org identifier.
	 * 
	 * @param aOrgId
	 *           The org identifier
	 */
	public void setOrgId(String aOrgId) {
		this.orgId = aOrgId;
	}

	/**
	 * @return the createdDate
	 */
	public Date getCreatedDate() {
		return this.createdDate;
	}

	/**
	 * @param aCreatedDate
	 *           the createdDate to set
	 */
	public void setCreatedDate(Date aCreatedDate) {
		this.createdDate = aCreatedDate;
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * <code>EPPOrgCreateResp</code> instance.
	 * 
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 * 
	 * @return Element Root DOM Element representing the EPPOrgCreateResp
	 *         instance.
	 * 
	 * @exception EPPEncodeException
	 *               Unable to encode EPPOrgCreateResp instance.
	 */
	protected Element doEncode(Document aDocument) throws EPPEncodeException {

		Element theElm = null;
		Text theVal = null;

		// Check required attributes
		if (this.orgId == null) {
			throw new EPPEncodeException("Undefined orgId in EPPOrgCreateResp");
		}
		if (this.createdDate == null) {
			throw new EPPEncodeException("Undefined createdDate in EPPOrgCreateResp");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPOrgMapFactory.NS, ELM_NAME);

		// Org Identifier
		EPPUtil.encodeString(aDocument, root, this.orgId, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_ORG_ID);

		// Created Date
		EPPUtil.encodeTimeInstant(aDocument, root, this.createdDate, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_CREATED_DATE);

		return root;
	}

	/**
	 * Decode the <code>EPPOrgCreateResp</code> attributes from the aElement DOM
	 * Element tree.
	 * 
	 * @param aElement
	 *           Root DOM Element to decode <code>EPPOrgCreateResp</code> from.
	 * 
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	protected void doDecode(Element aElement) throws EPPDecodeException {

		Element theElm = null;

		// Org Identifier
		this.orgId = EPPUtil.decodeString(aElement, EPPOrgMapFactory.NS, ELM_ORG_ID);

		// Created Date
		this.createdDate = EPPUtil.decodeTimeInstant(aElement, EPPOrgMapFactory.NS, ELM_CREATED_DATE);
	}

	/**
	 * Clone <code>EPPOrgCreateResp</code>.
	 * 
	 * @return clone of <code>EPPOrgCreateResp</code>
	 * 
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPOrgCreateResp clone = (EPPOrgCreateResp) super.clone();
		return clone;
	}

	/**
	 * Gets the EPP response type associated with <code>EPPOrgCreateResp</code>.
	 * 
	 * @return <code>EPPOrgCreateResp.ELM_NAME</code>
	 */
	public String getType() {
		return ELM_NAME;
	}

	/**
	 * Gets the EPP command namespace associated with
	 * <code>EPPOrgCreateResp</code>.
	 * 
	 * @return <code>EPPOrgMapFactory.NS</code>
	 */
	public String getNamespace() {
		return EPPOrgMapFactory.NS;
	}

	/**
	 * Compare an instance of <code>EPPOrgCreateResp</code> with this instance.
	 * 
	 * @param aObject
	 *           Object to compare with.
	 * 
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPOrgCreateResp)) {
			return false;
		}

		EPPOrgCreateResp other = (EPPOrgCreateResp) aObject;

		// Org Identifier
		if (!EqualityUtil.equals(this.orgId, other.orgId)) {
			cat.error("EPPOrgCreateResp.equals(): orgId not equal");
			return false;
		}

		// Created Date
		if (!EqualityUtil.equals(this.createdDate, other.createdDate)) {
			cat.error("EPPOrgCreateResp.equals(): createdDate not equal");
			return false;
		}

		return true;
	}

	/**
	 * Implementation of <code>Object.toString</code>, which will result in an
	 * indented XML <code>String</code> representation of the concrete
	 * <code>EPPCodecComponent</code>.
	 * 
	 * @return Indented XML <code>String</code> if successful; <code>ERROR</code>
	 *         otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

}