/***********************************************************
Copyright (C) 2021 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.serverstub.v1_0;

import com.verisign.epp.codec.gen.EPPMsgQueue;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPResult;
import com.verisign.epp.codec.maintenance.v1_0.EPPMaintenanceInfoResp;
import com.verisign.epp.codec.maintenance.v1_0.EPPMaintenanceMapFactory;
import com.verisign.epp.framework.EPPPollDataRecord;
import com.verisign.epp.framework.EPPPollHandler;
import com.verisign.epp.framework.EPPPollQueueException;

/**
 * The {@code MaintenancePollHandler} implements {@code EPPPollHandler} for
 * Maintenance Name Mapping.
 */
public class MaintenancePollHandler implements EPPPollHandler {

	/**
	 * The kind used for the {@code MaintenancePollHandler} poll messages.
	 */
	public final static String KIND = EPPMaintenanceMapFactory.NS;

	/**
	 * Gets the kind of poll messages it handles.
	 *
	 * @return The domain name XML namespace represented by
	 *         {@code @link EPPMaintenanceMapFactory.NS}.
	 */
	@Override
	public String getKind() {
		return KIND;
	}

	/**
	 * Converts a poll queue {@code EPPPollDataRecord} to an {@code EPPResponse}.
	 *
	 * @param aRecord
	 *           Poll queue record to convert to an {@code EPPResponse}.
	 *
	 * @return An {@code EPPResponse} that represents the poll queue record.
	 *
	 * @throws EPPPollQueueException
	 *            Error converting the {@code EPPPollDataRecord} to an
	 *            {@code EPPResponse}.
	 */
	@Override
	public EPPResponse toResponse(EPPPollDataRecord aRecord) throws EPPPollQueueException {
		if (!aRecord.getKind().equals(getKind())) {
			throw new EPPPollQueueException("Handler for kind " + aRecord.getKind() + " does not match");
		}

		// Get the concrete response from the record
		EPPResponse theResponse = (EPPResponse) aRecord.getData();

		if (theResponse instanceof EPPMaintenanceInfoResp) {
			theResponse.setMsgQueue(new EPPMsgQueue(Long.valueOf(aRecord.getSize()), aRecord.getMsgId(),
			      aRecord.getQDate(), "Pending action completed successfully."));
		}
		else {
			throw new EPPPollQueueException("Unable to handle message class <" + theResponse.getClass().getName());
		}

		theResponse.setResult(EPPResult.SUCCESS_POLL_MSG);

		return theResponse;
	}
}
