/***********************************************************
Copyright (C) 2021 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.maintenance.v1_0;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPInfoCmd;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * Command used to used to retrieve registry maintenance information.
 *
 * @see com.verisign.epp.codec.maintenance.v1_0.EPPMaintenanceInfoResp
 */
public class EPPMaintenanceInfoCmd extends EPPInfoCmd {

	/**
	 * The type of the info command, which is for an individual maintenance or a
	 * list of maintenances.
	 */
	public enum InfoType {
		list, id;
	}

	/**
	 * Category for logging
	 */
	    private static Logger cat = LoggerFactory.getLogger(EPPMaintenanceInfoCmd.class);
	      

	/**
	 * XML local name for {@code EPPMaintenanceInfoCmd}.
	 */
	public static final String ELM_LOCALNAME = "info";

	/**
	 * XML root tag for {@code EPPMaintenanceInfoCmd}.
	 */
	public static final String ELM_NAME = EPPMaintenanceMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML local name for the maintenance notification identifier element
	 */
	private static final String ELM_MAINTENANCE_ID = "id";

	/**
	 * XML local name for the maintenance notification list element
	 */
	private static final String ELM_MAINTENANCE_LIST = "list";

	/**
	 * Type of the info command with the default set to {@code InfoType.list}.
	 */
	private InfoType infoType = InfoType.list;

	/**
	 * Maintenance notification identifier. The {@code maintenanceId} is used
	 * when the {@code infoType} is set to {@code InfoType.id}.
	 */
	private String maintenanceId;

	/**
	 * {@code EPPMaintenanceInfoCmd} default constructor.
	 */
	public EPPMaintenanceInfoCmd() {
	}

	/**
	 * {@code EPPMaintenanceInfoCmd} constructor that only takes the client
	 * transaction identifier. The {@code infoType} is set to
	 * {@code InfoType.list}.
	 *
	 * @param aTransId
	 *           Transaction Id associated with command.
	 */
	public EPPMaintenanceInfoCmd(String aTransId) {
		super(aTransId);
	}

	/**
	 * {@code EPPMaintenanceInfoCmd} constructor that takes the transaction
	 * identifier and the maintenance identifier, while will automatically set
	 * the {@code infoType} to {@code InfoType.id}.
	 *
	 * @param aTransId
	 *           Transaction Id associated with command. Set to {@code null} if a
	 *           client transaction identifier is not desired.
	 * @param aMaintenanceId
	 *           Maintenance notification identifier to retrieve
	 */
	public EPPMaintenanceInfoCmd(String aTransId, String aMaintenanceId) {
		super(aTransId);
		this.setMaintenanceId(aMaintenanceId);
	}

	/**
	 * Gets the i{@code infoType}for the info command.
	 *
	 * @return the {@code infoType} for the info command.
	 */
	public InfoType getInfoType() {
		return this.infoType;
	}

	/**
	 * Sets the {@code infoType} for the info command. The maintenance
	 * notification identifier is used only when the {@code infoType} is set to
	 * {@code InfoType.id}.
	 *
	 * @param aInfoType
	 *           The {@code infoType} for the info command.
	 */
	public void setInfoType(InfoType aInfoType) {
		this.infoType = aInfoType;
	}

	/**
	 * Gets the maintenance identifier.
	 *
	 * @return The maintenance identifier if defined; {@code null} otherwise.
	 */
	public String getMaintenanceId() {
		return this.maintenanceId;
	}

	/**
	 * Sets the maintenance identifier. When set to a non-{@code null} value, the
	 * {@code infoType} is set to {@code InfoType.id}; otherwise {@code
	 * infoType} is set to {@code InfoType.list}
	 *
	 * @param aMaintenanceId
	 *           The maintenance notification identifier.
	 */
	public void setMaintenanceId(String aMaintenanceId) {
		this.maintenanceId = aMaintenanceId;
		if (this.maintenanceId != null) {
			this.infoType = InfoType.id;
		}
		else {
			this.infoType = InfoType.list;
		}
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPMaintenanceInfoCmd} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPMaintenanceInfoCmd} instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPMaintenanceInfoCmd} instance.
	 */
	@Override
	protected Element doEncode(Document aDocument) throws EPPEncodeException {

		// Check required attributes
		if (this.infoType == InfoType.id && this.maintenanceId == null) {
			throw new EPPEncodeException("Undefined maintenanceId in EPPMaintenanceInfoCmd");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPMaintenanceMapFactory.NS, ELM_NAME);

		switch (this.infoType) {
			// Maintenance Notification List
			case list:
				root.appendChild(aDocument.createElementNS(EPPMaintenanceMapFactory.NS,
				      EPPMaintenanceMapFactory.NS_PREFIX + ":" + ELM_MAINTENANCE_LIST));
				break;
			// Maintenance Notification Identifier
			case id:
				// Maintenance Notification Identifier
				EPPUtil.encodeString(aDocument, root, this.maintenanceId, EPPMaintenanceMapFactory.NS,
				      EPPMaintenanceMapFactory.NS_PREFIX + ":" + ELM_MAINTENANCE_ID);
				break;
		}

		return root;
	}

	/**
	 * Decode the {@code EPPMaintenanceInfoCmd} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPMaintenanceInfoCmd} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	protected void doDecode(Element aElement) throws EPPDecodeException {

		// Set the maintenance notification identifier and {@code infoType} to
		// {@link InfoType.id} if exists; otherwise set {@code infoType} to {@code
		// InfoType.list}.
		this.setMaintenanceId(EPPUtil.decodeString(aElement, EPPMaintenanceMapFactory.NS, ELM_MAINTENANCE_ID));
	}

	/**
	 * Compare an instance of {@code EPPMaintenanceInfoCmd} with this instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise.
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPMaintenanceInfoCmd)) {
			return false;
		}

		EPPMaintenanceInfoCmd other = (EPPMaintenanceInfoCmd) aObject;

		if (!EqualityUtil.equals(this.infoType, other.infoType)) {
			cat.error("EPPMaintenanceInfoCmd.equals(): infoType not equal");
			return false;
		}

		// Maintenance Identifier
		if (!EqualityUtil.equals(this.maintenanceId, other.maintenanceId)) {
			cat.error("EPPMaintenanceInfoCmd.equals(): maintenanceId not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone {@code EPPMaintenanceInfoCmd}.
	 *
	 * @return Deep copy clone of {@code EPPMaintenanceInfoCmd}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPMaintenanceInfoCmd clone = (EPPMaintenanceInfoCmd) super.clone();

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Gets the EPP command Namespace associated with
	 * {@code EPPMaintenanceInfoCmd}.
	 *
	 * @return {@code EPPMaintenanceMapFactory.NS}
	 */
	@Override
	public String getNamespace() {
		return EPPMaintenanceMapFactory.NS;
	}

}
