/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.launchpolicy.v01;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * Abstract base class for the command and response extensions that contains a
 * zone.
 *
 * @see EPPLaunchPolicyZone
 */
public abstract class EPPLaunchPolicyZoneContainer implements EPPCodecComponent {

	/**
	 * XML root tag for {@code EPPLaunchPolicyZoneContainer}.
	 */
	private String ELM_NAME = EPPLaunchPolicyExtFactory.NS_PREFIX + ":" + getLocalName();

	/**
	 * Instance of {@link EPPLaunchPolicyZone} that is contained by the
	 * extension.
	 */
	private EPPLaunchPolicyZone zone = null;

	/**
	 * Abstract method that the sub-class must define to return the local name
	 * for the root element.
	 *
	 * @return Local name of the root element of the extension.
	 */
	protected abstract String getLocalName();

	/**
	 * Gets the {@link Logger} to use for logging.
	 *
	 * @return {@link Logger} instance determined by the derived class.
	 */
	protected abstract Logger getLogger();

	/**
	 * Default constructor for {@code EPPLaunchPolicyZoneContainer}.
	 */
	public EPPLaunchPolicyZoneContainer() {
	}

	/**
	 * Constructor for {@code EPPLaunchPolicyZoneContainer} that takes the
	 * contained zone.
	 *
	 * @param aZone
	 *           The zone to contain
	 */
	public EPPLaunchPolicyZoneContainer(EPPLaunchPolicyZone aZone) {
		this.zone = aZone;
	}

	/**
	 * Gets the contained zone of the extension.
	 *
	 * @return Instance of contained {@link EPPLaunchPolicyZone}
	 */
	public EPPLaunchPolicyZone getZone() {
		return this.zone;
	}

	/**
	 * Sets the contained zone of the extension.
	 *
	 * @param aZone
	 *           Instance of {@link EPPLaunchPolicyZone}
	 */
	public void setZone(EPPLaunchPolicyZone aZone) {
		this.zone = aZone;
	}

	/**
	 * Encode instance into a DOM element tree. A DOM Document is passed as an
	 * argument and functions as a factory for DOM objects. The root element
	 * associated with the instance is created and each instance attribute is
	 * appended as a child node.
	 *
	 * @param aDocument
	 *           DOM Document, which acts is an Element factory
	 *
	 * @return Element Root element associated with the object
	 *
	 * @exception EPPEncodeException
	 *               Error encoding {@code EPPLaunchPolicyZoneContainer}
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {

		if (aDocument == null) {
			throw new EPPEncodeException("aDocument is null" + " on in EPPLaunchPolicyZoneContainer.encode(Document)");
		}

		// Check for required attributes
		if (this.zone == null) {
			throw new EPPEncodeException("zone is null in EPPLaunchPolicyZoneContainer.encode(Document).");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPLaunchPolicyExtFactory.NS, this.ELM_NAME);

		// Zone
		EPPUtil.encodeComp(aDocument, root, this.zone);

		return root;
	}

	/**
	 * Decode a DOM element tree to initialize the instance attributes. The
	 * {@code aElement} argument represents the root DOM element and is used to
	 * traverse the DOM nodes for instance attribute values.
	 *
	 * @param aElement
	 *           {@code Element} to decode
	 *
	 * @exception EPPDecodeException
	 *               Error decoding {@code Element}
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {

		// Zone
		this.zone = (EPPLaunchPolicyZone) EPPUtil.decodeComp(aElement, EPPLaunchPolicyExtFactory.NS,
		      EPPLaunchPolicyZone.ELM_NAME, EPPLaunchPolicyZone.class);
	}

	/**
	 * clone an {@code EPPCodecComponent}.
	 *
	 * @return clone of concrete {@code EPPLaunchPolicyZoneContainer}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPLaunchPolicyZoneContainer clone = (EPPLaunchPolicyZoneContainer) super.clone();

		if (this.zone != null) {
			clone.zone = (EPPLaunchPolicyZone) this.zone.clone();
		}

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Compare an instance of {@code EPPLaunchPolicyZoneContainer} with this
	 * instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return {@code true} if equal; {@code false} otherwise.
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPLaunchPolicyZoneContainer)) {
			getLogger().error("EPPLaunchPolicyZoneContainer.equals(): " + aObject.getClass().getName()
			      + " not EPPLaunchPolicyZoneContainer instance");

			return false;
		}

		EPPLaunchPolicyZoneContainer theComp = (EPPLaunchPolicyZoneContainer) aObject;

		// Zone
		if (!EqualityUtil.equals(this.zone, theComp.zone)) {
			getLogger().error("EPPLaunchPolicyZoneContainer.equals(): zone not equal");
			return false;
		}

		return true;
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 * 
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPLaunchPolicyExtFactory.NS;
	}

}
