/***********************************************************
Copyright (C) 2022 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.eai.v1_0;

import java.util.Vector;

import com.verisign.epp.codec.contact.EPPContactAddChange;
import com.verisign.epp.codec.contact.EPPContactAddress;
import com.verisign.epp.codec.contact.EPPContactCreateCmd;
import com.verisign.epp.codec.contact.EPPContactPostalDefinition;
import com.verisign.epp.codec.contact.EPPContactUpdateCmd;
import com.verisign.epp.codec.gen.EPPAuthInfo;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPCodecTst;
import com.verisign.epp.codec.gen.EPPEncodeDecodeStats;
import com.verisign.epp.codec.gen.EPPFactory;
import com.verisign.epp.util.TestThread;

import junit.framework.Assert;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

/**
 * Is a unit test of supporting the EAI addresses based on the functional
 * extension defined in the draft.
 */
public class EPPEAITst extends TestCase {
	/**
	 * Number of unit test iterations to run. This is set in
	 * {@code EPPCodecTst.main}
	 */
	static private long numIterations = 1;

	/**
	 * Creates a new EPPEAITst object.
	 *
	 * @param name
	 *           Name of test
	 */
	public EPPEAITst(String name) {
		super(name);
	}

	/**
	 * Unit test of passing an EAI address in a [RFC5733] contact create command.
	 */
	public void testContactCreate() {
		EPPContactCreateCmd theCommand;
		EPPEncodeDecodeStats commandStats;

		EPPCodecTst.printStart("testContactCreate");

		// Streets
		Vector streets = new Vector();
		streets.addElement("123 Example Dr.");
		streets.addElement("Suite 100");

		// Address
		EPPContactAddress address = new EPPContactAddress(streets, "Dulles", "VA", "20166-6503", "US");

		// Postal Definition
		EPPContactPostalDefinition name = new EPPContactPostalDefinition("Dörte Sörense", "Example Inc.",
		      EPPContactPostalDefinition.ATTR_TYPE_LOC, address);

		// Contact Command
		theCommand = new EPPContactCreateCmd("ABC-123451", "eai-contact", name, "DörteSörensen@example.com", new EPPAuthInfo(""));

		commandStats = EPPCodecTst.testEncodeDecode(theCommand);
		System.out.println(commandStats);

		EPPCodecTst.printEnd("testContactCreate");
	}

	/**
	 * Unit test of passing an EAI address in an [RFC5733] contact update
	 * command.
	 */
	public void testContactUpdate() {
		EPPContactUpdateCmd theCommand = null;
		EPPEncodeDecodeStats commandStats = null;

		EPPCodecTst.printStart("testContactUpdate");

		EPPContactAddChange changeItems = new EPPContactAddChange();
		changeItems.setEmail("DörteSörensen2@example.com");

		// Update Command
		theCommand = new EPPContactUpdateCmd("ABC-123452", "eai-contact", null, null, changeItems);

		commandStats = EPPCodecTst.testEncodeDecode(theCommand);
		System.out.println(commandStats);

		EPPCodecTst.printEnd("testContactUpdate");
	}

	/**
	 * JUNIT {@code setUp} method, which currently does nothing.
	 */
	@Override
	protected void setUp() {
	}

	/**
	 * JUNIT {@code tearDown}, which currently does nothing.
	 */
	@Override
	protected void tearDown() {
	}

	/**
	 * JUNIT {@code suite} static method, which returns the tests associated with
	 * {@code EPPEAITst}.
	 *
	 * @return Tests to execute
	 */
	public static Test suite() {
		EPPCodecTst.initEnvironment();

		TestSuite suite = new TestSuite(EPPEAITst.class);

		// iterations Property
		String numIterProp = System.getProperty("iterations");

		if (numIterProp != null) {
			numIterations = Integer.parseInt(numIterProp);
		}

		// Add map factories the EPPCodec.
		try {
			EPPFactory.getInstance().addMapFactory("com.verisign.epp.codec.contact.EPPContactMapFactory");
		}
		catch (EPPCodecException e) {
			Assert.fail("EPPCodecException adding EPPContactMapFactory to EPPCodec: " + e);
		}

		return suite;
	}

	/**
	 * Unit test main, which accepts the following system property options: <br>
	 *
	 * <ul>
	 * <li>iterations Number of unit test iterations to run</li>
	 * <li>validate Turn XML validation on ({@code true}) or off (
	 * {@code false}). If validate is not specified, validation will be off.</li>
	 * </ul>
	 *
	 *
	 * @param args
	 *           Command line arguments
	 */
	public static void main(String[] args) {
		// Number of Threads
		int numThreads = 1;
		String threadsStr = System.getProperty("threads");

		if (threadsStr != null) {
			numThreads = Integer.parseInt(threadsStr);
		}

		// Run test suite in multiple threads?
		if (numThreads > 1) {
			// Spawn each thread passing in the Test Suite
			for (int i = 0; i < numThreads; i++) {
				TestThread thread = new TestThread("EPPEAITst Thread " + i, EPPEAITst.suite());
				thread.start();
			}
		}
		else { // Single threaded mode.
			junit.textui.TestRunner.run(EPPEAITst.suite());
		}
	}

	/**
	 * Sets the number of iterations to run per test.
	 *
	 * @param aNumIterations
	 *           number of iterations to run per test
	 */
	public static void setNumIterations(long aNumIterations) {
		numIterations = aNumIterations;
	}

}
