/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.domain;

import java.util.Date;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPRenewCmd;
import com.verisign.epp.codec.gen.EPPUtil;

/**
 * Represents an EPP Domain &lt;renew&gt; command, which provides a transform
 * operation that allows a client to extend the validity period of a domain
 * object. The EPP &lt;renew&gt; command provides a transform operation that
 * allows a client to extend the validity period of a domain object. In addition
 * to the standard EPP command elements, the &lt;renew&gt; command MUST contain
 * a &lt;domain:renew&gt; element that identifies the domain namespace and the
 * location of the domain schema. The &lt;domain:renew&gt; element SHALL contain
 * the following child elements: <br>
 * <br>
 *
 * <ul>
 * <li>A &lt;domain:name&gt; element that contains the fully qualified domain
 * name of the object whose validity period is to be extended. Use
 * {@code getName} and {@code setName} to get and set the element.</li>
 * <li>A &lt;domain:curExpDate&gt; element that contains the date on which the
 * current validity period ends. This value ensures that repeated &lt;renew&gt;
 * commands do not result in multiple unanticipated successful renewals. Use
 * {@code getCurExpDate} and {@code setCurExpDate} to get and set the
 * element.</li>
 * <li>An OPTIONAL &lt;domain:period&gt; element that contains the initial
 * registration period of the domain object. Use {@code getPeriod} and
 * {@code setPeriod} to get and set the element. If return {@code null}, period
 * has not been specified yet.</li>
 * </ul>
 *
 * @see com.verisign.epp.codec.domain.EPPDomainRenewResp
 */
public class EPPDomainRenewCmd extends EPPRenewCmd {
	/**
	 * XML local name for {@code EPPDomainRenewCmd}.
	 */
	public static final String ELM_LOCALNAME = "renew";

	/**
	 * XML Element Name of {@code EPPDomainRenewCmd} root element.
	 */
	public static final String ELM_NAME = EPPDomainMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML Element Name for the {@code name} attribute.
	 */
	private final static String ELM_DOMAIN_NAME = "name";

	/**
	 * XML Element Name for the {@code currentExpirationYear} attribute.
	 */
	private final static String ELM_CURRENT_EXPIRATION_DATE = "curExpDate";

	/**
	 * Domain Name of domain to create.
	 */
	private String name = null;

	/**
	 * Registration Period
	 */
	private EPPDomainPeriod period = null;

	/**
	 * Current Expiration Date.
	 */
	private java.util.Date curExpDate = null;

	/**
	 * Allocates a new {@code EPPDomainRenewCmd} with default attribute values.
	 * the defaults include the following: <br>
	 * <br>
	 *
	 * <ul>
	 * <li>name is set to {@code null}</li>
	 * <li>period is set to {@code 1 year}</li>
	 * <li>current expiration date to {@code null}</li>
	 * </ul>
	 *
	 * <br>
	 * The name and current expiration year must be set before invoking
	 * {@code encode}.
	 */
	public EPPDomainRenewCmd() {
		this.name = null;
		this.period = new EPPDomainPeriod(1);
		this.curExpDate = null;
	}

	/**
	 * {@code EPPDomainRenewCmd} constructor that takes the domain name and the
	 * current expiration year as arguments. The period will default to 1 year.
	 *
	 * @param aTransId
	 *           Transaction Id associated with command.
	 * @param aName
	 *           Domain name to renew.
	 * @param aCurExpDate
	 *           The current expiration date of the domain
	 */
	public EPPDomainRenewCmd(String aTransId, String aName, Date aCurExpDate) {
		super(aTransId);

		this.name = aName;
		this.curExpDate = aCurExpDate;

		// default to 1 year
		this.period = new EPPDomainPeriod(1);
	}

	/**
	 * {@code EPPDomainRenewCmd} constructor that takes all of the attributes of
	 * the renew command as arguments.
	 *
	 * @param aTransId
	 *           Transaction Id associated with command.
	 * @param aName
	 *           Domain name to renew.
	 * @param aCurExpDate
	 *           The current expiration date of the domain.
	 * @param aPeriod
	 *           Registration period in years.
	 */
	public EPPDomainRenewCmd(String aTransId, String aName, Date aCurExpDate, EPPDomainPeriod aPeriod) {
		super(aTransId);

		this.name = aName;
		this.curExpDate = aCurExpDate;
		this.period = aPeriod;
	}

	/**
	 * Gets the EPP command Namespace associated with {@code EPPDomainRenewCmd}.
	 *
	 * @return {@code EPPDomainMapFactory.NS}
	 */
	@Override
	public String getNamespace() {
		return EPPDomainMapFactory.NS;
	}

	/**
	 * Gets the domain name to renew.
	 *
	 * @return Domain Name if defined; {@code null} otherwise.
	 */
	public String getName() {
		return this.name;
	}

	/**
	 * Sets the domain name to renew.
	 *
	 * @param aName
	 *           Domain Name
	 */
	public void setName(String aName) {
		this.name = aName;
	}

	/**
	 * Compare an instance of {@code EPPDomainRenewCmd} with this instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return DOCUMENT ME!
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPDomainRenewCmd)) {
			return false;
		}

		if (!super.equals(aObject)) {
			return false;
		}

		EPPDomainRenewCmd theMap = (EPPDomainRenewCmd) aObject;

		// Name
		if (!((this.name == null) ? (theMap.name == null) : this.name.equals(theMap.name))) {
			return false;
		}

		// Current Expiration Date
		if (!((this.curExpDate == null) ? (theMap.curExpDate == null) : this.curExpDate.equals(theMap.curExpDate))) {
			return false;
		}

		// Period
		if (!((this.period == null) ? (theMap.period == null) : this.period.equals(theMap.period))) {
			return false;
		}

		return true;
	}

	/**
	 * Clone {@code EPPDomainRenewCmd}.
	 *
	 * @return clone of {@code EPPDomainRenewCmd}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPDomainRenewCmd clone = (EPPDomainRenewCmd) super.clone();

		if (this.period != null) {
			clone.period = (EPPDomainPeriod) this.period.clone();
		}

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Get current expiration date.
	 *
	 * @return java.util.Date
	 */
	public Date getCurExpDate() {
		return this.curExpDate;
	}

	/**
	 * Gets the registration period of the renew command in years.
	 *
	 * @return Registration Period in years.
	 */
	public EPPDomainPeriod getPeriod() {
		return this.period;
	}

	/**
	 * Set current expiration date.
	 *
	 * @param newCurExpDate
	 *           java.util.Date
	 */
	public void setCurExpDate(Date newCurExpDate) {
		this.curExpDate = newCurExpDate;
	}

	/**
	 * Sets the registration period of the renew command in years.
	 *
	 * @param aPeriod
	 *           Registration Period in years.
	 */
	public void setPeriod(EPPDomainPeriod aPeriod) {
		this.period = aPeriod;
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPDomainRenewCmd} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Root DOM Element representing the {@code EPPDomainRenewCmd}
	 *         instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPDomainRenewCmd} instance.
	 */
	@Override
	protected Element doEncode(Document aDocument) throws EPPEncodeException {
		try {
			validateState();
		}
		catch (EPPCodecException e) {
			throw new EPPEncodeException("EPPDomainRenewCmd invalid state: " + e);
		}

		Element root = aDocument.createElementNS(EPPDomainMapFactory.NS, ELM_NAME);

		// Name
		EPPUtil.encodeString(aDocument, root, this.name, EPPDomainMapFactory.NS,
		      EPPDomainMapFactory.NS_PREFIX + ":" + ELM_DOMAIN_NAME);

		// Current Expiration Date
		EPPUtil.encodeDate(aDocument, root, this.curExpDate, EPPDomainMapFactory.NS,
		      EPPDomainMapFactory.NS_PREFIX + ":" + ELM_CURRENT_EXPIRATION_DATE);

		// Period with Attribute of Unit
		if (!this.period.isPeriodUnspec()) {
			EPPUtil.encodeComp(aDocument, root, this.period);
		}

		return root;
	}

	/**
	 * Decode the {@code EPPDomainRenewCmd} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPDomainRenewCmd} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	protected void doDecode(Element aElement) throws EPPDecodeException {
		// Domain Name
		this.name = EPPUtil.decodeString(aElement, EPPDomainMapFactory.NS, ELM_DOMAIN_NAME);

		// Current Expiration Year
		this.curExpDate = EPPUtil.decodeDate(aElement, EPPDomainMapFactory.NS, ELM_CURRENT_EXPIRATION_DATE);

		// Period
		this.period = (EPPDomainPeriod) EPPUtil.decodeComp(aElement, EPPDomainMapFactory.NS, EPPDomainPeriod.ELM_NAME,
		      EPPDomainPeriod.class);
	}

	/**
	 * Validates the state of the {@code EPPDomainRenewCmd} instance. A valid
	 * state means that all of the required attributes have been set. If
	 * validateState returns without an exception, the state is valid. If the
	 * state is not valid, the EPPCodecException will contain a description of
	 * the error. throws EPPCodecException State error. This will contain the
	 * name of the attribute that is not valid.
	 *
	 * @throws EPPCodecException
	 *            When not valid
	 */
	void validateState() throws EPPCodecException {
		if (this.name == null) {
			throw new EPPCodecException("name required attribute is not set");
		}

		if (this.curExpDate == null) {
			throw new EPPCodecException("currentExpirationYear required attribute is not set");
		}
	}

}
