/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.domain;

import java.util.Vector;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPAuthInfo;
import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPFactory;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * Represents attributes to add, remove or change with a
 * {@code EPPDomainUpdateCmd}. In {@code EPPDomainUpdateCmd}, an instance of
 * {@code EPPDomainAddRemove} is used to specify the attributes to add; an
 * instance of {@code EPPDomainAddRemove} is used to specify the attributes to
 * remove, and an instance of {@code EPPDomainAddRemove} is used to specify the
 * attributes to change<br>
 * <br>
 * The Domain Mapping Specification describes the following attributes:<br>
 *
 * <ul>
 * <li>Zero or more &lt;domain:ns&gt; elements that contain the fully qualified
 * host name of a known host object. Use {@code getServers} and
 * {@code setServers} to get and set the element.</li>
 * <li>Zero or more &lt;domain:contact&gt; elements that contain the registrant,
 * administrative, technical, and billing contact identifiers to be associated
 * with the domain. Use {@code getContacts} and {@code setContacts} to get and
 * set the element. This attribute will only be allowed if the Contact Mapping
 * is supported.</li>
 * <li>One or two &lt;domain:status&gt; elements that contain status values to
 * be applied to or removed from the domain object. Use {@code getStatuses} and
 * {@code setStatuses} to get and set the element.</li>
 * <li>For {@code change} only, A &lt;domain:registrant&gt; element that
 * contains the identifier for the human or organizational social information
 * (contact) object to be associated with the domain object as the object
 * registrant. This object identifier MUST be known to the server before the
 * contact object can be associated with the domain object. Use
 * {@code getRegistrant} and {@code setRegistrant} to get and set the
 * element.</li>
 * </ul>
 *
 * <br>
 * It is important to note that the maximum number of domain attribute elements
 * is subject to the number of values currently associated with the domain
 * object. {@code EPPDomainAddRemove} will delegate the validation of the
 * cardinality of the domain attributes elements to the EPP Server.
 *
 * @see com.verisign.epp.codec.domain.EPPDomainUpdateCmd
 */
public class EPPDomainAddRemove implements EPPCodecComponent {

	/**
	 * mode of {@code EPPDomainAddRemove} is not specified.
	 */
	private final static short MODE_NONE = 0;

	/**
	 * mode of {@code EPPDomainAddRemove} is to add attributes.
	 */
	public final static short MODE_ADD = 1;

	/**
	 * mode of {@code EPPDomainAddRemove} is to remove attributes.
	 */
	public final static short MODE_REMOVE = 2;

	/**
	 * XML tag name when the {@code mode} attribute is
	 * {@code EPPDomainAddRemove.MODE_ADD}. This is a package constant, so the
	 * container {@code EPPCodecComponent} can use it on a decode operation.
	 */
	final static String ELM_ADD = "add";

	/**
	 * XML tag name when the {@code mode} attribute is
	 * {@code EPPDomainAddRemove.MODE_REMOVE}. This is a package constant, so the
	 * container {@code EPPCodecComponent} can use it on a decode operation.
	 */
	final static String ELM_REMOVE = "rem";

	/**
	 * XML tag name when the {@code mode} attribute is
	 * {@code EPPDomainAddRemove.MODE_CHANGE}. This is a package constant, so the
	 * container {@code EPPCodecComponent} can use it on a decode operation.
	 */
	final static String ELM_CHANGE = "chg";

	/** mode of {@code EPPDomainAddRemove} is to change attributes. */
	final static short MODE_CHANGE = 3;

	/** XML tag name for the {@code servers} attribute. */
	private final static String ELM_SERVER = "ns";

	/** XML tag name for host object reference */
	private final static String ELM_HOST_OBJ = "hostObj";

	/** XML tag name for host attribute */
	private final static String ELM_HOST_ATTR = EPPHostAttr.ELM_NAME;

	/** XML tag name for the {@code contacts} attribute. */
	private final static String ELM_CONTACT = "contact";

	/** XML tag name for the {@code statuses} attribute. */
	private final static String ELM_STATUS = "status";

	/** XML tag name for the {@code servers} attribute. */
	private final static String ELM_REGISTRANT = "registrant";

	/** Category for logging */
	private static Logger cat = LoggerFactory.getLogger(EPPDomainAddRemove.class);

	/**
	 * Mode of EPPDomainAddRemove. Must be {@code MODE_ADD} or
	 * {@code MODE_REMOVE} to be valid. This attribute will be set by the parent
	 * container {@code EPPCodecComponent}. For example,
	 * {@code EPPDomainUpdateCmd} will set the mode for its
	 * {@code EPPDomainAddRemove} instances.
	 */
	private short mode = MODE_NONE;

	/**
	 * Name Servers to add or remove that contains either {@code String} or
	 * {@link EPPHostAttr} objects.
	 */
	private Vector<?> servers = null;

	/** Host Attribute Name Servers to add or remove. */
	private Vector<EPPHostAttr> attrServers = null;

	/** Contacts to add or remove. */
	private Vector<EPPDomainContact> contacts = null;

	/** Status to add or remove. */
	private Vector<EPPDomainStatus> statuses = null;

	/** authorization information to change. */
	private EPPAuthInfo authInfo = null;

	/** registrant to change. */
	private String registrant = null;

	/**
	 * Default constructor for {@code EPPDomainAddRemove}. All of the attribute
	 * default to {@code null} to indicate no modification.
	 */
	public EPPDomainAddRemove() {
		this.servers = null;
		this.servers = null;
		this.contacts = null;
		this.statuses = null;
		this.registrant = null;
		this.authInfo = null;
	}

	/**
	 * Constructor for {@code EPPDomainAddRemove} that includes the attributes as
	 * arguments.
	 *
	 * @param aServers
	 *           Vector of Name Server that are either name server {@code String}
	 *           or {@link EPPHostAttr} objects. Is {@code null} or empty for no
	 *           modifications.
	 * @param aContacts
	 *           Vector of {@code EPPDomainContact} instances. Is {@code null} or
	 *           empty for no modifications. If the Contact Mapping is not
	 *           supported, this value should be {@code null}.
	 * @param aStatuses
	 *           Vector of {@link EPPDomainStatus} instances. Is {@code null} or
	 *           empty for no modifications.
	 */
	public EPPDomainAddRemove(Vector<?> aServers, Vector<EPPDomainContact> aContacts,
	      Vector<EPPDomainStatus> aStatuses) {
		this.servers = aServers;
		this.contacts = aContacts;
		this.statuses = aStatuses;
	}

	/**
	 * Constructor for {@code EPPDomainAddRemove} that includes the attributes as
	 * arguments.
	 *
	 * @param aRegistrant
	 *           {@code String} registrant for the change mode
	 * @param aAuthInfo
	 *           {@code EPPAuthInfo} authorization information for the change
	 *           mode
	 */
	public EPPDomainAddRemove(String aRegistrant, EPPAuthInfo aAuthInfo) {
		this.registrant = aRegistrant;
		this.setAuthInfo(aAuthInfo);
	}

	/**
	 * Are name servers set?
	 *
	 * @return {@code true} if name servers are set; {@code false} otherwise.
	 */
	public boolean hasServers() {
		if (this.servers != null) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Add a name server to the list of name servers.
	 * 
	 * @param aServer
	 *           Name server to add
	 */
	public void addServer(String aServer) {
		if (this.servers == null) {
			this.servers = new Vector<String>();
		}

		((Vector<String>) this.servers).add(aServer);
	}

	/**
	 * Add a name server host attribute to the list of name servers.
	 * 
	 * @param aServer
	 *           Name server to add
	 */
	public void addServer(EPPHostAttr aServer) {
		if (this.servers == null) {
			this.servers = new Vector<EPPHostAttr>();
		}

		((Vector<EPPHostAttr>) this.servers).add(aServer);
	}

	/**
	 * Gets the name servers. The name servers can either be {@code String}
	 * instances containing the fully qualified name of a known name server host
	 * object, or {@link EPPHostAttr} instances containing the fully qualified
	 * name of a host and optionally the host IP addresses.
	 *
	 * @return {@code Vector} of name server {@code String} instances for host
	 *         object references or {@link EPPHostAttr} instances for host
	 *         attribute values if exists; {@code null} otherwise.
	 */
	public Vector<?> getServers() {
		return this.servers;
	}

	/**
	 * Sets the name servers. The name servers can either be {@code String}
	 * instances containing the fully qualified name of a known name server host
	 * object, or {@code EPPHostAttr} instances containing the fully qualified
	 * name of a host and optionally the host IP addresses.
	 *
	 * @param aServers
	 *           {@code Vector} of name server {@code String} instances for host
	 *           object references or {@code EPPHostAttr} instances for host
	 *           attribute values.
	 */
	public void setServers(Vector<?> aServers) {
		this.servers = aServers;
	}

	/**
	 * Are contacts set?
	 *
	 * @return {@code true} if contacts are set; {@code false} otherwise.
	 */
	public boolean hasContacts() {
		if (this.contacts != null) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Add a contact to the list of contacts.
	 * 
	 * @param aContact
	 *           Contact to add
	 */
	public void addContact(EPPDomainContact aContact) {
		if (this.contacts == null) {
			this.contacts = new Vector<EPPDomainContact>();
		}

		this.contacts.add(aContact);
	}

	/**
	 * Gets the contacts to add or remove.
	 *
	 * @return Vector of {@code EPPDomainContact} instances if set; {@code null}
	 *         otherwise.
	 */
	public Vector<EPPDomainContact> getContacts() {
		return this.contacts;
	}

	/**
	 * Sets the contacts to add or remove.
	 *
	 * @param aContacts
	 *           The contacts to add or remove.
	 */
	public void setContacts(Vector<EPPDomainContact> aContacts) {
		this.contacts = aContacts;
	}

	/**
	 * Are statuses set?
	 *
	 * @return {@code true} if statuses are set; {@code false} otherwise.
	 */
	public boolean hasStatuses() {
		if (this.statuses != null) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Add a status to the list of statuses.
	 * 
	 * @param aStatus
	 *           Status to add
	 */
	public void addStatus(EPPDomainStatus aStatus) {
		if (this.statuses == null) {
			this.statuses = new Vector<EPPDomainStatus>();
		}

		this.statuses.add(aStatus);
	}

	/**
	 * Gets the statuses to add or remove.
	 *
	 * @return Vector of status {@link EPPDomainStatus} instances if set;
	 *         {@code null} otherwise.
	 */
	public Vector<EPPDomainStatus> getStatuses() {
		return this.statuses;
	}

	/**
	 * Sets the statuses to add or remove.
	 *
	 * @param aStatuses
	 *           Vector of status {@link EPPDomainStatus} instances.
	 */
	public void setStatuses(Vector<EPPDomainStatus> aStatuses) {
		this.statuses = aStatuses;
	}

	/**
	 * Return if Domain Contacts is supported.
	 *
	 * @return {@code true} if contacts are supported; {@code false} otherwise.
	 */
	public boolean contactsSupported() {
		return EPPFactory.getInstance().hasService(EPPDomainMapFactory.NS_CONTACT);
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPDomainAddRemove} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Root DOM Element representing the {@code EPPDomainAddRemove}
	 *         instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPDomainAddRemove} instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		Element root;

		// Change mode
		if (this.mode == MODE_CHANGE) {
			root = aDocument.createElementNS(EPPDomainMapFactory.NS, EPPDomainMapFactory.NS_PREFIX + ":" + ELM_CHANGE);

			if (this.hasRegistrant()) {
				EPPUtil.encodeString(aDocument, root, this.registrant, EPPDomainMapFactory.NS,
				      EPPDomainMapFactory.NS_PREFIX + ":" + ELM_REGISTRANT);
			}

			if (this.hasAuthInfo()) {
				EPPUtil.encodeComp(aDocument, root, this.authInfo);
			}

			return root;
		}

		// Add or Remove mode
		if (this.mode == MODE_ADD) {
			root = aDocument.createElementNS(EPPDomainMapFactory.NS, EPPDomainMapFactory.NS_PREFIX + ":" + ELM_ADD);
		}
		else if (this.mode == MODE_REMOVE) {
			root = aDocument.createElementNS(EPPDomainMapFactory.NS, EPPDomainMapFactory.NS_PREFIX + ":" + ELM_REMOVE);
		}
		else {
			throw new EPPEncodeException("Invalid EPPDomainAddRemove mode of " + this.mode);
		}

		// Domain Name Servers
		if ((this.hasServers()) && (this.servers.size() > 0)) {
			Element theServersElm = aDocument.createElementNS(EPPDomainMapFactory.NS,
			      EPPDomainMapFactory.NS_PREFIX + ":" + ELM_SERVER);
			root.appendChild(theServersElm);

			Object theNS = this.servers.get(0);

			// Name Server Host objects?
			if (theNS instanceof String) {
				EPPUtil.encodeVector(aDocument, theServersElm, this.servers, EPPDomainMapFactory.NS,
				      EPPDomainMapFactory.NS_PREFIX + ":" + ELM_HOST_OBJ);
			}

			// Name Server Host attributes?
			else if (theNS instanceof EPPHostAttr) {
				EPPUtil.encodeCompVector(aDocument, theServersElm, this.servers);
			}
			else {
				throw new EPPEncodeException(
				      "EPPDomainAddRemove.encode: Invalid NS server class " + theNS.getClass().getName());
			}
		}

		// Contacts
		if (this.hasContacts()) {
			if (contactsSupported()) {
				EPPUtil.encodeCompVector(aDocument, root, this.contacts);
			}
			else {
				throw new EPPEncodeException("Contacts specified when the Contact Mapping is not supported");
			}
		}

		// Statuses
		EPPUtil.encodeCompVector(aDocument, root, this.statuses);

		return root;
	}

	/**
	 * Decode the {@code EPPDomainAddRemove} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPDomainAddRemove} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement.
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {
		if (aElement.getLocalName().equals(EPPUtil.getLocalName(ELM_ADD))) {
			this.mode = MODE_ADD;
		}
		else if (aElement.getLocalName().equals(EPPUtil.getLocalName(ELM_REMOVE))) {
			this.mode = MODE_REMOVE;
		}
		else if (aElement.getLocalName().equals(EPPUtil.getLocalName(ELM_CHANGE))) {
			this.mode = MODE_CHANGE;
		}
		else {
			throw new EPPDecodeException("Invalid EPPDomainAddRemove mode of " + aElement.getLocalName());
		}

		// Change Mode
		if (this.mode == MODE_CHANGE) {
			// Registrant
			this.registrant = EPPUtil.decodeString(aElement, EPPDomainMapFactory.NS, ELM_REGISTRANT);

			// AuthInfo
			this.authInfo = (EPPAuthInfo) EPPUtil.decodeComp(aElement, EPPDomainMapFactory.NS,
			      EPPDomainMapFactory.ELM_DOMAIN_AUTHINFO, EPPAuthInfo.class);
		}
		else { // Add & Remove Mode

			// Servers
			Element theServersElm = EPPUtil.getElementByTagNameNS(aElement, EPPDomainMapFactory.NS, ELM_SERVER);

			if (theServersElm != null) {
				Element theServerElm = EPPUtil.getFirstElementChild(theServersElm);

				if (theServerElm != null) {
					if (theServerElm.getLocalName().equals(EPPUtil.getLocalName(ELM_HOST_OBJ))) {
						this.servers = EPPUtil.decodeVector(theServersElm, EPPDomainMapFactory.NS, ELM_HOST_OBJ);
					}
					else if (theServerElm.getLocalName().equals(EPPUtil.getLocalName(ELM_HOST_ATTR))) {
						this.servers = EPPUtil.decodeCompVector(theServersElm, EPPDomainMapFactory.NS, ELM_HOST_ATTR,
						      EPPHostAttr.class);
					}
					else {
						throw new EPPDecodeException(
						      "EPPDomainAddRemove.doDecode: Invalid host child element " + theServersElm.getLocalName());
					}

					if (this.servers.size() == 0) {
						this.servers = null;
					}
				}

				// end if (theServerElm != null)
			}

			// end if (theServersElm != null)
			// Contacts
			this.contacts = EPPUtil.decodeCompVector(aElement, EPPDomainMapFactory.NS, ELM_CONTACT,
			      EPPDomainContact.class);

			if (this.contacts.size() == 0) {
				this.contacts = null;
			}

			// Statuses
			this.statuses = EPPUtil.decodeCompVector(aElement, EPPDomainMapFactory.NS, ELM_STATUS, EPPDomainStatus.class);
			if (this.statuses.size() == 0) {
				this.statuses = null;
			}
		}
	}

	/**
	 * implements a deep {@code EPPDomainAddRemove} compare.
	 *
	 * @param aObject
	 *           {@code EPPDomainAddRemove} instance to compare with
	 *
	 * @return {@code true} when equal; {@code false} otherwise.
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPDomainAddRemove)) {
			cat.error("EPPDomainAddRemove.equals(): " + aObject.getClass().getName() + " not EPPDomainAddRemove instance");

			return false;
		}

		EPPDomainAddRemove theComp = (EPPDomainAddRemove) aObject;

		// Mode
		if (this.mode != theComp.mode) {
			cat.error("EPPDomainAddRemove.equals(): mode not equal");

			return false;
		}

		// Servers
		if (!EqualityUtil.equals(this.servers, theComp.servers)) {
			cat.error("EPPDomainAddRemove.equals(): servers not equal");

			return false;
		}

		// Contacts
		if (contactsSupported()) {
			if (!EqualityUtil.equals(this.contacts, theComp.contacts)) {
				cat.error("EPPDomainAddRemove.equals(): contacts not equal");

				return false;
			}
		}

		// Statuses
		if (!EqualityUtil.equals(this.statuses, theComp.statuses)) {
			cat.error("EPPDomainAddRemove.equals(): statuses not equal");

			return false;
		}

		// Registrant
		if (!EqualityUtil.equals(this.registrant, theComp.registrant)) {
			cat.error("EPPDomainAddRemove.equals(): registrant not equal");

			return false;
		}

		// AuthInfo
		if (!EqualityUtil.equals(this.authInfo, theComp.authInfo)) {
			cat.error("EPPDomainAddRemove.equals(): authInfo not equal");

			return false;
		}

		return true;
	}

	/**
	 * Clone {@code EPPDomainAddRemove}.
	 *
	 * @return clone of {@code EPPDomainAddRemove}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPDomainAddRemove clone = null;

		clone = (EPPDomainAddRemove) super.clone();

		if (this.hasServers()) {
			clone.servers = (Vector<String>) this.servers.clone();
		}

		if (this.hasContacts()) {
			clone.contacts = (Vector<EPPDomainContact>) this.contacts.clone();

			for (int i = 0; i < this.contacts.size(); i++) {
				clone.contacts.setElementAt((EPPDomainContact) (this.contacts.elementAt(i)).clone(), i);
			}
		}

		if (this.hasStatuses()) {
			clone.statuses = (Vector<EPPDomainStatus>) this.statuses.clone();
		}

		if (this.hasRegistrant()) {
			clone.registrant = this.registrant;
		}

		if (this.hasAuthInfo()) {
			clone.authInfo = (EPPAuthInfo) this.authInfo.clone();
		}

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Is the authorization information set?
	 *
	 * @return {@code true} if the authorization information is set;
	 *         {@code false} otherwise.
	 */
	public boolean hasAuthInfo() {
		if (this.authInfo != null) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Get authorization information for the change mode
	 *
	 * @return {@link EPPAuthInfo} if set; {@code null} otherwise.
	 */
	public EPPAuthInfo getAuthInfo() {
		return this.authInfo;
	}

	/**
	 * Set authorization information for the change mode
	 *
	 * @param aAuthInfo
	 *           {@link EPPAuthInfo} instance. Pass {@code null} to unset
	 *           it.
	 */
	public void setAuthInfo(EPPAuthInfo aAuthInfo) {
		if (aAuthInfo != null) {
			this.authInfo = aAuthInfo;
			this.authInfo.setRootName(EPPDomainMapFactory.NS, EPPDomainMapFactory.ELM_DOMAIN_AUTHINFO);
		}
		else {
			this.authInfo = null;
		}
	}

	/**
	 * is the registrant set?
	 *
	 * @return {@code true} if the registrant is set; {@code false} otherwise.
	 */
	public boolean hasRegistrant() {
		if (this.registrant != null) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Get registrant for the change mode
	 *
	 * @return Registrant if set; {@code null} otherwise.
	 */
	public String getRegistrant() {
		return this.registrant;
	}

	/**
	 * Set registrant for the change mode.
	 *
	 * @param aRegistrant
	 *           Registrant to set. Pass {@code null} to unset it.
	 */
	public void setRegistrant(String aRegistrant) {
		this.registrant = aRegistrant;
	}

	/**
	 * Is the {@code EPPDomainAddRemove} empty?
	 *
	 * @return {@code true} if all of the attributes are not set; {@code false}
	 *         otherwise.
	 */
	public boolean isEmpty() {
		return (!this.hasServers()) && (!this.hasContacts()) && (!this.hasStatuses()) && (!this.hasRegistrant())
		      && (!this.hasAuthInfo());
	}

	/**
	 * Gets the mode of {@code EPPDomainAddRemove}. There are two valid modes
	 * {@code EPPDomainAddRemove.MODE_ADD} and
	 * {@code EPPDomainAddRemove.MODE_REMOVE}. If no mode has been satisfied,
	 * than the mode is set to {@code EPPDomainAddRemove.MODE_NONE}.
	 *
	 * @return One of the {@code EPPDomainAddRemove_MODE} constants.
	 */
	short getMode() {
		return this.mode;
	}

	/**
	 * Sets the mode of {@code EPPDomainAddRemove}. There are two valid modes
	 * {@code EPPDomainAddRemove.MODE_ADD} and
	 * {@code EPPDomainAddRemove.MODE_REMOVE}. If no mode has been satisfied,
	 * than the mode is set to {@code EPPDomainAddRemove.MODE_NONE}
	 *
	 * @param aMode
	 *           {@code EPPDomainAddRemove.MODE_ADD} or
	 *           {@code EPPDomainAddRemove.MODE_REMOVE}.
	 */
	void setMode(short aMode) {
		this.mode = aMode;
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPDomainMapFactory.NS;
	}
}
