/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.framework.validate.v02;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPFactory;
import com.verisign.epp.codec.gen.EPPMessage;
import com.verisign.epp.codec.validate.v02.EPPValidateCheckCmd;
import com.verisign.epp.codec.validate.v02.EPPValidateMapFactory;
import com.verisign.epp.framework.EPPEvent;
import com.verisign.epp.framework.EPPEventException;
import com.verisign.epp.framework.EPPEventHandler;
import com.verisign.epp.framework.EPPEventResponse;
import com.verisign.epp.framework.EPPHandleEventException;


/**
 * The {@code EPPValidateHandler} class provides an interface for handling
 * Validate Commands in a Server implementation. EPPEvents are handled by the
 * handleEvent() method here and routed to the appropriate abstract member
 * function. Subclasses should override the abstract methods to define the
 * desired behavior of a particular command when it is received. A subclassed
 * instance of {@code EPPValidateHandler} should be registered with the
 * {@code EPPDispatcher} so that EPPEvents related to the Mapping will be
 * handled there.<br>
 *
 * @see EPPEventHandler
 * @see EPPEvent
 */
public abstract class EPPValidateHandler implements EPPEventHandler {
	/** Category for logging */
	private static Logger cat = LoggerFactory.getLogger(EPPValidateHandler.class);
	      

	/**
	 * Construct an instance of {@code EPPDomainVerificationCodeHandler}
	 */
	/**
	 * Whenever an EPPDomainVerificationCodeHandler instance is created load the
	 * corresponding Map Factory into the Codec
	 */
	public EPPValidateHandler() {
		try {
			EPPFactory.getInstance().addMapFactory(EPPValidateMapFactory.class.getName());
		}
		catch (EPPCodecException e) {
			cat.error("Couldn't load the Map Factory associated with the Validate Mapping", e);
			System.exit(1);
		}
	}

	/**
	 * Gets the XML Namespace for the Mapping.
	 *
	 * @return {@code EPPDomainVerificationCodeMapFactory.NS}
	 */
	@Override
	public String getNamespace() {
		return EPPValidateMapFactory.NS;
	}

	/**
	 * This method receives an {@code EPPEvent} and routes it to the appropriate
	 * abstract method.
	 *
	 * @param aEvent
	 *           An {@code EPPEvent} that contains the {@code EPPCommand}
	 * @param aData
	 *           Any data that a Server needs to send to this
	 *           {@code EPPDomainVerificationCodeHandler} instance
	 *
	 * @return Response to the event
	 *
	 * @exception EPPEventException
	 *               Thrown if an unrecognized {@code EPPEvent} is received
	 */
	@Override
	public EPPEventResponse handleEvent(EPPEvent aEvent, Object aData) throws EPPEventException {
		try {
			this.preHandleEvent(aEvent, aData);
		}
		catch (EPPHandleEventException e) {
			return new EPPEventResponse(e.getResponse());
		}

		EPPMessage message = aEvent.getMessage();
		EPPEventResponse response = null;

		if (message instanceof EPPValidateCheckCmd) {
			response = doValidateCheck(aEvent, aData);
		}

		try {
			this.postHandleEvent(aEvent, aData);
		}
		catch (EPPHandleEventException e) {
			return new EPPEventResponse(e.getResponse());
		}

		return response;
	}

	/**
	 * Invoked when the Validate Check command is received. Subclasses should
	 * define the behavior when a Validate Check command is received.
	 *
	 * @param aEvent
	 *           The {@code EPPEvent} that is being handled
	 * @param aData
	 *           Any data that a Server needs to send to this
	 *           {@code EPPValidateHandler}
	 *
	 * @return EPPEventResponse The response that should be sent back to the
	 *         client.
	 */
	protected abstract EPPEventResponse doValidateCheck(EPPEvent aEvent, Object aData);

	/**
	 * Handles any common behavior that all registry commands need to execute
	 * before they execute their command specific behavior. The default
	 * implementation does nothing.
	 *
	 * @param aEvent
	 *           The {@code EPPEvent} that is being handled
	 * @param aData
	 *           Any data that a Server needs to send to this
	 *           {@code EPPDomainVerificationCodeHandler}
	 *
	 * @exception EPPHandleEventException
	 *               Thrown if an error condition occurs. It must contain an
	 *               {@code EPPEventResponse}
	 */
	protected void preHandleEvent(EPPEvent aEvent, Object aData) throws EPPHandleEventException {
	}

	/**
	 * Handles any common behavior that all registry commands need to execute
	 * after they execute their command specific behavior. The default
	 * implementation does nothing
	 *
	 * @param aEvent
	 *           The {@code EPPEvent} that is being handled
	 * @param aData
	 *           Any data that a Server needs to send to this
	 *           {@code EPPDomainVerificationCodeHandler}
	 *
	 * @exception EPPHandleEventException
	 *               Thrown if an error condition occurs. It must contain an
	 *               {@code EPPEventResponse}
	 */
	protected void postHandleEvent(EPPEvent aEvent, Object aData) throws EPPHandleEventException {
	}
}
