/***********************************************************
Copyright (C) 2021 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.util.secureauthinfo;

import java.security.SecureRandom;
import java.util.Random;

import com.verisign.epp.codec.gen.EPPAuthInfo;

/**
 * Strong random authorization information factory, that follows section 3.1
 * "Secure Random Authorization Information" of the EPP Secure Authorization
 * Information for Transfer draft. There are multiple methods that can be used
 * to create an authorization information by length, with a passed
 * {@link Random}, and using a defined set of symbols to randomize.
 *
 */
public class EPPAuthInfoFactory {

	/**
	 * Array of printable ASCII symbols (0x21-0x7E) to use to randomize the
	 * authorization information password.
	 */
	private static char[] printableAsciiSymbols;

	/**
	 * Default {@link Random} class used to generate the strong random authorization information.
	 */
	private static SecureRandom random = new SecureRandom();

	/**
	 * Initialize the printableAsciiSymbols array with the printable ASCII
	 * symbols (0x21-0x7E)
	 */
	static {
		printableAsciiSymbols = new char[94];
		int theBaseSymbol = 33; // Decimal base symbol '!'

		for (int theIndex = 0; theIndex < 94; theIndex++) {
			printableAsciiSymbols[theIndex] = (char) (theBaseSymbol + theIndex);
		}
	}

	/**
	 * Create a strong random authorization information password character array
	 * value,that follows section 3.1 "Secure Random Authorization Information"
	 * of the EPP Secure Authorization Information for Transfer draft, using the
	 * set of all printable ASCII characters exception space (0x20), and a
	 * provided length.
	 *
	 * @param aLength
	 *           Length of random authorization information to create.
	 *
	 * @return Strong random authorization information password
	 */
	public static char[] createAuthInfoPassword(int aLength) {
		return createAuthInfoPassword(aLength, random, printableAsciiSymbols);
	}

	/**
	 * Create a strong random authorization information password {@code String}
	 * value,that follows section 3.1 "Secure Random Authorization Information"
	 * of the EPP Secure Authorization Information for Transfer draft, using the
	 * set of all printable ASCII characters exception space (0x20), and a
	 * provided length.
	 *
	 * @param aLength
	 *           Length of random authorization information to create.
	 *
	 * @return Strong random authorization information password
	 */
	public static String createAuthInfoPasswordStr(int aLength) {
		return new String(createAuthInfoPassword(aLength));
	}

	/**
	 * Create a strong random authorization information password character array
	 * value using a passed in length, {@link Random}, and array of symbols to
	 * randomize from.
	 *
	 * @param aLength
	 *           Length of the authorization information password to create.
	 * @param aRandom
	 *           {@link Random} instance to use to randomize the symbols in the
	 *           authorization information password.
	 * @param aSymbols
	 *           The array of symbols to randomize in the authorization
	 *           information password.
	 * @return Strong random authorization information password
	 */
	public static char[] createAuthInfoPassword(int aLength, Random aRandom, char[] aSymbols) {
		if (aLength < 1) {
			throw new IllegalArgumentException("Length of auth info password must be >= 1");
		}
		if (aSymbols.length < 2) {
			throw new IllegalArgumentException("Symbols of auth info password must be >= 2");
		}

		char[] theAuthInfoArr = new char[aLength];

		for (int theIndex = 0; theIndex < aLength; theIndex++) {
			theAuthInfoArr[theIndex] = aSymbols[aRandom.nextInt(printableAsciiSymbols.length)];
		}

		return theAuthInfoArr;
	}

	/**
	 * Create a strong random authorization information password {@code String}
	 * value using a passed in length, {@link Random}, and array of symbols to
	 * randomize from.
	 *
	 * @param aLength
	 *           Length of the authorization information password to create.
	 * @param aRandom
	 *           {@link Random} instance to use to randomize the symbols in the
	 *           authorization information password.
	 * @param aSymbols
	 *           The array of symbols to randomize in the authorization
	 *           information password.
	 * @return Strong random authorization information password
	 */
	public static String createAuthInfoPasswordStr(int aLength, Random aRandom, char[] aSymbols) {
		return new String(createAuthInfoPassword(aLength, aRandom, aSymbols));
	}

	/**
	 * Create an {@link EPPAuthInfo} instance with a strong random authorization
	 * information value, that follows section 3.1 "Secure Random Authorization
	 * Information" of the EPP Secure Authorization Information for Transfer
	 * draft, using the set of all printable ASCII characters exception space
	 * (0x20).
	 *
	 * @param aLength
	 *           Length of the authorization information password to create.
	 * @return {@link EPPAuthInfo} instance with a strong random authorization
	 *         information value.
	 */
	public static EPPAuthInfo createAuthInfo(int aLength) {
		return new EPPAuthInfo(createAuthInfoPasswordStr(aLength));
	}

}
