/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.org;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * <code>EPPOrgRole</code> contains the role type and optional role id of the
 * organization.
 */
public class EPPOrgRole implements EPPCodecComponent {

	/**
	 * Category for logging
	 */
	    private static Logger cat = LoggerFactory.getLogger(EPPOrgRole.class);

	/**
	 * XML local name for <code>EPPOrgRole</code>.
	 */
	public static final String ELM_LOCALNAME = "role";

	/**
	 * XML root tag for <code>EPPOrgRole</code>.
	 */
	public static final String ELM_NAME = EPPOrgMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML attribute name for the role status
	 */
	private static final String ATTR_ROLE_STATUS = "roleStatus";

	/**
	 * XML local name for the OPTIONAL role type element
	 */
	private static final String ELM_TYPE = "type";

	/**
	 * XML local name for the state (<code>status</code>) element.
	 */
	private final static String ELM_STATUS = "status";

	/**
	 * XML local name for the OPTIONAL role identifier element
	 */
	private static final String ELM_ROLE_ID = "roleID";

	/**
	 * The role type of the organization.
	 */
	private String type;

	/**
	 * Role statuses. Zero or more statuses can be set on create and one or more
	 * statuses must be set with an info response.
	 */
	private List<EPPOrgRoleStatus> statuses = new ArrayList<EPPOrgRoleStatus>();

	/**
	 * OPTIONAL role identifier contains a third party assigned identifier, such
	 * as IANA ID for registrars.
	 */
	private String roleId;

	/**
	 * Default constructor for <code>EPPOrgRole</code>.
	 */
	public EPPOrgRole() {
	}

	/**
	 * Constructor for <code>EPPOrgRole</code> that takes the role type.
	 * 
	 * @param aType
	 *           Organization role type
	 * 
	 */
	public EPPOrgRole(String aType) {
		this.setType(aType);
	}

	/**
	 * Constructor for <code>EPPOrgRole</code> that takes the role with a list of
	 * statuses.
	 * 
	 * @param aRole
	 *           Organization role
	 * @param aStatuses
	 *           Organization role statuses
	 */
	public EPPOrgRole(String aRole, List<EPPOrgRoleStatus> aStatuses) {
		this.setType(aRole);
		this.setStatuses(aStatuses);
	}

	/**
	 * Constructor for <code>EPPOrgRole</code> that takes the role, a single
	 * status, and a role identifier.
	 * 
	 * @param aRole
	 *           Organization role
	 * @param aStatus
	 *           Single organization role statuses
	 * @param aRoleId
	 *           Role identifier
	 */
	public EPPOrgRole(String aRole, EPPOrgRoleStatus aStatus, String aRoleId) {
		this.setType(aRole);
		this.addStatus(aStatus);
		this.setRoleId(aRoleId);
	}

	/**
	 * Constructor for <code>EPPOrgRole</code> that takes the role, a list of
	 * statuses, and a role identifier.
	 * 
	 * @param aRole
	 *           Organization role
	 * @param aStatuses
	 *           Organization role statuses
	 * @param aRoleId
	 *           Role identifier
	 */
	public EPPOrgRole(String aRole, List<EPPOrgRoleStatus> aStatuses, String aRoleId) {
		this.setType(aRole);
		this.setStatuses(aStatuses);
		this.setRoleId(aRoleId);
	}

	/**
	 * Gets the organization role type.
	 * 
	 * @return The organization role type if defined;<code>null</code> otherwise.
	 */
	public String getType() {
		return this.type;
	}

	/**
	 * Sets the organization role type.
	 * 
	 * @param aType
	 *           The organization role type
	 */
	public void setType(String aType) {
		this.type = aType;
	}

	/**
	 * Is there at least one status?
	 * 
	 * @return <code>true</code> if there at least one status; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasStatus() {
		return !this.statuses.isEmpty();
	}

	/**
	 * Gets the role statuses of the role.
	 * 
	 * @return List of role statuses of the role.
	 */
	public List<EPPOrgRoleStatus> getStatuses() {
		return this.statuses;
	}

	/**
	 * Sets the role statuses of the role.
	 * 
	 * @param aStatuses
	 *           Role statuses
	 */
	public void setStatuses(List<EPPOrgRoleStatus> aStatuses) {
		if (aStatuses == null) {
			this.statuses = new ArrayList<EPPOrgRoleStatus>();
		}
		else {
			this.statuses = aStatuses;
		}
	}

	/**
	 * Add a role status to list of role statuses.
	 * 
	 * @param aStatus
	 *           Role status to add.
	 */
	public void addStatus(EPPOrgRoleStatus aStatus) {
		this.statuses.add(aStatus);
	}

	/**
	 * Is the organization role identifier defined?
	 * 
	 * @return <code>true</code> if the organization role identifier is defined;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasRoleId() {
		return (this.roleId != null ? true : false);
	}

	/**
	 * Gets the OPTIONAL organization role identifier such as IANA ID for
	 * registrars..
	 * 
	 * @return The organization role identifier if defined;<code>null</code>
	 *         otherwise.
	 */
	public String gerRoleId() {
		return this.roleId;
	}

	/**
	 * Sets the OPTIONAL organization role identifier such as IANA ID for
	 * registrars..
	 * 
	 * @param aRoleId
	 *           The organization identifier
	 */
	public void setRoleId(String aRoleId) {
		this.roleId = aRoleId;
	}

	/**
	 * encode instance into a DOM element tree. A DOM Document is passed as an
	 * argument and functions as a factory for DOM objects. The root element
	 * associated with the instance is created and each instance attribute is
	 * appended as a child node.
	 * 
	 * @param aDocument
	 *           DOM Document, which acts is an Element factory
	 * 
	 * @return Element Root element associated with the object
	 * 
	 * @exception EPPEncodeException
	 *               Error encoding <code>EPPOrgRole</code>
	 */
	public Element encode(Document aDocument) throws EPPEncodeException {

		// Check required attributes
		if (this.type == null) {
			throw new EPPEncodeException("Undefined type in EPPOrgRole");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPOrgMapFactory.NS, ELM_NAME);

		// Type
		EPPUtil.encodeString(aDocument, root, this.type, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_TYPE);

		// Statuses
		EPPUtil.encodeList(aDocument, root, this.statuses, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_STATUS);

		// Role Id
		EPPUtil.encodeString(aDocument, root, this.roleId, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_ROLE_ID);

		return root;
	}

	/**
	 * decode a DOM element tree to initialize the instance attributes. The
	 * <code>aElement</code> argument represents the root DOM element and is used
	 * to traverse the DOM nodes for instance attribute values.
	 * 
	 * @param aElement
	 *           <code>Element</code> to decode
	 * 
	 * @exception EPPDecodeException
	 *               Error decoding <code>Element</code>
	 */
	public void decode(Element aElement) throws EPPDecodeException {
		// Type
		this.type = EPPUtil.decodeString(aElement, EPPOrgMapFactory.NS, ELM_TYPE);

		// Statuses
		this.statuses = EPPUtil.decodeEnumList(aElement, EPPOrgMapFactory.NS, ELM_STATUS, EPPOrgRoleStatus.class);

		// Role Id
		this.roleId = EPPUtil.decodeString(aElement, EPPOrgMapFactory.NS, ELM_ROLE_ID);
	}

	/**
	 * clone an <code>EPPCodecComponent</code>.
	 * 
	 * @return clone of concrete <code>EPPOrgExt</code>
	 * 
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPOrgRole clone = (EPPOrgRole) super.clone();

		// Statuses
		clone.statuses = (List) ((ArrayList) this.statuses).clone();

		return clone;
	}

	/**
	 * Implementation of <code>Object.toString</code>, which will result in an
	 * indented XML <code>String</code> representation of the concrete
	 * <code>EPPCodecComponent</code>.
	 * 
	 * @return Indented XML <code>String</code> if successful; <code>ERROR</code>
	 *         otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Compare an instance of <code>EPPOrgExt</code> with this instance.
	 * 
	 * @param aObject
	 *           Object to compare with.
	 * 
	 * @return <code>true</code> if equal; <code>false</code> otherwise.
	 */
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPOrgRole)) {
			return false;
		}

		EPPOrgRole other = (EPPOrgRole) aObject;

		// Type
		if (!EqualityUtil.equals(this.type, other.type)) {
			cat.error("EPPOrgRole.equals(): type not equal");
			return false;
		}

		// Statuses
		if (!EPPUtil.equalLists(this.statuses, other.statuses)) {
			cat.error("EPPOrgRole.equals(): statuses not equal");
			return false;
		}

		// Role Id
		if (!EqualityUtil.equals(this.roleId, other.roleId)) {
			cat.error("EPPOrgRole.equals(): roleId not equal");
			return false;
		}

		return true;
	}

	/**
	 * Returns the XML namespace associated with the
	 * <code>EPPCodecComponent</code>.
	 * 
	 * @return XML namespace for the <code>EPPCodecComponent</code>.
	 */
	@Override
	public String getNamespace() {
		return EPPOrgMapFactory.NS;
	}

}
