/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.org;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPTransId;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * Response information associated with an org object.
 * 
 * @see com.verisign.epp.codec.org.EPPOrgInfoCmd
 */
public class EPPOrgInfoResp extends EPPResponse {

	/**
	 * Category for logging
	 */
	    private static Logger cat = LoggerFactory.getLogger(EPPOrgInfoResp.class);
	      

	/**
	 * XML local name for <code>EPPOrgInfoResp</code>.
	 */
	public static final String ELM_LOCALNAME = "infData";

	/**
	 * XML root tag for <code>EPPOrgInfoResp</code>.
	 */
	public static final String ELM_NAME = EPPOrgMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML local name for the org identifier element
	 */
	private static final String ELM_ORG_ID = "id";

	/**
	 * XML local name for the Registry Object IDentifier (<code>roid</code>)
	 * element.
	 */
	private final static String ELM_ROID = "roid";

	/**
	 * XML local name for the status (<code>status</code>) element.
	 */
	private final static String ELM_STATUS = "status";

	/**
	 * XML local name for the OPTIONAL parent identifier (<code>parentId</code>)
	 * element.
	 */
	private final static String ELM_PARENT_ID = "parentId";

	/**
	 * XML local name for the OPTIONAL voice number (<code>voice</code>) element.
	 */
	private final static String ELM_VOICE = "voice";

	/**
	 * XML local name for the OPTIONAL fax number (<code>fax</code>) element.
	 */
	private final static String ELM_FAX = "fax";

	/**
	 * XML local name for the email (<code>email</code>) element.
	 */
	private final static String ELM_EMAIL = "email";

	/**
	 * XML local name for the OPTIONAL URL (<code>url</code>) element.
	 */
	private final static String ELM_URL = "url";

	/**
	 * XML local name for the client identifier (<code>clID</code>) element.
	 */
	private final static String ELM_CLIENT_ID = "clID";

	/**
	 * XML local name for the created by identifier (<code>crID</code>) element.
	 */
	private final static String ELM_CREATED_BY = "crID";

	/**
	 * XML local name for the created date (<code>crDate</code>) element.
	 */
	private final static String ELM_CREATED_DATE = "crDate";

	/**
	 * XML local name for the OPTIONAL updated by identifier (<code>upID</code>)
	 * element.
	 */
	private final static String ELM_LAST_UPDATED_BY = "upID";

	/**
	 * XML local name for the OPTIONAL updated date (<code>upDate</code>)
	 * element.
	 */
	private final static String ELM_LAST_UPDATED_DATE = "upDate";

	/**
	 * XML extension "x" attribute value for voice or fax elements.
	 */
	private final static String ATTR_EXT = "x";

	/**
	 * The org identifier of the org.
	 */
	private String orgId;

	/**
	 * Registry Object IDentifier
	 */
	private String roid;

	/**
	 * Organization roles
	 */
	private List<EPPOrgRole> roles = new ArrayList<EPPOrgRole>();

	/**
	 * Organization statuses
	 */
	private List<EPPOrgStatus> statuses = new ArrayList<EPPOrgStatus>();

	/**
	 * OPTIONAL Parent Identifier
	 */
	private String parentId;

	/**
	 * Zero to two postal information elements, represented by the
	 * {@link EPPOrgPostalDefinition} class.
	 */
	private List<EPPOrgPostalDefinition> postalInfo = new ArrayList<EPPOrgPostalDefinition>();

	/**
	 * OPTIONAL Voice number
	 */
	private String voice;

	/**
	 * OPTIONAL Voice Extension
	 */
	private String voiceExt;

	/**
	 * OPTIONAL FAX number
	 */
	private String fax;

	/**
	 * OPTIONAL FAX Extension
	 */
	private String faxExt;

	/**
	 * OPTIONAL Email Address
	 */
	private String email;

	/**
	 * OPTIONAL URL
	 */
	private String url;

	/**
	 * Org contacts
	 */
	private List<EPPOrgContact> contacts = new ArrayList<EPPOrgContact>();

	/**
	 * Sponsoring Client Identifier
	 */
	private String clientId;

	/**
	 * Identifier of the client that created the org
	 */
	private String createdBy;

	/**
	 * Date and time the org was created.
	 */
	private Date createdDate;

	/**
	 * Identifier of the client that last updated the org
	 */
	private String lastUpdatedBy;

	/**
	 * Date and time the org was last updated.
	 */
	private Date lastUpdatedDate;

	/**
	 * <code>EPPOrgInfoResp</code> default constructor.
	 */
	public EPPOrgInfoResp() {
	}

	/**
	 * <code>EPPOrgInfoResp</code> constructor that only takes the transaction
	 * identifier.
	 * 
	 * @param aTransId
	 *           Transaction Id associated with response.
	 */
	public EPPOrgInfoResp(EPPTransId aTransId) {
		super(aTransId);
	}

	/**
	 * <code>EPPOrgInfoResp</code> constructor that only takes the transaction
	 * identifier and the org identifier.
	 * 
	 * @param aTransId
	 *           Transaction Id associated with command. Set to <code>null</code>
	 *           if a client transaction identifier is not desired.
	 * @param aOrgId
	 *           Org identifier
	 * 
	 */
	public EPPOrgInfoResp(EPPTransId aTransId, String aOrgId) {
		super(aTransId);
		this.orgId = aOrgId;
	}

	/**
	 * Gets the org identifier.
	 * 
	 * @return The org identifier if defined;<code>null</code> otherwise.
	 */
	public String getOrgId() {
		return this.orgId;
	}

	/**
	 * Sets the org identifier.
	 * 
	 * @param aOrgId
	 *           The org identifier
	 */
	public void setOrgId(String aOrgId) {
		this.orgId = aOrgId;
	}

	/**
	 * @return the roid
	 */
	public String getRoid() {
		return this.roid;
	}

	/**
	 * @param aRoid
	 *           the roid to set
	 */
	public void setRoid(String aRoid) {
		this.roid = aRoid;
	}

	/**
	 * Is there at least one role?
	 * 
	 * @return <code>true</code> if there at least one role; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasRole() {
		return !this.roles.isEmpty();
	}

	/**
	 * Gets the roles of the org.
	 * 
	 * @return List of roles of the org.
	 */
	public List<EPPOrgRole> getRoles() {
		return this.roles;
	}

	/**
	 * Sets the roles of the org.
	 * 
	 * @param aRoles
	 *           Organization roles
	 */
	public void setRoles(List<EPPOrgRole> aRoles) {
		if (aRoles == null) {
			this.roles = new ArrayList<EPPOrgRole>();
		}
		else {
			this.roles = aRoles;
		}
	}

	/**
	 * Add a role to list of org roles.
	 * 
	 * @param aRole
	 *           Org role to add.
	 */
	public void addRole(EPPOrgRole aRole) {
		this.roles.add(aRole);
	}

	/**
	 * Is there at least one status?
	 * 
	 * @return <code>true</code> if there at least one status; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasStatus() {
		return !this.statuses.isEmpty();
	}

	/**
	 * Gets the statuses of the org.
	 * 
	 * @return List of statuses of the org.
	 */
	public List<EPPOrgStatus> getStatuses() {
		return this.statuses;
	}

	/**
	 * Sets the statuses of the org.
	 * 
	 * @param aStatuses
	 *           Organization statuses
	 */
	public void setStatuses(List<EPPOrgStatus> aStatuses) {
		if (aStatuses == null) {
			this.statuses = new ArrayList<EPPOrgStatus>();
		}
		else {
			this.statuses = aStatuses;
		}
	}

	/**
	 * Add a status to list of org statuses.
	 * 
	 * @param aStatus
	 *           Org status to add.
	 */
	public void addStatus(EPPOrgStatus aStatus) {
		this.statuses.add(aStatus);
	}

	/**
	 * Is the parentId defined?
	 * 
	 * @return <code>true</code> if the parentId is defined; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasParentId() {
		return (this.parentId != null ? true : false);
	}

	/**
	 * Gets the parent organization id.
	 * 
	 * @return The parent organization id if defined; <code>null</code>
	 *         otherwise.
	 */
	public String getParentId() {
		return this.parentId;
	}

	/**
	 * Sets the parent organization id.
	 * 
	 * @param aParentId
	 *           the parentId to set
	 */
	public void setParentId(String aParentId) {
		this.parentId = aParentId;
	}

	/**
	 * Is there any postal information set?
	 * 
	 * @return <code>true</code> if there is at least one
	 *         {@link EPPOrgPostalDefinition} set in the postal information;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasPostalInfo() {
		if (this.postalInfo != null && !this.postalInfo.isEmpty()) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Adds a postal definition to the postal information for the org.
	 * 
	 * @param aPostalInfo
	 *           Postal definition to add to the postal information.
	 */
	public void addPostalInfo(EPPOrgPostalDefinition aPostalInfo) {
		this.postalInfo.add(aPostalInfo);
	}

	/**
	 * Gets the postal information for the org. There can be one or two
	 * {@link EPPOrgPostalDefinition} objects in the postal information list.
	 * 
	 * @return Postal information for the org
	 */
	public List<EPPOrgPostalDefinition> getPostalInfo() {
		return this.postalInfo;
	}

	/**
	 * Sets the postal information for the org. There can be one or two
	 * {@link EPPOrgPostalDefinition} objects in the postal information list.
	 * 
	 * @param aPostalInfo
	 *           Postal information for the org.
	 */
	public void setPostalInfo(List<EPPOrgPostalDefinition> aPostalInfo) {
		this.postalInfo = aPostalInfo;
	}

	/**
	 * @return the voice
	 */
	public String getVoice() {
		return this.voice;
	}

	/**
	 * @param aVoice
	 *           the voice to set
	 */
	public void setVoice(String aVoice) {
		this.voice = aVoice;
	}

	/**
	 * @return the voiceExt
	 */
	public String getVoiceExt() {
		return this.voiceExt;
	}

	/**
	 * @param aVoiceExt
	 *           the voiceExt to set
	 */
	public void setVoiceExt(String aVoiceExt) {
		this.voiceExt = aVoiceExt;
	}

	/**
	 * @return the fax
	 */
	public String getFax() {
		return this.fax;
	}

	/**
	 * @param aFax
	 *           the fax to set
	 */
	public void setFax(String aFax) {
		this.fax = aFax;
	}

	/**
	 * @return the faxExt
	 */
	public String getFaxExt() {
		return this.faxExt;
	}

	/**
	 * @param aFaxExt
	 *           the faxExt to set
	 */
	public void setFaxExt(String aFaxExt) {
		this.faxExt = aFaxExt;
	}

	/**
	 * @return the email
	 */
	public String getEmail() {
		return this.email;
	}

	/**
	 * @param aEmail
	 *           the email to set
	 */
	public void setEmail(String aEmail) {
		this.email = aEmail;
	}

	/**
	 * Gets the URL of the website of the org.
	 * 
	 * @return URL of the website of the org.
	 */
	public String getUrl() {
		return this.url;
	}

	/**
	 * Sets the URL of the website of the org.
	 * 
	 * @param aUrl
	 *           URL of the website of the org.
	 */
	public void setUrl(String aUrl) {
		this.url = aUrl;
	}

	/**
	 * Is there any contacts set?
	 * 
	 * @return <code>true</code> if there is at least one {@link EPPOrgContact}
	 *         set; <code>false</code> otherwise.
	 */
	public boolean hasContacts() {
		if (this.contacts != null && !this.contacts.isEmpty()) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Adds a contact to the list of contacts.
	 * 
	 * @param aContact
	 *           Contact add to the list of contacts.
	 */
	public void addContact(EPPOrgContact aContact) {
		this.contacts.add(aContact);
	}

	/**
	 * @return the contacts
	 */
	public List<EPPOrgContact> getContacts() {
		return this.contacts;
	}

	/**
	 * @param aContacts
	 *           the contacts to set
	 */
	public void setContacts(List<EPPOrgContact> aContacts) {
		this.contacts = aContacts;
	}

	/**
	 * @return the clientId
	 */
	public String getClientId() {
		return this.clientId;
	}

	/**
	 * @param aClientId
	 *           the clientId to set
	 */
	public void setClientId(String aClientId) {
		this.clientId = aClientId;
	}

	/**
	 * @return the createdBy
	 */
	public String getCreatedBy() {
		return this.createdBy;
	}

	/**
	 * @param aCreatedBy
	 *           the createdBy to set
	 */
	public void setCreatedBy(String aCreatedBy) {
		this.createdBy = aCreatedBy;
	}

	/**
	 * @return the createdDate
	 */
	public Date getCreatedDate() {
		return this.createdDate;
	}

	/**
	 * @param aCreatedDate
	 *           the createdDate to set
	 */
	public void setCreatedDate(Date aCreatedDate) {
		this.createdDate = aCreatedDate;
	}

	/**
	 * @return the lastUpdatedBy
	 */
	public String getLastUpdatedBy() {
		return this.lastUpdatedBy;
	}

	/**
	 * @param aLastUpdatedBy
	 *           the lastUpdatedBy to set
	 */
	public void setLastUpdatedBy(String aLastUpdatedBy) {
		this.lastUpdatedBy = aLastUpdatedBy;
	}

	/**
	 * Gets the last updated date.
	 * 
	 * @return the lastUpdatedDate Last updated date if set; <code>null</code>
	 *         otherwise.
	 */
	public Date getLastUpdatedDate() {
		return this.lastUpdatedDate;
	}

	/**
	 * Sets the last updated date.
	 * 
	 * @param aLastUpdatedDate
	 *           Last updated date and time
	 */
	public void setLastUpdatedDate(Date aLastUpdatedDate) {
		this.lastUpdatedDate = aLastUpdatedDate;
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * <code>EPPOrgInfoResp</code> instance.
	 * 
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 * 
	 * @return Element Root DOM Element representing the EPPOrgInfoResp instance.
	 * 
	 * @exception EPPEncodeException
	 *               Unable to encode EPPOrgInfoResp instance.
	 */
	protected Element doEncode(Document aDocument) throws EPPEncodeException {

		Element theElm = null;
		Text theVal = null;

		// Check required attributes
		if (this.orgId == null) {
			throw new EPPEncodeException("Undefined orgId in EPPOrgInfoResp");
		}
		if (this.roles.isEmpty()) {
			throw new EPPEncodeException("No roles set in EPPOrgInfoResp");
		}
		if (this.statuses.isEmpty()) {
			throw new EPPEncodeException("No statuses set in EPPOrgInfoResp");
		}
		if (this.clientId == null) {
			throw new EPPEncodeException("Undefined clientId in EPPOrgInfoResp");
		}
		if (this.createdBy == null) {
			throw new EPPEncodeException("Undefined createdBy in EPPOrgInfoResp");
		}
		if (this.createdDate == null) {
			throw new EPPEncodeException("Undefined createdDate in EPPOrgInfoResp");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPOrgMapFactory.NS, ELM_NAME);

		// Org Identifier
		EPPUtil.encodeString(aDocument, root, this.orgId, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_ORG_ID);

		// Roid
		EPPUtil.encodeString(aDocument, root, this.roid, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_ROID);

		// Roles
		EPPUtil.encodeCompList(aDocument, root, this.roles);

		// Statuses
		EPPUtil.encodeList(aDocument, root, this.statuses, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_STATUS);

		// Parent Id
		EPPUtil.encodeString(aDocument, root, this.parentId, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_PARENT_ID);

		// Postal Information
		EPPUtil.encodeCompList(aDocument, root, this.postalInfo);

		// Voice and Voice Extension
		if (this.voice != null) {
			theElm = aDocument.createElementNS(EPPOrgMapFactory.NS, EPPOrgMapFactory.NS_PREFIX + ":" + ELM_VOICE);
			theVal = aDocument.createTextNode(this.voice);
			theElm.appendChild(theVal);

			if (this.voiceExt != null) {
				theElm.setAttribute(ATTR_EXT, this.voiceExt);
			}

			root.appendChild(theElm);
		}

		// Fax and Fax Extension
		if (this.fax != null) {
			theElm = aDocument.createElementNS(EPPOrgMapFactory.NS, EPPOrgMapFactory.NS_PREFIX + ":" + ELM_FAX);
			theVal = aDocument.createTextNode(this.fax);
			theElm.appendChild(theVal);

			if (this.faxExt != null) {
				theElm.setAttribute(ATTR_EXT, this.faxExt);
			}

			root.appendChild(theElm);
		}

		// Email
		EPPUtil.encodeString(aDocument, root, this.email, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_EMAIL);

		// URL
		EPPUtil.encodeString(aDocument, root, this.url, EPPOrgMapFactory.NS, EPPOrgMapFactory.NS_PREFIX + ":" + ELM_URL);

		// Contacts
		EPPUtil.encodeCompList(aDocument, root, this.contacts);

		// Client Id
		EPPUtil.encodeString(aDocument, root, this.clientId, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_CLIENT_ID);

		// Created By
		EPPUtil.encodeString(aDocument, root, this.createdBy, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_CREATED_BY);

		// Created Date
		EPPUtil.encodeTimeInstant(aDocument, root, this.createdDate, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_CREATED_DATE);

		// Last Updated By
		EPPUtil.encodeString(aDocument, root, this.lastUpdatedBy, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_LAST_UPDATED_BY);

		// Last Updated Date
		EPPUtil.encodeTimeInstant(aDocument, root, this.lastUpdatedDate, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_LAST_UPDATED_DATE);

		return root;
	}

	/**
	 * Decode the <code>EPPOrgInfoResp</code> attributes from the aElement DOM
	 * Element tree.
	 * 
	 * @param aElement
	 *           Root DOM Element to decode <code>EPPOrgInfoResp</code> from.
	 * 
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	protected void doDecode(Element aElement) throws EPPDecodeException {

		Element theElm = null;

		// Org Identifier
		this.orgId = EPPUtil.decodeString(aElement, EPPOrgMapFactory.NS, ELM_ORG_ID);

		// Roid
		this.roid = EPPUtil.decodeString(aElement, EPPOrgMapFactory.NS, ELM_ROID);

		// Roles
		this.roles = EPPUtil.decodeCompList(aElement, EPPOrgMapFactory.NS, EPPOrgRole.ELM_NAME, EPPOrgRole.class);

		// Statuses
		this.statuses = EPPUtil.decodeEnumList(aElement, EPPOrgMapFactory.NS, ELM_STATUS, EPPOrgStatus.class);

		// Parent Id
		this.parentId = EPPUtil.decodeString(aElement, EPPOrgMapFactory.NS, ELM_PARENT_ID);

		// Postal Information
		this.postalInfo = EPPUtil.decodeCompList(aElement, EPPOrgMapFactory.NS, EPPOrgPostalDefinition.ELM_NAME,
		      EPPOrgPostalDefinition.class);

		// Voice and Voice Extension
		theElm = EPPUtil.getElementByTagNameNS(aElement, EPPOrgMapFactory.NS, ELM_VOICE);
		if (theElm != null) {
			this.voice = EPPUtil.getTextContent(theElm);
			this.voiceExt = EPPUtil.decodeStringAttr(theElm, ATTR_EXT);
		}
		else {
			this.voice = null;
			this.voiceExt = null;
		}

		// Fax and Fax Extension
		theElm = EPPUtil.getElementByTagNameNS(aElement, EPPOrgMapFactory.NS, ELM_FAX);
		if (theElm != null) {
			this.fax = EPPUtil.getTextContent(theElm);
			this.faxExt = EPPUtil.decodeStringAttr(theElm, ATTR_EXT);
		}
		else {
			this.fax = null;
			this.faxExt = null;
		}

		// Email
		this.email = EPPUtil.decodeString(aElement, EPPOrgMapFactory.NS, ELM_EMAIL);

		// URL
		this.url = EPPUtil.decodeString(aElement, EPPOrgMapFactory.NS, ELM_URL);

		// Contacts
		this.contacts = EPPUtil.decodeCompList(aElement, EPPOrgMapFactory.NS, EPPOrgContact.ELM_NAME,
		      EPPOrgContact.class);

		// Client Id
		this.clientId = EPPUtil.decodeString(aElement, EPPOrgMapFactory.NS, ELM_CLIENT_ID);

		// Created By
		this.createdBy = EPPUtil.decodeString(aElement, EPPOrgMapFactory.NS, ELM_CREATED_BY);

		// Created Date
		this.createdDate = EPPUtil.decodeTimeInstant(aElement, EPPOrgMapFactory.NS, ELM_CREATED_DATE);

		// Last Updated By
		this.lastUpdatedBy = EPPUtil.decodeString(aElement, EPPOrgMapFactory.NS, ELM_LAST_UPDATED_BY);

		// Last Updated Date
		this.lastUpdatedDate = EPPUtil.decodeTimeInstant(aElement, EPPOrgMapFactory.NS, ELM_LAST_UPDATED_DATE);
	}

	/**
	 * Clone <code>EPPOrgInfoResp</code>.
	 * 
	 * @return clone of <code>EPPOrgInfoResp</code>
	 * 
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPOrgInfoResp clone = (EPPOrgInfoResp) super.clone();

		// Roles
		clone.roles = (List) ((ArrayList) this.roles).clone();

		// Statuses
		clone.statuses = (List) ((ArrayList) this.statuses).clone();

		if (this.hasPostalInfo()) {
			clone.postalInfo = (List) ((ArrayList) this.postalInfo).clone();
		}

		if (this.hasContacts()) {
			clone.contacts = (List) ((ArrayList) this.contacts).clone();
		}

		return clone;
	}

	/**
	 * Gets the EPP response type associated with <code>EPPOrgInfoResp</code>.
	 * 
	 * @return <code>EPPOrgInfoResp.ELM_NAME</code>
	 */
	public String getType() {
		return ELM_NAME;
	}

	/**
	 * Gets the EPP command namespace associated with <code>EPPOrgInfoResp</code>
	 * .
	 * 
	 * @return <code>EPPOrgMapFactory.NS</code>
	 */
	public String getNamespace() {
		return EPPOrgMapFactory.NS;
	}

	/**
	 * Compare an instance of <code>EPPOrgInfoResp</code> with this instance.
	 * 
	 * @param aObject
	 *           Object to compare with.
	 * 
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPOrgInfoResp)) {
			return false;
		}

		EPPOrgInfoResp other = (EPPOrgInfoResp) aObject;

		// Org Identifier
		if (!EqualityUtil.equals(this.orgId, other.orgId)) {
			cat.error("EPPOrgInfoResp.equals(): orgId not equal");
			return false;
		}

		// Roid
		if (!EqualityUtil.equals(this.roid, other.roid)) {
			cat.error("EPPOrgInfoResp.equals(): roid not equal");
			return false;
		}

		// Roles
		if (!EPPUtil.equalLists(this.roles, other.roles)) {
			cat.error("EPPOrgInfoResp.equals(): roles not equal");
			return false;
		}

		// Statuses
		if (!EPPUtil.equalLists(this.statuses, other.statuses)) {
			cat.error("EPPOrgInfoResp.equals(): statuses not equal");
			return false;
		}

		// Parent Id
		if (!EqualityUtil.equals(this.parentId, other.parentId)) {
			cat.error("EPPOrgInfoResp.equals(): parentId not equal");
			return false;
		}

		// Parent Id
		if (!EqualityUtil.equals(this.parentId, other.parentId)) {
			cat.error("EPPOrgInfoResp.equals(): parentId not equal");
			return false;
		}

		// Postal Information
		if (!EPPUtil.equalLists(this.postalInfo, other.postalInfo)) {
			cat.error("EPPOrgInfoResp.equals(): postalInfo not equal");
			return false;
		}

		// Voice
		if (!EqualityUtil.equals(this.voice, other.voice)) {
			cat.error("EPPOrgInfoResp.equals(): voice not equal");
			return false;
		}

		// Voice Ext
		if (!EqualityUtil.equals(this.voiceExt, other.voiceExt)) {
			cat.error("EPPOrgInfoResp.equals(): voiceExt not equal");
			return false;
		}

		// Fax
		if (!EqualityUtil.equals(this.fax, other.fax)) {
			cat.error("EPPOrgInfoResp.equals(): fax not equal");
			return false;
		}

		// Fax Ext
		if (!EqualityUtil.equals(this.faxExt, other.faxExt)) {
			cat.error("EPPOrgInfoResp.equals(): faxExt not equal, this.faxExt = " + this.faxExt + ", other.faxExt = "
			      + other.faxExt);
			return false;
		}

		// Email
		if (!EqualityUtil.equals(this.email, other.email)) {
			cat.error("EPPOrgInfoResp.equals(): email not equal");
			return false;
		}

		// Url
		if (!EqualityUtil.equals(this.url, other.url)) {
			cat.error("EPPOrgInfoResp.equals(): url not equal");
			return false;
		}

		// Contacts
		if (!EPPUtil.equalLists(this.contacts, other.contacts)) {
			cat.error("EPPOrgInfoResp.equals(): contacts not equal");
			return false;
		}

		// Client Id
		if (!EqualityUtil.equals(this.clientId, other.clientId)) {
			cat.error("EPPOrgInfoResp.equals(): clientId not equal");
			return false;
		}

		// Created By
		if (!EqualityUtil.equals(this.createdBy, other.createdBy)) {
			cat.error("EPPOrgInfoResp.equals(): createdBy not equal");
			return false;
		}

		// Created Date
		if (!EqualityUtil.equals(this.createdDate, other.createdDate)) {
			cat.error("EPPOrgInfoResp.equals(): createdDate not equal");
			return false;
		}

		// Last Updated By
		if (!EqualityUtil.equals(this.lastUpdatedBy, other.lastUpdatedBy)) {
			cat.error("EPPOrgInfoResp.equals(): lastUpdatedBy not equal");
			return false;
		}

		// Last Updated Date
		if (!EqualityUtil.equals(this.lastUpdatedDate, other.lastUpdatedDate)) {
			cat.error("EPPOrgInfoResp.equals(): lastUpdateDate not equal");
			return false;
		}

		return true;
	}

	/**
	 * Implementation of <code>Object.toString</code>, which will result in an
	 * indented XML <code>String</code> representation of the concrete
	 * <code>EPPCodecComponent</code>.
	 * 
	 * @return Indented XML <code>String</code> if successful; <code>ERROR</code>
	 *         otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

}