/***********************************************************
Copyright (C) 2019 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.loginsecpolicy.v04;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.codec.registry.policy.EPPRegistrySystemInfoInterface;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPLoginSecPolicySystem} contains all of the login security policy
 * information associated with the system.
 */
public class EPPLoginSecPolicySystem implements EPPCodecComponent, EPPRegistrySystemInfoInterface {

	/**
	 * XML local name for {@code EPPLoginSecPolicySystem}.
	 */
	public static final String ELM_LOCALNAME = "system";

	/**
	 * XML root tag for {@code EPPLoginSecPolicySystem}.
	 */
	public static final String ELM_NAME = EPPLoginSecPolicyExtFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML Element Name of {@code userAgentSupport} attribute.
	 */
	private final static String ELM_USER_AGENT_SUPPORT = "userAgentSupport";

	/**
	 * Category for logging
	 */
	    private static Logger cat = LoggerFactory.getLogger(EPPLoginSecPolicySystem.class);
	      

	/**
	 * The login password format policy.
	 */
	private EPPLoginSecPolicyPassword password;

	/**
	 * OPTIONAL boolean value that indicates the server supports the
	 * &lt;loginSec:userAgentSupport&gt; element, with the default value of "0" (or
	 * "false"). A value of "1" (or "true") means that the server processes the
	 * &lt;loginSecPolicy:userAgentSupport&gt; element. A value of "0" (or "false") means that
	 * the server ignores the &lt;loginSecPolicy:userAgentSupport&gt; element if passed by the
	 * client.
	 */
	private Boolean userAgentSupport;

	/**
	 * Security events
	 */
	private List<EPPLoginSecPolicyEvent> events = new ArrayList<EPPLoginSecPolicyEvent>();

	/**
	 * Default constructor for {@code EPPLoginSecPolicySystem}.
	 */
	public EPPLoginSecPolicySystem() {
	}

	/**
	 * Constructor for {@code EPPLoginSecPolicySystem} that takes the required
	 * password attribute.
	 * 
	 * @param aPassword
	 *           The login password format policy
	 */
	public EPPLoginSecPolicySystem(EPPLoginSecPolicyPassword aPassword) {
		this.password = aPassword;
	}

	/**
	 * Constructor for {@code EPPLoginSecPolicySystem} that the required password
	 * attribute and the optional attributes with a single event.
	 *
	 * @param aPassword
	 *           The login password format policy
	 * @param aUserAgentSupport
	 *           OPTIONAL boolean value that indicates the server supports the
	 *           &lt;loginSec:userAgentSupport&gt; element. Set to {@code null} if
	 *           undefined.
	 * @param aEvent
	 *           Login security event. Set to {@code null} if undefined.
	 */
	public EPPLoginSecPolicySystem(EPPLoginSecPolicyPassword aPassword, Boolean aUserAgentSupport,
	      EPPLoginSecPolicyEvent aEvent) {
		this.password = aPassword;
		this.userAgentSupport = aUserAgentSupport;
		this.addEvent(aEvent);
	}

	/**
	 * Constructor for {@code EPPLoginSecPolicySystem} that the required password
	 * attribute and the optional attributes with a list of events.
	 *
	 * @param aPassword
	 *           The login password format policy
	 * @param aUserAgentSupport
	 *           OPTIONAL boolean value that indicates the server supports the
	 *           &lt;loginSec:userAgentSupport&gt; element. Set to {@code null} if
	 *           undefined.
	 * @param aEvents
	 *           List of events.
	 */
	public EPPLoginSecPolicySystem(EPPLoginSecPolicyPassword aPassword, Boolean aUserAgentSupport,
	      List<EPPLoginSecPolicyEvent> aEvents) {
		this.password = aPassword;
		this.userAgentSupport = aUserAgentSupport;
		this.setEvents(aEvents);
	}

	/**
	 * Gets the login password format policy.
	 *
	 * @return the password format policy if defined; {@code null} otherwise.
	 */
	public EPPLoginSecPolicyPassword getPassword() {
		return this.password;
	}

	/**
	 * Sets the login password format policy.
	 *
	 * @param aPassword
	 *           the password format policy to set
	 */
	public void setPassword(EPPLoginSecPolicyPassword aPassword) {
		this.password = aPassword;
	}

	/**
	 * Is the user agent flag defined?
	 *
	 * @return {@code true} if the user agent flag is defined; {@code false}
	 *         otherwise.
	 */
	public boolean hasUserAgentSupport() {
		return (this.userAgentSupport != null ? true : false);
	}

	/**
	 * Gets the OPTIONAL boolean value that indicates the server supports the
	 * &lt;loginSec:userAgentSupport&gt; element.
	 *
	 * @return {@code true} or {@code false} indicating whether the server
	 *         supports the &lt;loginSec:userAgentSupport&gt; element if defined;
	 *         {@code null} otherwise.
	 */
	public Boolean getUserAgentSupport() {
		return this.userAgentSupport;
	}

	/**
	 * Sets the OPTIONAL boolean value that indicates the server supports the
	 * &lt;loginSec:userAgentSupport&gt; element.
	 *
	 * @param aUserAgentSupport
	 *           {@code true} or {@code false} indicating whether the server
	 *           supports the &lt;loginSec:userAgentSupport&gt; element. Set to
	 *           {@code null} if undefined.
	 */
	public void setUserAgentSupport(Boolean aUserAgentSupport) {
		this.userAgentSupport = aUserAgentSupport;
	}

	/**
	 * Are the events defined?
	 *
	 * @return {@code true} if the evets are defined; {@code false} otherwise.
	 */
	public boolean hasEvens() {
		return !this.events.isEmpty();
	}

	/**
	 * Gets the list of events if defined.
	 *
	 * @return List of events if defined; empty list otherwise.
	 */
	public List<EPPLoginSecPolicyEvent> getEvents() {
		return this.events;
	}

	/**
	 * Adds an event to the list of events.
	 *
	 * @param aEvent
	 *           The event to add.
	 */
	public void addEvent(EPPLoginSecPolicyEvent aEvent) {
		if (aEvent == null) {
			return;
		}

		this.events.add(aEvent);
	}

	/**
	 * Sets the list of events.
	 *
	 * @param aEvents
	 *           The events to set.
	 */
	public void setEvents(List<EPPLoginSecPolicyEvent> aEvents) {
		if (aEvents == null) {
			this.events = new ArrayList<EPPLoginSecPolicyEvent>();
		}
		else {
			this.events = aEvents;
		}
	}

	/**
	 * Encode instance into a DOM element tree. A DOM Document is passed as an
	 * argument and functions as a factory for DOM objects. The root element
	 * associated with the instance is created and each instance attribute is
	 * appended as a child node.
	 *
	 * @param aDocument
	 *           DOM Document, which acts is an Element factory
	 *
	 * @return Element Root element associated with the object
	 *
	 * @exception EPPEncodeException
	 *               Error encoding {@code EPPLoginSecPolicySystem}
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {

		if (aDocument == null) {
			throw new EPPEncodeException("aDocument is null in EPPLoginSecPolicySystem.encode(Document)");
		}
		if (this.password == null) {
			throw new EPPEncodeException("password is null in EPPLoginSecPolicySystem.encode(Document)");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPLoginSecPolicyExtFactory.NS, ELM_NAME);

		// Password
		EPPUtil.encodeComp(aDocument, root, this.password);

		// User Agent
		EPPUtil.encodeBoolean(aDocument, root, this.userAgentSupport, EPPLoginSecPolicyExtFactory.NS,
		      EPPLoginSecPolicyExtFactory.NS_PREFIX + ":" + ELM_USER_AGENT_SUPPORT);

		// Events
		EPPUtil.encodeCompList(aDocument, root, this.events);

		return root;
	}

	/**
	 * Decode a DOM element tree to initialize the instance attributes. The
	 * {@code aElement} argument represents the root DOM element and is used to
	 * traverse the DOM nodes for instance attribute values.
	 *
	 * @param aElement
	 *           {@code Element} to decode
	 *
	 * @exception EPPDecodeException
	 *               Error decoding {@code Element}
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {

		// Password
		this.password = (EPPLoginSecPolicyPassword) EPPUtil.decodeComp(aElement, EPPLoginSecPolicyExtFactory.NS,
		      EPPLoginSecPolicyPassword.ELM_LOCALNAME, EPPLoginSecPolicyPassword.class);

		// User Agent
		this.userAgentSupport = EPPUtil.decodeBoolean(aElement, EPPLoginSecPolicyExtFactory.NS, ELM_USER_AGENT_SUPPORT);

		// Events
		this.events = EPPUtil.decodeCompList(aElement, EPPLoginSecPolicyExtFactory.NS,
		      EPPLoginSecPolicyEvent.ELM_LOCALNAME, EPPLoginSecPolicyEvent.class);
	}

	/**
	 * Clone an {@code EPPCodecComponent} instance.
	 *
	 * @return clone of concrete {@code EPPLoginSecPolicySystem}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {

		EPPLoginSecPolicySystem clone = (EPPLoginSecPolicySystem) super.clone();

		// Events
		clone.events = new ArrayList<EPPLoginSecPolicyEvent>();
		for (EPPLoginSecPolicyEvent event : this.events) {
			clone.events.add((EPPLoginSecPolicyEvent) event.clone());
		}

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Compare an instance of {@code EPPLoginSecPolicySystem} with this instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return {@code true} if equal; {@code false} otherwise.
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPLoginSecPolicySystem)) {
			cat.error("EPPLoginSecPolicySystem.equals(): " + aObject.getClass().getName()
			      + " not EPPLoginSecPolicySystem instance");

			return false;
		}

		EPPLoginSecPolicySystem other = (EPPLoginSecPolicySystem) aObject;

		// Password
		if (!EqualityUtil.equals(this.password, other.password)) {
			cat.error("EPPLoginSecPolicySystem.equals(): password not equal");
			return false;
		}

		// User Agent
		if (!EqualityUtil.equals(this.userAgentSupport, other.userAgentSupport)) {
			cat.error("EPPLoginSecPolicySystem.equals(): userAgentSupport not equal");
			return false;
		}

		// Events
		if (!EqualityUtil.equals(this.events, other.events)) {
			cat.error("EPPLoginSecPolicySystem.equals(): events not equal");
			return false;
		}

		return true;
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPLoginSecPolicyExtFactory.NS;
	}

}
