/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.gen;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * Represents an EPPProtocol extension that is sent by an EPP Client and
 * received by an EPP Server. An {@code EPPProtocolExtension} can be encoded and
 * decoded by {@code EPPCodec}. <br>
 * <br>
 * Every EPPProtocol extension must extend {@code EPPProtocolExtension} and
 * implement the Template Method Design Pattern {@code doGenEncode} and
 * {@code doGenDecode} methods. An {@code EPPProtocolExtension} client will call
 * {@code encode} or {@code decode}, which in turn will call {@code doGenEncode}
 * or {@code doGenDecode}, respectively. There is one derived
 * {@code EPPProtocolExtension} for each type of extension defined in the
 * general EPP Specification.
 */
public abstract class EPPProtocolExtension implements EPPMessage {
	/** XML root tag name for {@code EPPProtocolExtension}. */
	final static String ELM_NAME = "extension";

	/**
	 * Allocates a new {@code EPPProtocolExtension} with default attribute
	 * values.
	 */
	public EPPProtocolExtension() {
	}

	/**
	 * Gets the EPP namespace associated with the {@code EPPProtocolExtension}.
	 *
	 * @return Namespace URI associated with the {@code EPPProtocolExtension}.
	 */
	abstract public String getNamespace();

	/**
	 * encode {@code EPPProtocolExtension} into a DOM element tree. The
	 * &lt;extensiongt; element is created and the attribute nodes are appending
	 * as children. This method is a <i>Template Method</i> in the Template
	 * Method Design Pattern.
	 *
	 * @param aDocument
	 *           DOM Document to create elements from
	 *
	 * @return &lt;extensiongt; root element tree.
	 *
	 * @exception EPPEncodeException
	 *               Error encoding the DOM element tree.
	 */
	public Element encode(Document aDocument) throws EPPEncodeException {
		// <extension>
		Element root = aDocument.createElementNS(EPPCodec.NS, ELM_NAME);

		// Encode the extension element (derived class).
		Element extElm = doEncode(aDocument);

		if (extElm != null) {
			root.appendChild(extElm);
		}

		return root;
	}

	/**
	 * decode {@code EPPProtocolExtension} from a DOM element tree. The
	 * "extension" element needs to be the value of the {@code aElement}
	 * argument. This method is a <i>Template Method</i> in the Template Method
	 * Design Pattern.
	 *
	 * @param aElement
	 *           &lt;extensiongt; root element tree.
	 *
	 * @exception EPPDecodeException
	 *               Error decoding the DOM element tree.
	 */
	public void decode(Element aElement) throws EPPDecodeException {
		Element extensionTypeElm = EPPUtil.getFirstElementChild(aElement);

		if (extensionTypeElm == null) {
			throw new EPPDecodeException("No child protocol extension Element found");
		}

		// do the necessary decode in the child class for the ext element
		doDecode(extensionTypeElm);

		// Extension Element
	}

	/**
	 * implements a deep {@code EPPProtocolExtension} compare.
	 *
	 * @param aObject
	 *           {@code EPPProtocolExtension} instance to compare with
	 *
	 * @return {@code true} if equal; {@code false} otherwise.
	 */
	public boolean equals(Object aObject) {
		return true;
	}

	/**
	 * Clone {@code EPPProtocolExtension}.
	 *
	 * @return clone of {@code EPPProtocolExtension}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPProtocolExtension clone = null;

		clone = (EPPProtocolExtension) super.clone();

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Encodes the atributes of a protocol extension of
	 * {@code EPPProtocolExtension}. An example of a protocol extension is
	 * {@code EPPAlert}. {@code encode} is a <i>Template Method</i> and this
	 * method is a <i>Primitive Operation</i> within the Template Method Design
	 * Pattern.
	 *
	 * @param aDocument
	 *           DOM document used as a factory of DOM objects.
	 *
	 * @return instance root DOM element along with attribute child nodes.
	 *
	 * @exception EPPEncodeException
	 *               Error encoding the DOM element tree.
	 */
	protected abstract Element doEncode(Document aDocument) throws EPPEncodeException;

	/**
	 * Decodes the atributes of a general extension of
	 * {@code EPPProtocolExtension}. An example of a protocol extension is
	 * {@code EPPAlert}. {@code decode} is a <i>Template Method</i> and this
	 * method is a <i>Primitive Operation</i> within the Template Method Design
	 * Pattern.
	 *
	 * @param aElement
	 *           root DOM element associated with instance
	 *
	 * @exception EPPDecodeException
	 *               Error decoding the DOM element tree.
	 */
	protected abstract void doDecode(Element aElement) throws EPPDecodeException;
}
