/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.domain;

import java.util.Date;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPTransId;
import com.verisign.epp.codec.gen.EPPUtil;

/**
 * Represents an EPP Domain &lt;domain:renData&gt; response to a {@code
 * EPPDomainRenewCmd}. When a &lt;renew&gt; command has been processed
 * successfully, the EPP &lt;resData&gt; element MUST contain a child
 * &lt;domain:renData&gt; element that identifies the domain namespace and the
 * location of the domain schema. The &lt;domain:name&gt; element contains the
 * following child elements:<br>
 * <br>
 *
 * <ul>
 * <li>A &lt;domain:name&gt; element that contains the fully qualified domain
 * name that has been created or whose validity period has been extended. Use
 * {@code getName} and {@code setName} to get and set the element.</li>
 * <li>An OPTIONAL &lt;domain:exDate&gt; element that contains the end of the
 * domain's validity period. Use {@code getExpirationDate} and
 * {@code setExpirationDate} to get and set the element.</li>
 * </ul>
 * 
 * @see com.verisign.epp.codec.domain.EPPDomainRenewCmd
 */
public class EPPDomainRenewResp extends EPPResponse {
	/**
	 * XML local name for {@code EPPDomainRenewResp}.
	 */
	public static final String ELM_LOCALNAME = "renData";

	/**
	 * XML Element Name of {@code EPPDomainRenewResp} root element.
	 */
	public static final String ELM_NAME = EPPDomainMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/** XML Element Name for the {@code name} attribute. */
	private final static String ELM_DOMAIN_NAME = "name";

	/** XML Element Name for the {@code expirationDate} attribute. */
	private final static String ELM_EXPIRATION_DATE = "exDate";

	/**
	 * fully qualified domain name that has been created or whose validity period
	 * has been extended.
	 */
	private String name = null;

	/** the end of the domain's validity period. */
	private Date expirationDate = null;

	/**
	 * {@code EPPDomainRenewResp} default constructor. Must call required setter
	 * methods before encode. the defaults include the following: <br>
	 * <br>
	 *
	 * <ul>
	 * <li>name is set to {@code null}</li>
	 * <li>expiration date is set to {@code null}</li>
	 * </ul>
	 *
	 * <br>
	 * The name must be set before invoking {@code encode}.
	 */
	public EPPDomainRenewResp() {
	}

	/**
	 * {@code EPPDomainRenewResp} constructor that takes the required attribute
	 * values as parameters.
	 *
	 * @param aTransId
	 *           Transaction Id associated with response.
	 * @param aName
	 *           Domain name
	 */
	public EPPDomainRenewResp(EPPTransId aTransId, String aName) {
		super(aTransId);

		this.name = aName;
	}

	/**
	 * {@code EPPDomainRenewResp} constructor that takes the required attribute
	 * values as parameters.
	 *
	 * @param aTransId
	 *           transaction Id associated with response.
	 * @param aName
	 *           domain name
	 * @param aExpirationDate
	 *           expiration date of the domain
	 */
	public EPPDomainRenewResp(EPPTransId aTransId, String aName, Date aExpirationDate) {
		super(aTransId);

		this.name = aName;
		this.expirationDate = aExpirationDate;
	}

	/**
	 * Gets the EPP command type associated with {@code EPPDomainRenewResp}.
	 *
	 * @return EPPDomainRenewResp.ELM_NAME
	 */
	@Override
	public String getType() {
		return ELM_NAME;
	}

	/**
	 * Gets the EPP command Namespace associated with {@code EPPDomainRenewResp}.
	 *
	 * @return {@code EPPDomainMapFactory.NS}
	 */
	@Override
	public String getNamespace() {
		return EPPDomainMapFactory.NS;
	}

	/**
	 * Compare an instance of {@code EPPDomainRenewResp} with this instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return {@code true} when equal; {@code false} otherwise.
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPDomainRenewResp)) {
			return false;
		}

		if (!super.equals(aObject)) {
			return false;
		}

		EPPDomainRenewResp theComp = (EPPDomainRenewResp) aObject;

		// Name
		if (!((this.name == null) ? (theComp.name == null) : this.name.equals(theComp.name))) {
			return false;
		}

		// Expiration Date
		if (!((this.expirationDate == null) ? (theComp.expirationDate == null) :
		      this.expirationDate.equals(theComp.expirationDate))) {
			return false;
		}

		return true;
	}

	/**
	 * Clone {@code EPPDomainRenewResp}.
	 *
	 * @return clone of {@code EPPDomainRenewResp}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPDomainRenewResp clone = (EPPDomainRenewResp) super.clone();

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Gets the domain name
	 *
	 * @return Domain Name {@code String} instance if defined; {@code null}
	 *         otherwise.
	 */
	public String getName() {
		return this.name;
	}

	/**
	 * Sets the domain name.
	 *
	 * @param aName
	 *           Domain Name
	 */
	public void setName(String aName) {
		this.name = aName;
	}

	/**
	 * Gets the expiration date and time of the domain.
	 *
	 * @return Expiration date and time of the domain if defined; {@code null}
	 *         otherwise.
	 */
	public Date getExpirationDate() {
		return this.expirationDate;
	}

	/**
	 * Sets the expiration date and time of the domain.
	 *
	 * @param aExpirationDate
	 *           Expiration date and time of the domain.
	 */
	public void setExpirationDate(Date aExpirationDate) {
		this.expirationDate = aExpirationDate;
	}

	/**
	 * Validate the state of the {@code EPPDomainRenewResp} instance. A valid
	 * state means that all of the required attributes have been set. If
	 * validateState returns without an exception, the state is valid. If the
	 * state is not valid, the EPPCodecException will contain a description of
	 * the error. throws EPPCodecException State error. This will contain the
	 * name of the attribute that is not valid.
	 *
	 * @throws EPPCodecException
	 *            When the state is invalid
	 */
	void validateState() throws EPPCodecException {
		if (this.name == null) {
			throw new EPPCodecException("name required attribute is not set");
		}
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPDomainRenewResp} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPDomainRenewResp} instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPDomainRenewResp} instance.
	 */
	@Override
	protected Element doEncode(Document aDocument) throws EPPEncodeException {
		// Validate Attributes
		try {
			validateState();
		}
		catch (EPPCodecException e) {
			throw new EPPEncodeException("Invalid state on EPPDomainRenewResp.encode: " + e);
		}

		Element root = aDocument.createElementNS(EPPDomainMapFactory.NS, ELM_NAME);

		// Name
		EPPUtil.encodeString(aDocument, root, this.name, EPPDomainMapFactory.NS,
		      EPPDomainMapFactory.NS_PREFIX + ":" + ELM_DOMAIN_NAME);

		// Expiration Date
		if (this.expirationDate != null) {
			EPPUtil.encodeTimeInstant(aDocument, root, this.expirationDate, EPPDomainMapFactory.NS,
			      EPPDomainMapFactory.NS_PREFIX + ":" + ELM_EXPIRATION_DATE);
		}

		return root;
	}

	/**
	 * Decode the {@code EPPDomainRenewResp} attributes from the {@code aElement}
	 * DOM Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPDomainRenewResp} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode {@code aElement}
	 */
	@Override
	protected void doDecode(Element aElement) throws EPPDecodeException {
		// Name
		this.name = EPPUtil.decodeString(aElement, EPPDomainMapFactory.NS, ELM_DOMAIN_NAME);

		// Expiration Date
		this.expirationDate = EPPUtil.decodeTimeInstant(aElement, EPPDomainMapFactory.NS, ELM_EXPIRATION_DATE);

		// Validate Attributes
		try {
			validateState();
		}
		catch (EPPCodecException e) {
			throw new EPPDecodeException("Invalid state on EPPDomainRenewResp.decode: " + e);
		}
	}

}
