/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.contact;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUpdateCmd;
import com.verisign.epp.codec.gen.EPPUtil;

/**
 * Represents an EPP Host &lt;update&gt; command. The EPP &lt;update&gt; command
 * provides a transform operation that allows a client to modify the attributes
 * of a contact object. In addition to the standard EPP command elements, the
 * &lt;update&gt; command MUST contain a &lt;contact:update&gt; element that
 * identifies the contact namespace and the location of the contact schema. The
 * &lt;contact:update&gt; element SHALL contain the following child elements:
 * <br>
 * <br>
 *
 * <ul>
 * <li>A &lt;contact:id&gt; element that contains the server-unique identifier
 * of the contact object to be updated. Use {@code getId} and {@code setId} to
 * get and set the element.</li>
 * <li>An OPTIONAL &lt;contact:add&gt; element that contains attribute values to
 * be added to the host object. Use {@code getAdd} and {@code setAdd} to get and
 * set the element.</li>
 * <li>An OPTIONAL &lt;contact:rem&gt; element that contains attribute values to
 * be removed from the host object. Use {@code getRemove} and {@code setRemove}
 * to get and set the element.</li>
 * <li>An OPTIONAL &lt;contact:chg&gt; element that contains attribute values to
 * be changed to the host object. Use {@code getChange} and {@code setChange} to
 * get and set the element.</li>
 * </ul>
 *
 * <br>
 * {@code EPPReponse} is the response associated with
 * {@code EPPContactUpdateCmd}. <br>
 * <br>
 *
 * @see com.verisign.epp.codec.gen.EPPResponse
 * @see com.verisign.epp.codec.contact.EPPContactAddChange
 */
public class EPPContactUpdateCmd extends EPPUpdateCmd {
	/** XML Element Name of {@code EPPHostUpdateCmd} root element. */
	final static String ELM_NAME = "contact:update";

	/** XML Element Name for the name attribute. */
	private final static String ELM_CONTACT_ID = "contact:id";

	/** Attributes to add */
	private EPPContactAddChange add = null;

	/** Attributes to remove */
	private EPPContactAddChange remove = null;

	/** Attributes to change */
	private EPPContactAddChange change = null;

	/** Contact ID */
	private java.lang.String id = null;

	/** Category for logging */
	private static Logger cat = LoggerFactory.getLogger(EPPContactUpdateCmd.class);

	/**
	 * {@code EPPContactUpdateCmd} default constructor. The name is initialized
	 * to {@code null}. The name must be set before invoking {@code encode}.
	 */
	public EPPContactUpdateCmd() {
	}


	/**
	 * {@code EPPContactUpdateCmd} constructor.
	 *
	 * @param aTransId
	 *           Transaction Id associated with the command.
	 * @param aId
	 *           Contact ID
	 */
	public EPPContactUpdateCmd(String aTransId, String aId) {
		super(aTransId);

		this.id = aId;
	}


	/**
	 * {@code EPPContactUpdateCmd} constructor that takes the required attributes
	 * as arguments.
	 *
	 * @param aTransId
	 *           Transaction Id associated with the command.
	 * @param aId
	 *           Contact ID.
	 * @param aAdd
	 *           Attributes to add to the contact. {@code null} if no additions.
	 * @param aRemove
	 *           Attributes to remove from the contact. {@code null} if no
	 *           removals.
	 * @param aChange
	 *           Attributes to remove from the contact. {@code null} if no
	 *           changes.
	 */
	public EPPContactUpdateCmd(String aTransId, String aId, EPPContactAddChange aAdd, EPPContactAddChange aRemove,
	      EPPContactAddChange aChange) {
		super(aTransId);

		this.id = aId;

		this.add = aAdd;

		if (this.add != null) {
			this.add.setMode(EPPContactAddChange.MODE_ADD);
		}

		this.remove = aRemove;

		if (this.remove != null) {
			this.remove.setMode(EPPContactAddChange.MODE_REMOVE);
		}

		this.change = aChange;

		if (this.change != null) {
			this.change.setMode(EPPContactAddChange.MODE_CHANGE);
		}
	}

	/**
	 * Gets the EPP command Namespace associated with
	 * {@code EPPContactUpdateCmd}.
	 *
	 * @return {@code EPPHostMapFactory.NS}
	 */
	@Override
	public String getNamespace() {
		return EPPContactMapFactory.NS;
	}

	// End EPPContactUpdateCmd.getNamespace()

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPContactUpdateCmd} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Root DOM Element representing the {@code EPPContactUpdateCmd}
	 *         instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPContactUpdateCmd} instance.
	 */
	protected Element doEncode(Document aDocument) throws EPPEncodeException {
		// Validate state
		if (this.id == null) {
			throw new EPPEncodeException("required attribute id is not set");
		}

		if ((this.add == null) && (this.remove == null) && (this.change == null)) {
			throw new EPPEncodeException("at least one from [add, remove, change] needs to be set");
		}

		Element root = aDocument.createElementNS(EPPContactMapFactory.NS, ELM_NAME);

		root.setAttribute("xmlns:contact", EPPContactMapFactory.NS);

		// id
		EPPUtil.encodeString(aDocument, root, this.id, EPPContactMapFactory.NS, ELM_CONTACT_ID);

		// Add
		EPPUtil.encodeComp(aDocument, root, this.add);

		// Remove
		EPPUtil.encodeComp(aDocument, root, this.remove);

		// Change
		EPPUtil.encodeComp(aDocument, root, this.change);

		return root;
	}


	/**
	 * Decode the {@code EPPContactUpdateCmd} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPContactUpdateCmd} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	protected void doDecode(Element aElement) throws EPPDecodeException {
		// Host Name
		this.id = EPPUtil.decodeString(aElement, EPPContactMapFactory.NS, ELM_CONTACT_ID);

		// Add
		this.add = (EPPContactAddChange) EPPUtil.decodeComp(aElement, EPPContactMapFactory.NS,
		      EPPContactAddChange.ELM_ADD, EPPContactAddChange.class);

		if (this.add != null) {
			this.add.setMode(EPPContactAddChange.MODE_ADD);
		}

		// Remove
		this.remove = (EPPContactAddChange) EPPUtil.decodeComp(aElement, EPPContactMapFactory.NS,
		      EPPContactAddChange.ELM_REMOVE, EPPContactAddChange.class);

		if (this.remove != null) {
			this.remove.setMode(EPPContactAddChange.MODE_REMOVE);
		}

		// Change
		this.change = (EPPContactAddChange) EPPUtil.decodeComp(aElement, EPPContactMapFactory.NS,
		      EPPContactAddChange.ELM_CHANGE, EPPContactAddChange.class);

		if (this.change != null) {
			this.change.setMode(EPPContactAddChange.MODE_CHANGE);
		}
	}

	// End EPPContactUpdateCmd.doDecode(Node)

	/**
	 * Compare an instance of {@code EPPContactUpdateCmd} with this instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return {@code true} if equal; {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		cat.debug("equals(Object): enter");

		if (!(aObject instanceof EPPContactUpdateCmd)) {
			cat.error("equals(Object): aObject not EPPContactUpdateCmd");
			return false;
		}

		if (!super.equals(aObject)) {
			cat.error("equals(Object): parent not equal");
			return false;
		}

		EPPContactUpdateCmd theComp = (EPPContactUpdateCmd) aObject;

		// Id
		if (!((this.id == null) ? (theComp.id == null) : this.id.equals(theComp.id))) {
			cat.error("equals(Object): id not equal");
			return false;
		}

		// Add
		if (!((this.add == null) ? (theComp.add == null) : this.add.equals(theComp.add))) {
			cat.error("equals(Object): add not equal");
			return false;
		}

		// Remove
		if (!((this.remove == null) ? (theComp.remove == null) : this.remove.equals(theComp.remove))) {
			cat.error("equals(Object): remove not equal");
			return false;
		}

		// Change
		if (!((this.change == null) ? (theComp.change == null) : this.change.equals(theComp.change))) {
			cat.error("equals(Object): change not equal");
			return false;
		}

		cat.debug("equals(Object): exit, return true");
		return true;
	}

	// End EPPContactUpdateCmd.equals(Object)

	/**
	 * Clone {@code EPPContactUpdateCmd}.
	 *
	 * @return clone of {@code EPPContactUpdateCmd}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPContactUpdateCmd clone = (EPPContactUpdateCmd) super.clone();

		if (clone.add != null) {
			clone.add = (EPPContactAddChange) this.add.clone();
		}

		if (clone.remove != null) {
			clone.remove = (EPPContactAddChange) this.remove.clone();
		}

		if (clone.change != null) {
			clone.change = (EPPContactAddChange) this.change.clone();
		}

		return clone;
	}

	// End EPPContactUpdateCmd.clone()

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Returns {@code true} if id exists.
	 *
	 * @return {@code true} if id exists; otherwise {@code false}.
	 */
	public boolean hasId() {
		return (this.id != null);
	}
	
	/**
	 * Gets the contact id.
	 *
	 * @return Contact id if set; {@code null} otherwise.
	 */
	public String getId() {
		return this.id;
	}

	/**
	 * Sets the contact id.
	 *
	 * @param aId
	 *           Contact id
	 */
	public void setId(String aId) {
		this.id = aId;
	}

	/**
	 * Returns {@code true} if add exists.
	 *
	 * @return {@code true} if add exists; otherwise {@code false}.
	 */
	public boolean hasAdd() {
		return (this.add != null);
	}

	/**
	 * Gets the items to add to the contact.
	 *
	 * @return Object that contains the list of elements to add to the contact if
	 *         defined; {@code null} otherwise.
	 */
	public EPPContactAddChange getAdd() {
		return this.add;
	}

	/**
	 * Sets the items to add to the contact.
	 *
	 * @param aAdd
	 *          Items to add to the contact.
	 */
	public void setAdd(EPPContactAddChange aAdd) {
		this.add = aAdd;

		if (this.add != null) {
			this.add.setMode(EPPContactAddChange.MODE_ADD);
		}
	}

	/**
	 * Returns {@code true} if change exists.
	 *
	 * @return {@code true} if change exists; otherwise {@code false}.
	 */
	public boolean hasChange() {
		return (this.change != null);
	}

	/**
	 * Gets the items to change to the contact.
	 *
	 * @return Object that contains the list of elements to change to the contact
	 *         if defined; {@code null} otherwise.
	 */
	public EPPContactAddChange getChange() {
		return this.change;
	}

	/**
	 * Sets the items to change in the contact.
	 *
	 * @param aChange
	 *           Items to change in the contact.
	 */
	public void setChange(EPPContactAddChange aChange) {
		this.change = aChange;

		if (this.change != null) {
			this.change.setMode(EPPContactAddChange.MODE_CHANGE);
		}
	}

	/**
	 * Returns {@code true} if remove exists.
	 *
	 * @return {@code true} if remove exists; otherwise {@code false}.
	 */
	public boolean hasRemove() {
		return (this.remove != null);
	}

	/**
	 * Gets the items to remove from the contact.
	 *
	 * @return Items to remove from the contact if defined; {@code null}
	 *         otherwise.
	 */
	public EPPContactAddChange getRemove() {
		return this.remove;
	}

	/**
	 * Sets the items to remove from the contact.
	 *
	 * @param aRemove
	 *           Items to remove from the contact.
	 */
	public void setRemove(EPPContactAddChange aRemove) {
		this.remove = aRemove;

		if (this.remove != null) {
			this.remove.setMode(EPPContactAddChange.MODE_REMOVE);
		}
	}

}
