/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.serverstub;

import java.util.Calendar;
import java.util.Date;
import java.util.Enumeration;
import java.util.GregorianCalendar;
import java.util.Vector;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.verisign.epp.codec.domain.EPPDomainCheckCmd;
import com.verisign.epp.codec.domain.EPPDomainCheckResp;
import com.verisign.epp.codec.domain.EPPDomainCheckResult;
import com.verisign.epp.codec.domain.EPPDomainCreateCmd;
import com.verisign.epp.codec.domain.EPPDomainCreateResp;
import com.verisign.epp.codec.domain.EPPDomainDeleteCmd;
import com.verisign.epp.codec.domain.EPPDomainInfoCmd;
import com.verisign.epp.codec.domain.EPPDomainInfoResp;
import com.verisign.epp.codec.domain.EPPDomainMapFactory;
import com.verisign.epp.codec.domain.EPPDomainPendActionMsg;
import com.verisign.epp.codec.domain.EPPDomainRenewCmd;
import com.verisign.epp.codec.domain.EPPDomainRenewResp;
import com.verisign.epp.codec.domain.EPPDomainStatus;
import com.verisign.epp.codec.domain.EPPDomainTransferCmd;
import com.verisign.epp.codec.domain.EPPDomainTransferResp;
import com.verisign.epp.codec.domain.EPPDomainUpdateCmd;
import com.verisign.epp.codec.gen.EPPAuthInfo;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPCommand;
import com.verisign.epp.codec.gen.EPPFactory;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPResult;
import com.verisign.epp.codec.gen.EPPTransId;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.codec.rgpext.EPPRgpExtInfData;
import com.verisign.epp.codec.rgpext.EPPRgpExtReport;
import com.verisign.epp.codec.rgpext.EPPRgpExtRestore;
import com.verisign.epp.codec.rgpext.EPPRgpExtStatus;
import com.verisign.epp.codec.rgpext.EPPRgpExtUpdate;
import com.verisign.epp.codec.rgppoll.EPPRgpPollMapFactory;
import com.verisign.epp.codec.rgppoll.EPPRgpPollResponse;
import com.verisign.epp.codec.rgppoll.EPPRgpPollStatus;
import com.verisign.epp.framework.EPPDomainHandler;
import com.verisign.epp.framework.EPPEvent;
import com.verisign.epp.framework.EPPEventResponse;
import com.verisign.epp.framework.EPPHandleEventException;
import com.verisign.epp.framework.EPPPollQueueException;
import com.verisign.epp.framework.EPPPollQueueMgr;

/**
 * The <code>RgpDomainHandler</code> class is a concrete implementation of the
 * abstract <code>com.verisign.epp.framework.EPPRgpDomainHandler</code> class.
 * It defines the Server's response to all received EPP Host Commands. <br>
 * <br>
 *
 * @author $Author: jim $
 * @version $Revision: 1.4 $
 *
 * @see com.verisign.epp.framework.EPPEvent
 * @see com.verisign.epp.framework.EPPEventResponse
 */
public class RgpDomainHandler extends EPPDomainHandler {

	private static final String svrTransId = "54322-XYZ";

	private static final String roid = "NS1EXAMPLE1-VRSN";

	private static Logger cat = LoggerFactory.getLogger(RgpDomainHandler.class);

	/**
	 * Constructs an instance of RgpDomainHandler
	 */
	public RgpDomainHandler() {

		super();

		try {
			EPPFactory.getInstance().addMapFactory(EPPRgpPollMapFactory.class.getName());
		}
		catch (EPPCodecException e) {
			cat.error("Couldn't load the Map Factory associated with the EPPRgpPollMapFactory Mapping", e);
			System.exit(1);
		}
	}

	/**
	 * Handles any common behavior that all Domain commands need to execute
	 * before they execute their command specific behavior.
	 *
	 * @param aEvent
	 *           The <code>EPPEvent</code> that is being handled
	 * @param aData
	 *           Any data that a Server needs to send to this
	 *           <code>EPPRgpDomainHandler</code> This is assumed to be an
	 *           instance of SessionData here.
	 *
	 * @exception EPPHandleEventException
	 *               Thrown if an error condition occurs. It must contain an
	 *               <code>EPPEventResponse</code>
	 */
	protected void preHandleEvent(EPPEvent aEvent, Object aData) throws EPPHandleEventException {
		SessionData sessionData = (SessionData) aData;
		EPPCommand theMessage = (EPPCommand) aEvent.getMessage();

		if (!sessionData.isLoggedIn()) {
			/**
			 * The client isn't logged in so they can't successfully invoke a
			 * command. Sending COMMAND_FAILED_END
			 */
			/**
			 * Create the transId for the response with the client trans id and the
			 * server trans id.
			 */
			EPPTransId transId = new EPPTransId(theMessage.getTransId(), svrTransId);

			// Create the Response (Standard EPPResponse)
			EPPResponse theResponse = new EPPResponse(transId);

			theResponse.setResult(EPPResult.COMMAND_FAILED_END);
			throw new EPPHandleEventException("The client has not established a session", theResponse);
		}
	}

	/**
	 * Handles any common behavior that all Domain commands need to execute after
	 * they execute their command specific behavior.
	 *
	 * @param aEvent
	 *           The <code>EPPEvent</code> that is being handled
	 * @param aData
	 *           Any data that a Server needs to send to this
	 *           <code>EPPRgpDomainHandler</code>
	 *
	 * @exception EPPHandleEventException
	 *               Thrown if an error condition occurs. It must contain an
	 *               <code>EPPEventResponse</code>
	 */
	protected void postHandleEvent(EPPEvent aEvent, Object aData) throws EPPHandleEventException {
	}

	/**
	 * Invoked when a Domain Check command is received.
	 *
	 * @param aEvent
	 *           The <code>EPPEvent</code> that is being handled
	 * @param aData
	 *           Any data that a Server needs to send to this
	 *           <code>EPPDomaindHandler</code>
	 *
	 * @return EPPEventResponse The response that should be sent back to the
	 *         client.
	 */
	protected EPPEventResponse doDomainCheck(EPPEvent aEvent, Object aData) {
		EPPDomainCheckCmd theMessage = (EPPDomainCheckCmd) aEvent.getMessage();
		EPPResponse theResponse;

		// This is just a vector of strings representing Domain Names
		Vector vNames = theMessage.getNames();
		Enumeration eNames = vNames.elements();

		Vector vResults = new Vector();

		// create a Vector of Ping Results.
		boolean known = true;

		while (eNames.hasMoreElements()) {
			String domainName = (String) eNames.nextElement();
			known = !known;
			vResults.addElement(new EPPDomainCheckResult(domainName, known));
		}

		/**
		 * Create the transId for the response with the client trans id and the
		 * server trans id.
		 */
		EPPTransId transId = new EPPTransId(theMessage.getTransId(), svrTransId);

		// Response for many domain names
		theResponse = new EPPDomainCheckResp(transId, vResults);
		theResponse.setResult(EPPResult.SUCCESS);

		return new EPPEventResponse(theResponse);
	}

	/**
	 * Invoked when a Domain Renew command is received.
	 *
	 * @param aEvent
	 *           The <code>EPPEvent</code> that is being handled
	 * @param aData
	 *           Any data that a Server needs to send to this
	 *           <code>EPPDomaindHandler</code>
	 *
	 * @return EPPEventResponse The response that should be sent back to the
	 *         client.
	 */
	protected EPPEventResponse doDomainRenew(EPPEvent aEvent, Object aData) {
		EPPDomainRenewCmd theMessage = (EPPDomainRenewCmd) aEvent.getMessage();
		EPPResponse theResponse;

		/**
		 * Create the transId for the response with the client trans id and the
		 * server trans id.
		 */
		EPPTransId transId = new EPPTransId(theMessage.getTransId(), svrTransId);

		// Test with transId, domain name, roid, and expiration date
		theResponse = new EPPDomainRenewResp(transId, theMessage.getName(), new Date());

		theResponse.setResult(EPPResult.SUCCESS);

		return new EPPEventResponse(theResponse);
	}

	/**
	 * Invoked when a Domain Delete command is received.
	 *
	 * @param aEvent
	 *           The <code>EPPEvent</code> that is being handled
	 * @param aData
	 *           Any data that a Server needs to send to this
	 *           <code>EPPDomaindHandler</code>
	 *
	 * @return EPPEventResponse The response that should be sent back to the
	 *         client.
	 */
	protected EPPEventResponse doDomainDelete(EPPEvent aEvent, Object aData) {
		EPPDomainDeleteCmd theMessage = (EPPDomainDeleteCmd) aEvent.getMessage();

		// Test with transId, domain name, and expiration date

		/**
		 * Create the transId for the response with the client trans id and the
		 * server trans id.
		 */
		EPPTransId transId = new EPPTransId(theMessage.getTransId(), svrTransId);

		EPPResponse theResponse = new EPPDomainCreateResp(transId, theMessage.getName(), new Date());
		theResponse.setResult(EPPResult.SUCCESS);

		return new EPPEventResponse(theResponse);
	}

	/**
	 * Invoked when a Domain Create command is received.
	 *
	 * @param aEvent
	 *           The <code>EPPEvent</code> that is being handled
	 * @param aData
	 *           Any data that a Server needs to send to this
	 *           <code>EPPDomaindHandler</code>
	 *
	 * @return EPPEventResponse The response that should be sent back to the
	 *         client.
	 */
	protected EPPEventResponse doDomainCreate(EPPEvent aEvent, Object aData) {
		EPPDomainCreateCmd theMessage = (EPPDomainCreateCmd) aEvent.getMessage();

		/**
		 * Create the transId for the response with the client trans id and the
		 * server trans id.
		 */
		EPPTransId transId = new EPPTransId(theMessage.getTransId(), svrTransId);
		EPPResponse theResponse = new EPPDomainCreateResp(transId, theMessage.getName(), new Date());

		if (theMessage.getName().equals("pending.com")) {
			EPPTransId pendingTransId = new EPPTransId(theMessage.getTransId(), svrTransId);
			EPPDomainPendActionMsg thePollMsg = new EPPDomainPendActionMsg(transId, "pending.com", true, pendingTransId,
			      new Date());

			try {
				EPPPollQueueMgr.getInstance().put(null, EPPDomainMapFactory.NS, thePollMsg, null);
			}
			catch (EPPPollQueueException ex) {
				cat.error("doDomainCreate: Error putting message [" + thePollMsg + "]");

				EPPResult theResult = new EPPResult(EPPResult.COMMAND_FAILED);
				theResponse = new EPPResponse(transId, theResult);

				return new EPPEventResponse(theResponse);
			}

			theResponse.setResult(EPPResult.SUCCESS_PENDING);
		}
		else {
			theResponse.setResult(EPPResult.SUCCESS);
		}

		return new EPPEventResponse(theResponse);
	}

	/**
	 * Invoked when a Domain Transfer command is received.
	 *
	 * @param aEvent
	 *           The <code>EPPEvent</code> that is being handled
	 * @param aData
	 *           Any data that a Server needs to send to this
	 *           <code>EPPDomaindHandler</code>
	 *
	 * @return EPPEventResponse The response that should be sent back to the
	 *         client.
	 */
	protected EPPEventResponse doDomainTransfer(EPPEvent aEvent, Object aData) {
		EPPDomainTransferCmd theMessage = (EPPDomainTransferCmd) aEvent.getMessage();

		/**
		 * Create the transId for the response with the client trans id and the
		 * server trans id.
		 */
		EPPTransId transId = new EPPTransId(theMessage.getTransId(), svrTransId);

		// Is a transfer request?
		if (theMessage.getOp().equals(EPPCommand.OP_REQUEST)) {
			// Insert transfer response in poll queue
			EPPDomainTransferResp thePollMsg = new EPPDomainTransferResp();

			thePollMsg.setName("example");
			thePollMsg.setTransferStatus(EPPResponse.TRANSFER_PENDING);
			thePollMsg.setRequestClient("ClientX");
			thePollMsg.setRequestDate(new Date());
			thePollMsg.setActionClient("ClientY");
			thePollMsg.setActionDate(new Date());
			thePollMsg.setExpirationDate(new Date());
			thePollMsg.setResult(EPPResult.SUCCESS);

			try {
				EPPPollQueueMgr.getInstance().put(null, EPPDomainMapFactory.NS, thePollMsg, null);
			}
			catch (EPPPollQueueException ex) {
				cat.error("doDomainTransfer: Error putting message [" + thePollMsg + "]");

				EPPResult theResult = new EPPResult(EPPResult.COMMAND_FAILED);
				EPPResponse theResponse = new EPPResponse(transId, theResult);

				return new EPPEventResponse(theResponse);
			}
		}

		EPPDomainTransferResp theResponse = new EPPDomainTransferResp(transId, theMessage.getName());
		theResponse.setTransferStatus(EPPResponse.TRANSFER_PENDING);

		theResponse.setRequestClient("ClientX");
		theResponse.setRequestDate(new Date());

		theResponse.setActionClient("ClientY");
		theResponse.setActionDate(new Date());

		/**
		 * The Expiration date is optional
		 */
		theResponse.setExpirationDate(new Date());

		theResponse.setResult(EPPResult.SUCCESS);

		return new EPPEventResponse(theResponse);
	}

	/**
	 * Invoked when a Domain Update command is received.
	 *
	 * @param aEvent
	 *           The <code>EPPEvent</code> that is being handled
	 * @param aData
	 *           Any data that a Server needs to send to this
	 *           <code>EPPDomaindHandler</code>
	 *
	 * @return EPPEventResponse The response that should be sent back to the
	 *         client.
	 */
	protected EPPEventResponse doDomainUpdate(EPPEvent aEvent, Object aData) {

		cat.debug("RgpDomainHandler.doDomainUpdate(): enter");

		EPPDomainUpdateCmd theMessage = (EPPDomainUpdateCmd) aEvent.getMessage();

		EPPResponse theResponse = null;

		// Get the RGP update extension
		EPPRgpExtUpdate rgpUpdate = (EPPRgpExtUpdate) theMessage.getExtension(EPPRgpExtUpdate.class);

		// Create Update Response (Standard EPPResponse)
		EPPTransId pendingTransId = new EPPTransId(theMessage.getTransId(), svrTransId);

		// create poll response if there's an RGP update extension
		if (rgpUpdate != null) {

			if (rgpUpdate.getRestore().getOp().equals(EPPRgpExtRestore.REPORT)) {
				cat.info("RgpDomainHandler.doDomainUpdate(): restore report");

				// Output the attributes of the report.
				EPPRgpExtReport theReport = rgpUpdate.getRestore().getReport();

				cat.info("RgpDomainHandler.doDomainUpdate(): preData = " + theReport.getPreData());
				cat.info("RgpDomainHandler.doDomainUpdate(): postData = " + theReport.getPostData());
				cat.info("RgpDomainHandler.doDomainUpdate(): deleteTime = " + theReport.getDeleteTime());
				cat.info("RgpDomainHandler.doDomainUpdate(): restoreTime = " + theReport.getRestoreTime());
				cat.info("RgpDomainHandler.doDomainUpdate(): restoreReason = "
				      + (theReport.getRestoreReason() != null ? theReport.getRestoreReason().getMessage() : "null"));
				cat.info("RgpDomainHandler.doDomainUpdate(): statement1 = "
				      + (theReport.getStatement1() != null ? theReport.getStatement1().getMessage() : "null"));
				cat.info("RgpDomainHandler.doDomainUpdate(): statement2 = "
				      + (theReport.getStatement2() != null ? theReport.getStatement2().getMessage() : "null"));
			}
			else { // Restore Request
				cat.info("RgpDomainHandler.doDomainUpdate(): restore request");
				EPPRgpPollResponse thePollMsg = new EPPRgpPollResponse();

				thePollMsg.setName(theMessage.getName());
				thePollMsg.setReportDueDate(new Date(System.currentTimeMillis() + 3600));
				thePollMsg.setTransId(pendingTransId);
				thePollMsg.setReqDate(new Date(System.currentTimeMillis()));
				thePollMsg.setStatus(new EPPRgpPollStatus(EPPRgpPollStatus.PENDING_RESTORE));

				try {
					EPPPollQueueMgr.getInstance().put(null, EPPRgpPollMapFactory.NS, thePollMsg, null);
				}
				catch (EPPPollQueueException ex) {
					cat.error("doDomainUpdate: Error putting message [" + thePollMsg + "]");

					EPPResult theResult = new EPPResult(EPPResult.COMMAND_FAILED);
					theResponse = new EPPResponse(pendingTransId, theResult);

					return new EPPEventResponse(theResponse);
				}

			}
		}
		else {
			cat.error("doDomainUpdate: RGP extension NOT found");

			EPPResult theResult = new EPPResult(EPPResult.PARAM_VALUE_POLICY_ERROR);
			theResponse = new EPPResponse(pendingTransId, theResult);

			return new EPPEventResponse(theResponse);
		}

		/**
		 * Create the transId for the response with the client trans id and the
		 * server trans id.
		 */
		EPPTransId transId = new EPPTransId(theMessage.getTransId(), svrTransId);

		theResponse = new EPPResponse(transId);
		theResponse.setResult(EPPResult.SUCCESS);

		cat.debug("RgpDomainHandler.doDomainUpdate(): enter");
		return new EPPEventResponse(theResponse);
	}

	/**
	 * Invoked when a Domain Info command is received.
	 * 
	 * @param aEvent
	 *           The <code>EPPEvent</code> that is being handled
	 * @param aData
	 *           Any data that a Server needs to send to this
	 *           <code>EPPDomaindHandler</code>
	 * 
	 * @return EPPEventResponse The response that should be sent back to the
	 *         client.
	 */
	protected EPPEventResponse doDomainInfo(EPPEvent aEvent, Object aData) {
		EPPDomainInfoCmd theMessage = (EPPDomainInfoCmd) aEvent.getMessage();

		// EPPDomainInfo Response requires a vector of status
		Vector statuses = new Vector();
		statuses.addElement(new EPPDomainStatus(EPPDomainStatus.ELM_STATUS_OK));
		statuses.addElement(new EPPDomainStatus(EPPDomainStatus.ELM_STATUS_CLIENT_HOLD));

		/**
		 * Create the transId for the response with the client trans id and the
		 * server trans id.
		 */
		EPPTransId transId = new EPPTransId(theMessage.getTransId(), svrTransId);

		// Required EPPDomainInfoResp attributes.
		// trans id, domain name, roid, client id, statuses, created by id,
		// expiration date, Auth Info
		EPPDomainInfoResp theResponse = new EPPDomainInfoResp(transId, roid, "example.com", "ClientX", statuses,
		      "ClientY", new Date(), new EPPAuthInfo("2fooBAR"));

		// Add Name Servers
		Vector theNses = new Vector();
		theNses.addElement("ns1.example.com");
		theNses.addElement("ns2.example.com");
		theResponse.setNses(theNses);

		// Add Subordinate Hosts
		Vector theHosts = new Vector();
		theHosts.addElement("ns1.example.com");
		theHosts.addElement("ns2.example.com");
		theResponse.setHosts(theHosts);

		// Set the expiration date to today plus one year
		GregorianCalendar theCal = new GregorianCalendar();
		theCal.setTime(new Date());
		theCal.add(Calendar.YEAR, 1);

		theResponse.setExpirationDate(theCal.getTime());

		theResponse.setResult(EPPResult.SUCCESS);

		// Set the grace period statuses?
		if (theMessage.getName().equalsIgnoreCase("graceperiod.com")) {
			EPPRgpExtInfData rgpExt = new EPPRgpExtInfData();

			// Auto Renew Period
			EPPRgpExtStatus autoRenewPeriodStatus = new EPPRgpExtStatus(EPPRgpExtStatus.AUTO_RENEW_PERIOD);
			autoRenewPeriodStatus.setMessage("endDate=" + EPPUtil.encodeTimeInstant(new Date()));
			rgpExt.addStatus(autoRenewPeriodStatus);

			// Renew Period
			EPPRgpExtStatus renewPeriodStatus = new EPPRgpExtStatus(EPPRgpExtStatus.RENEW_PERIOD);
			renewPeriodStatus.setMessage("endDate=" + EPPUtil.encodeTimeInstant(new Date()));
			rgpExt.addStatus(renewPeriodStatus);

			theResponse.addExtension(rgpExt);
		}
		else { // Set a pending period status (redemptionPeriod)
			// Set EPP status to pendingDelete
			statuses = new Vector();
			statuses.addElement(new EPPDomainStatus(EPPDomainStatus.ELM_STATUS_PENDING_DELETE));
			theResponse.setStatuses(statuses);

			EPPRgpExtInfData rgpExt = new EPPRgpExtInfData();

			EPPRgpExtStatus rgpStatus = new EPPRgpExtStatus();
			rgpStatus.setStatus(EPPRgpExtStatus.REDEMPTION_PERIOD);
			rgpExt.addStatus(rgpStatus);

			theResponse.addExtension(rgpExt);
		}

		return new EPPEventResponse(theResponse);
	}
}
