/***********************************************************
Copyright (C) 2013 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.relateddomainext;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;


/**
 * EPPCodecComponent that encodes and decodes a &lt;relDom:registered&gt; tag
 * <p>
 * Title: EPP 1.0 Related Domain - registered tag
 * </p>
 * <p>
 * Description: The EPPRelatedDomainExtRegistered object represents the
 * collection of domains that are registered in a family of related domains. As
 * such it is composed of a collection of {@link EPPRelatedDomainExtName}
 * objects. <br>
 * As XML, it is represented by a &lt;relDom:registered&gt; element containing a
 * number of &lt;relDom:name&gt; elements.
 * </p>
 */
public class EPPRelatedDomainExtRegistered implements EPPCodecComponent {

	/**
	 * Serial version id - increment this if the structure changes.
	 */
	private static final long serialVersionUID = 1L;

	/**
	 * Category for logging
	 */
	    private static Logger cat = LoggerFactory.getLogger(EPPRelatedDomainExtRegistered.class);
	      

	/**
	 * Element tag name for registered
	 */
	public static final String ELM_NAME = EPPRelatedDomainExtFactory.NS_PREFIX + ":registered";

	/**
	 * List of domain names represented by the {@link EPPRelatedDomainExtName} to
	 * be associated with the &lt;relDom:registered&gt; element
	 */
	private List<EPPRelatedDomainExtName> registeredDomains = null;

	/**
	 * Default constructor
	 */
	public EPPRelatedDomainExtRegistered() {
	}

	/**
	 * Constructor with a list of registered domain names.
	 * 
	 * @param aRegisteredDomains
	 *           List of registered domain names
	 */
	public EPPRelatedDomainExtRegistered(final List<EPPRelatedDomainExtName> aRegisteredDomains) {
		this.registeredDomains = aRegisteredDomains;

	}

	/**
	 * A deep clone of the EPPRelatedDomainExtRegistered
	 * 
	 * @see java.lang.Object#clone()
	 */
	public Object clone() throws CloneNotSupportedException {
		final EPPRelatedDomainExtRegistered theClone = new EPPRelatedDomainExtRegistered();

		if (this.registeredDomains != null) {
			theClone.registeredDomains = new ArrayList<EPPRelatedDomainExtName>();

			for (EPPRelatedDomainExtName registeredDomain : this.registeredDomains) {

				if (registeredDomain != null) {
					theClone.registeredDomains.add((EPPRelatedDomainExtName) registeredDomain.clone());
				}
				else {
					theClone.registeredDomains.add(null);
				}

			}

		}
		else {
			theClone.registeredDomains = null;
		}

		return theClone;
	}

	/**
	 * Populate the data of this instance with the data stored in the given
	 * Element of the DOM tree
	 * 
	 * @param aElement
	 *           The root element of the report fragment of XML
	 * @throws EPPDecodeException
	 *            Thrown if any errors occur during decoding.
	 */
	public void decode(final Element aElement) throws EPPDecodeException {

		this.registeredDomains = EPPUtil.decodeCompList(aElement, EPPRelatedDomainExtFactory.NS,
		      EPPRelatedDomainExtName.ELM_DOMAIN_NAME, EPPRelatedDomainExtName.class);
	}

	/**
	 * Append all data from the list of the list of registered related domain
	 * names represented by {@link EPPRelatedDomainExtName} to given DOM Document
	 * 
	 * @param aDocument
	 *           The DOM Document to append data to
	 * @return Encoded DOM {@code Element}
	 * @throws EPPEncodeException
	 *            Thrown when errors occur during the encode attempt or if the
	 *            instance is invalid.
	 */
	public Element encode(final Document aDocument) throws EPPEncodeException {

		if (aDocument == null) {
			throw new EPPEncodeException("aDocument is null" + " in EPPRelatedDomainExtRegistered.encode(Document)");
		}

		try {
			// Validate States
			validateState();
		}
		catch (final EPPCodecException e) {
			cat.error("EPPRelatedDomainExtRegistered.encode(): Invalid state on encode: " + e);
			throw new EPPEncodeException("EPPRelatedDomainExtRegistered invalid state: " + e);
		}

		final Element root = aDocument.createElementNS(EPPRelatedDomainExtFactory.NS, ELM_NAME);

		// dsData
		EPPUtil.encodeCompList(aDocument, root, this.registeredDomains);
		return root;
	}

	/**
	 * A deep comparison of this with another EPPRelatedDomainExtRegistered.
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(final Object aObj) {

		if (!(aObj instanceof EPPRelatedDomainExtRegistered)) {
			return false;
		}

		final EPPRelatedDomainExtRegistered theComp = (EPPRelatedDomainExtRegistered) aObj;

		if (!EPPUtil.equalLists(this.registeredDomains, theComp.registeredDomains)) {
			cat.error("EPPRelatedDomainExtRegistered.equals(): registeredDomains not equal");
			return false;
		}

		return true;
	}

	/**
	 * Adds a registered domain represented by {@link EPPRelatedDomainExtName} to
	 * the list of registered domains.
	 * 
	 * @param aRegisteredDomain
	 *           Registered domain to add to the list.
	 */
	public void addRegisteredDomain(EPPRelatedDomainExtName aRegisteredDomain) {
		if (this.registeredDomains == null) {
			this.registeredDomains = new ArrayList<EPPRelatedDomainExtName>();
		}

		this.registeredDomains.add(aRegisteredDomain);
	}

	/**
	 * Are there any registered domains included in the list of registered
	 * domains?
	 * 
	 * @return {@code true} if the registered domain list is not {@code null} and
	 *         there is at least one registered domain in the list; {@code false}
	 *         otherwise.
	 */
	public boolean hasRegisteredDomains() {
		if (this.registeredDomains != null && this.registeredDomains.size() > 0) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Returns the registeredDomains
	 * 
	 * @return the registeredDomains
	 */
	public List<EPPRelatedDomainExtName> getRegisteredDomains() {
		return this.registeredDomains;
	}

	/**
	 * Sets registeredDomains value to aRegisteredDomains
	 * 
	 * @param aRegisteredDomains
	 *           the registered domains to set
	 */
	public void setRegisteredDomains(final List<EPPRelatedDomainExtName> aRegisteredDomains) {
		this.registeredDomains = aRegisteredDomains;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 * 
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 * 
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPRelatedDomainExtFactory.NS;
	}

	/**
	 * Validate the state of the {@code EPPRelatedDomainExtRegistered} instance.
	 * A valid state means that all of the required attributes have been set. If
	 * validateState returns without an exception, the state is valid. If the
	 * state is not valid, the {@code EPPCodecException} will contain a
	 * description of the error. throws EPPCodecException State error. This will
	 * contain the name of the attribute that is not valid.
	 * 
	 * @throws EPPCodecException
	 *            Thrown if the instance is in an invalid state
	 */
	private void validateState() throws EPPCodecException {
		if ((this.registeredDomains == null) || (this.registeredDomains.size() == 0)) {
			throw new EPPCodecException("EPPRelatedDomainExtRegistered contains no elements.");
		}

	}

}
