/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPRegistryBatchJob} contains the information for an individual batch
 * job.
 */
public class EPPRegistryBatchJob implements EPPCodecComponent {
	/**
	 * Logger
	 */
	    private static Logger cat = LoggerFactory.getLogger(EPPRegistryBatchJob.class);
	      

	/**
	 * XML local name for {@code EPPRegistryBatchJob}.
	 */
	public static final String ELM_LOCALNAME = "batchJob";

	/**
	 * XML root tag for {@code EPPRegistryBatchJob}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML local name for the {@code name} attribute.
	 */
	final static String ELM_JOB_NAME = "name";

	/**
	 * XML local name for the {@code description} attribute.
	 */
	final static String ELM_DESCRIPTION = "description";

	/**
	 * Name of the batch job, like "autoRenew" or "pendingDelete".
	 */
	private String name = null;

	/**
	 * OPTIONAL free-form description of batch job, like "Auto Renew Batch" or
	 * "Pending Delete Batch".
	 */
	private String description = null;

	/**
	 * One or more &lt;registry:batchJob&gt; elements containing the batch job
	 * information.
	 */
	private List<EPPRegistryBatchSchedule> schedules = new ArrayList<EPPRegistryBatchSchedule>();

	/**
	 * Default constructor for {@code EPPRegistryBatchJob}.
	 */
	public EPPRegistryBatchJob() {
	}

	/**
	 * {@code EPPRegistryBatchJob} constructor that the name and the optional
	 * description. At least one schedule must be set. individual schedule.
	 *
	 * @param aName
	 *           Name of the batch job, like "autoRenew" or "pendingDelete".
	 * @param aDescription
	 *           Description of the batch job
	 */
	public EPPRegistryBatchJob(String aName, String aDescription) {
		this.name = aName;
		this.description = aDescription;
	}

	/**
	 * {@code EPPRegistryBatchJob} constructor that the required name and an
	 * individual schedule.
	 *
	 * @param aName
	 *           Name of the batch job, like "autoRenew" or "pendingDelete".
	 * @param aSchedule
	 *           An individual batch schedule using XML schema "time" type format
	 */
	public EPPRegistryBatchJob(String aName, EPPRegistryBatchSchedule aSchedule) {
		this.name = aName;
		this.addSchedule(aSchedule);
	}

	/**
	 * {@code EPPRegistryBatchJob} constructor that the required name, an
	 * individual schedule, along with an optional description.
	 *
	 * @param aName
	 *           Name of the batch job, like "autoRenew" or "pendingDelete".
	 * @param aSchedule
	 *           An individual batch schedule using the XML schema "time" type
	 *           format
	 * @param aDescription
	 *           Description of the batch job
	 */
	public EPPRegistryBatchJob(String aName, EPPRegistryBatchSchedule aSchedule, String aDescription) {
		this.name = aName;
		this.addSchedule(aSchedule);
		this.description = aDescription;
	}

	/**
	 * {@code EPPRegistryBatchJob} constructor that the required name, a list of
	 * schedules, along with an optional description.
	 *
	 * @param aName
	 *           Name of the batch job, like "autoRenew" or "pendingDelete".
	 * @param aSchedules
	 *           A list of schedules with each schedule using the XML schema
	 *           "time" type format
	 * @param aDescription
	 *           Description of the batch job
	 */
	public EPPRegistryBatchJob(String aName, List<EPPRegistryBatchSchedule> aSchedules, String aDescription) {
		this.name = aName;
		this.setSchedules(aSchedules);
		this.description = aDescription;
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryBatchJob} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPRegistryBatchJob} instance.
	 *
	 * @exception EPPEncodeException
	 *               - Unable to encode {@code EPPRegistryBatchJob} instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		// Validate state
		if (this.name == null) {
			throw new EPPEncodeException("EPPRegistryBatchJob name is null");
		}
		if (!this.hasSchedules()) {
			throw new EPPEncodeException("EPPRegistryBatchJob schedules is empty");
		}

		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

		// Name
		EPPUtil.encodeString(aDocument, root, this.name, EPPRegistryMapFactory.NS,
		      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_JOB_NAME);

		// Description
		if (this.description != null) {
			EPPUtil.encodeString(aDocument, root, this.description, EPPRegistryMapFactory.NS,
			      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_DESCRIPTION);
		}

		// Schedules
		EPPUtil.encodeCompList(aDocument, root, this.schedules);

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryBatchJob} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryBatchJob} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {
		// Name
		this.name = EPPUtil.decodeString(aElement, EPPRegistryMapFactory.NS, ELM_JOB_NAME);

		// Description
		this.description = EPPUtil.decodeString(aElement, EPPRegistryMapFactory.NS, ELM_DESCRIPTION);

		// Schedules
		this.schedules = EPPUtil.decodeCompList(aElement, EPPRegistryMapFactory.NS, EPPRegistryBatchSchedule.ELM_NAME,
		      EPPRegistryBatchSchedule.class);
	}

	/**
	 * Clone {@code EPPRegistryBatchJob}.
	 *
	 * @return clone of {@code EPPRegistryBatchJob}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}

	/**
	 * implements a deep {@code EPPRegistryBatchJob} compare.
	 *
	 * @param aObject
	 *           {@code EPPRegistryBatchJob} instance to compare with
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryBatchJob)) {
			return false;
		}

		EPPRegistryBatchJob theComp = (EPPRegistryBatchJob) aObject;

		// Name
		if (!EqualityUtil.equals(this.name, theComp.name)) {
			cat.error("EPPRegistryBatchJob.equals(): name not equal");
			return false;
		}

		// Description
		if (!EqualityUtil.equals(this.description, theComp.description)) {
			cat.error("EPPRegistryBatchJob.equals(): description not equal");
			return false;
		}

		// Schedules
		if (!EqualityUtil.equals(this.schedules, theComp.schedules)) {
			cat.error("EPPRegistryBatchJob.equals(): schedules not equal");
			return false;
		}

		return true;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Is the name defined?
	 *
	 * @return {@code true} if the name is defined; {@code false} otherwise.
	 */
	public boolean hasName() {
		return (this.name != null ? true : false);
	}

	/**
	 * Gets the name of the batch job, like "autoRenew" or "pendingDelete".
	 *
	 * @return Name of the batch job if defined; {@code null} otherwise.
	 */
	public String getName() {
		return this.name;
	}

	/**
	 * Sets the name of the batch job, like "autoRenew" or "pendingDelete".
	 *
	 * @param aName
	 *           Name of the batch job
	 */
	public void setName(String aName) {
		this.name = aName;
	}

	/**
	 * Is the description defined?
	 *
	 * @return {@code true} if the description is defined; {@code false}
	 *         otherwise.
	 */
	public boolean hasDescription() {
		return (this.description != null ? true : false);
	}

	/**
	 * Gets the OPTIONAL free-form description of batch job, like
	 * "Auto Renew Batch" or "Pending Delete Batch".
	 *
	 * @return batch job description if defined; {@code null} otherwise.
	 */
	public String getDescription() {
		return this.description;
	}

	/**
	 * Sets the OPTIONAL free-form description of batch job, like
	 * "Auto Renew Batch" or "Pending Delete Batch".
	 *
	 * @param aDescription
	 *           the description to set
	 */
	public void setDescription(String aDescription) {
		this.description = aDescription;
	}

	/**
	 * Is the schedules defined?
	 *
	 * @return {@code true} if the schedules is defined; {@code false} otherwise.
	 */
	public boolean hasSchedules() {
		return (this.schedules != null && !this.schedules.isEmpty() ? true : false);
	}

	/**
	 * Gets the {@code List} of {@link EPPRegistryBatchSchedule} instances
	 * containing the batch job schedules.
	 *
	 * @return {@code List} of {@link EPPRegistryBatchSchedule} instances if
	 *         defined; {@code null} otherwise.
	 */
	public List<EPPRegistryBatchSchedule> getSchedules() {
		return this.schedules;
	}

	/**
	 * Sets the {@code List} of {@link EPPRegistryBatchSchedule} instances
	 * containing the batch job schedules.
	 *
	 * @param aSchedules
	 *           {@code List} of {@link EPPRegistryBatchSchedule} instances
	 */
	public void setSchedules(List<EPPRegistryBatchSchedule> aSchedules) {
		if (aSchedules == null) {
			this.schedules = new ArrayList<EPPRegistryBatchSchedule>();
		}
		else {
			this.schedules = aSchedules;
		}
	}

	/**
	 * Adds a schedule to the list of batch job schedules.
	 * 
	 * @param aSchedule
	 *           Schedule to add to the list of batch job schedules
	 */
	public void addSchedule(EPPRegistryBatchSchedule aSchedule) {
		if (aSchedule == null) {
			return;
		}

		if (this.schedules == null) {
			this.schedules = new ArrayList<EPPRegistryBatchSchedule>();
		}

		this.schedules.add(aSchedule);
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

}
