/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.org;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUpdateCmd;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * Command used to update a org object.
 */
public class EPPOrgUpdateCmd extends EPPUpdateCmd {

	/**
	 * Category for logging
	 */
	    private static Logger cat = LoggerFactory.getLogger(EPPOrgUpdateCmd.class);
	      

	/**
	 * XML local name for <code>EPPOrgUpdateCmd</code>.
	 */
	public static final String ELM_LOCALNAME = "update";

	/**
	 * XML root tag for <code>EPPOrgUpdateCmd</code>.
	 */
	public static final String ELM_NAME = EPPOrgMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML local name for the add element
	 */
	private static final String ELM_ADD = "add";

	/**
	 * XML local name for the status (<code>status</code>) element.
	 */
	private final static String ELM_STATUS = "status";

	/**
	 * XML local name for the rem element
	 */
	private static final String ELM_REM = "rem";

	/**
	 * XML local name for the chg element
	 */
	private static final String ELM_CHG = "chg";

	/**
	 * XML local name for the org identifier element
	 */
	private static final String ELM_ORG_ID = "id";

	/**
	 * XML local name for the OPTIONAL parent identifier (<code>parentId</code>)
	 * element.
	 */
	private final static String ELM_PARENT_ID = "parentId";

	/**
	 * XML local name for the OPTIONAL voice number (<code>voice</code>) element.
	 */
	private final static String ELM_VOICE = "voice";

	/**
	 * XML local name for the OPTIONAL fax number (<code>fax</code>) element.
	 */
	private final static String ELM_FAX = "fax";

	/**
	 * XML local name for the email (<code>email</code>) element.
	 */
	private final static String ELM_EMAIL = "email";

	/**
	 * XML local name for the OPTIONAL URL (<code>url</code>) element.
	 */
	private final static String ELM_URL = "url";

	/**
	 * XML extension "x" attribute value for voice or fax elements.
	 */
	private final static String ATTR_EXT = "x";

	/**
	 * The org identifier of the org of a sponsoring registrar.
	 */
	private String orgId;

	/**
	 * Parent Identifier
	 */
	private String parentId;

	/**
	 * One or two postal information elements, represented by the
	 * {@link EPPOrgPostalDefinition} class.
	 */
	private List<EPPOrgPostalDefinition> postalInfo = new ArrayList<EPPOrgPostalDefinition>();

	/**
	 * Voice number
	 */
	private String voice;

	/**
	 * Voice Extension
	 */
	private String voiceExt;

	/**
	 * FAX number
	 */
	private String fax;

	/**
	 * FAX Extension
	 */
	private String faxExt;

	/**
	 * Email Address
	 */
	private String email;

	/**
	 * URL
	 */
	private String url;

	/**
	 * Add Org contacts
	 */
	private List<EPPOrgContact> addContacts = new ArrayList<EPPOrgContact>();

	/**
	 * Remove Org contacts
	 */
	private List<EPPOrgContact> remContacts = new ArrayList<EPPOrgContact>();

	/**
	 * Add Org roles
	 */
	private List<EPPOrgRole> addRoles = new ArrayList<EPPOrgRole>();

	/**
	 * Remove Org roles
	 */
	private List<EPPOrgRole> remRoles = new ArrayList<EPPOrgRole>();

	/**
	 * Add Org statuses
	 */
	private List<EPPOrgStatus> addStatuses = new ArrayList<EPPOrgStatus>();

	/**
	 * Remove Org statuses
	 */
	private List<EPPOrgStatus> remStatuses = new ArrayList<EPPOrgStatus>();

	/**
	 * <code>EPPOrgUpdateCmd</code> default constructor with the default form of
	 * <code>Form.LIST_FORM</code>.
	 */
	public EPPOrgUpdateCmd() {
	}

	/**
	 * <code>EPPOrgUpdateCmd</code> constructor that takes the client transaction
	 * identifier.
	 * 
	 * @param aTransId
	 *           Client transaction identifier.
	 */
	public EPPOrgUpdateCmd(String aTransId) {
		super(aTransId);
	}

	/**
	 * Gets the org identifier.
	 * 
	 * @return The org identifier if defined;<code>null</code> otherwise.
	 */
	public String getOrgId() {
		return this.orgId;
	}

	/**
	 * Sets the org identifier.
	 * 
	 * @param aOrgId
	 *           The org identifier
	 */
	public void setOrgId(String aOrgId) {
		this.orgId = aOrgId;
	}

	/**
	 * Has the parent identifier been set?
	 * 
	 * @return <code>true</code> if the parent identifier has been set;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasParentId() {
		return this.parentId != null ? true : false;
	}

	/**
	 * Gets the parent org identifier for a hierarchy of orgs.
	 * 
	 * @return Parent identifier if set; <code>null</code> otherwise.
	 */
	public String getParentId() {
		return this.parentId;
	}

	/**
	 * @param aParentId
	 *           the parentId to set
	 */
	public void setParentId(String aParentId) {
		this.parentId = aParentId;
	}

	/**
	 * Is there any postal information set?
	 * 
	 * @return <code>true</code> if there is at least one
	 *         {@link EPPOrgPostalDefinition} set in the postal information;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasPostalInfo() {
		if (this.postalInfo != null && !this.postalInfo.isEmpty()) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Adds a postal definition to the postal information for the org.
	 * 
	 * @param aPostalInfo
	 *           Postal definition to add to the postal information.
	 */
	public void addPostalInfo(EPPOrgPostalDefinition aPostalInfo) {
		this.postalInfo.add(aPostalInfo);
	}

	/**
	 * Gets the postal information for the org. There can be one or two
	 * {@link EPPOrgPostalDefinition} objects in the postal information list.
	 * 
	 * @return Postal information for the org
	 */
	public List<EPPOrgPostalDefinition> getPostalInfo() {
		return this.postalInfo;
	}

	/**
	 * Sets the postal information for the org. There can be one or two
	 * {@link EPPOrgPostalDefinition} objects in the postal information list.
	 * 
	 * @param aPostalInfo
	 *           Postal information for the org.
	 */
	public void setPostalInfo(List<EPPOrgPostalDefinition> aPostalInfo) {
		this.postalInfo = aPostalInfo;
	}

	/**
	 * Has the voice been set?
	 * 
	 * @return <code>true</code> if the voice has been set; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasVoice() {
		return this.voice != null ? true : false;
	}

	/**
	 * @return the voice
	 */
	public String getVoice() {
		return this.voice;
	}

	/**
	 * @param aVoice
	 *           the voice to set
	 */
	public void setVoice(String aVoice) {
		this.voice = aVoice;
	}

	/**
	 * Has the voice extension been set?
	 * 
	 * @return <code>true</code> if the voice extension has been set;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasVoiceExt() {
		return this.voiceExt != null ? true : false;
	}

	/**
	 * @return the voiceExt
	 */
	public String getVoiceExt() {
		return this.voiceExt;
	}

	/**
	 * @param aVoiceExt
	 *           the voiceExt to set
	 */
	public void setVoiceExt(String aVoiceExt) {
		this.voiceExt = aVoiceExt;
	}

	/**
	 * Has the fax been set?
	 * 
	 * @return <code>true</code> if the fax has been set; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasFax() {
		return this.fax != null ? true : false;
	}

	/**
	 * @return the fax
	 */
	public String getFax() {
		return this.fax;
	}

	/**
	 * @param aFax
	 *           the fax to set
	 */
	public void setFax(String aFax) {
		this.fax = aFax;
	}

	/**
	 * Has the fax extension been set?
	 * 
	 * @return <code>true</code> if the fax extension has been set;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasFaxExt() {
		return this.faxExt != null ? true : false;
	}

	/**
	 * @return the faxExt
	 */
	public String getFaxExt() {
		return this.faxExt;
	}

	/**
	 * @param aFaxExt
	 *           the faxExt to set
	 */
	public void setFaxExt(String aFaxExt) {
		this.faxExt = aFaxExt;
	}

	/**
	 * Has the email been set?
	 * 
	 * @return <code>true</code> if the email has been set; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasEmail() {
		return this.email != null ? true : false;
	}

	/**
	 * @return the email
	 */
	public String getEmail() {
		return this.email;
	}

	/**
	 * @param aEmail
	 *           the email to set
	 */
	public void setEmail(String aEmail) {
		this.email = aEmail;
	}

	/**
	 * Has the url been set?
	 * 
	 * @return <code>true</code> if the url has been set; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasUrl() {
		return this.url != null ? true : false;
	}

	/**
	 * Gets the URL of the website of the org.
	 * 
	 * @return URL of the website of the org.
	 */
	public String getUrl() {
		return this.url;
	}

	/**
	 * Sets the URL of the website of the org.
	 * 
	 * @param aUrl
	 *           URL of the website of the org.
	 */
	public void setUrl(String aUrl) {
		this.url = aUrl;
	}

	/**
	 * Is there any add contacts set?
	 * 
	 * @return <code>true</code> if there is at least one add
	 *         {@link EPPOrgContact} set; <code>false</code> otherwise.
	 */
	public boolean hasAddContacts() {
		if (this.addContacts != null && !this.addContacts.isEmpty()) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Adds an add contact to the list of add contacts.
	 * 
	 * @param aContact
	 *           Contact to add to the list of add contacts.
	 */
	public void addAddContact(EPPOrgContact aContact) {
		this.addContacts.add(aContact);
	}

	/**
	 * Gets the add contacts.
	 * 
	 * @return List of add contacts set.
	 */
	public List<EPPOrgContact> getAddContacts() {
		return this.addContacts;
	}

	/**
	 * Sets the add contacts.
	 * 
	 * @param aContacts
	 *           List of add contacts
	 */
	public void setAddContacts(List<EPPOrgContact> aContacts) {
		this.addContacts = aContacts;
	}

	/**
	 * Is there any remove contacts set?
	 * 
	 * @return <code>true</code> if there is at least one remove
	 *         {@link EPPOrgContact} set; <code>false</code> otherwise.
	 */
	public boolean hasRemContacts() {
		if (this.remContacts != null && !this.remContacts.isEmpty()) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Adds a remove contact to the list of remove contacts.
	 * 
	 * @param aContact
	 *           Contact to add to the list of remove contacts.
	 */
	public void addRemContact(EPPOrgContact aContact) {
		this.remContacts.add(aContact);
	}

	/**
	 * Gets the remove contacts.
	 * 
	 * @return List of remove contacts set.
	 */
	public List<EPPOrgContact> getRemContacts() {
		return this.remContacts;
	}

	/**
	 * Sets the remove contacts.
	 * 
	 * @param aContacts
	 *           List of remove contacts
	 */
	public void setRemContacts(List<EPPOrgContact> aContacts) {
		this.remContacts = aContacts;
	}

	/**
	 * Is there any add roles set?
	 * 
	 * @return <code>true</code> if there is at least one add {@link EPPOrgRole}
	 *         set; <code>false</code> otherwise.
	 */
	public boolean hasAddRoles() {
		if (this.addRoles != null && !this.addRoles.isEmpty()) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Adds an add role to the list of add roles.
	 * 
	 * @param aRole
	 *           Role to add to the list of add roles.
	 */
	public void addAddRole(EPPOrgRole aRole) {
		this.addRoles.add(aRole);
	}

	/**
	 * Gets the add roles.
	 * 
	 * @return List of add roles set.
	 */
	public List<EPPOrgRole> getAddRoles() {
		return this.addRoles;
	}

	/**
	 * Sets the add roles.
	 * 
	 * @param aRoles
	 *           List of add roles
	 */
	public void setAddRoles(List<EPPOrgRole> aRoles) {
		this.addRoles = aRoles;
	}

	/**
	 * Is there any remove roles set?
	 * 
	 * @return <code>true</code> if there is at least one remove
	 *         {@link EPPOrgRole} set; <code>false</code> otherwise.
	 */
	public boolean hasRemRoles() {
		if (this.remRoles != null && !this.remRoles.isEmpty()) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Adds a remove role to the list of remove roles.
	 * 
	 * @param aRole
	 *           Role to add to the list of remove roles.
	 */
	public void addRemRole(EPPOrgRole aRole) {
		this.remRoles.add(aRole);
	}

	/**
	 * Gets the remove roles.
	 * 
	 * @return List of remove roles set.
	 */
	public List<EPPOrgRole> getRemRoles() {
		return this.remRoles;
	}

	/**
	 * Sets the remove roles.
	 * 
	 * @param aRoles
	 *           List of remove roles
	 */
	public void setRemRoles(List<EPPOrgRole> aRoles) {
		this.remRoles = aRoles;
	}

	/**
	 * Is there any add statuses set?
	 * 
	 * @return <code>true</code> if there is at least one add
	 *         {@link EPPOrgStatus} set; <code>false</code> otherwise.
	 */
	public boolean hasAddStatuses() {
		if (this.addStatuses != null && !this.addStatuses.isEmpty()) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Adds an add status to the list of add statuses.
	 * 
	 * @param aStatus
	 *           Status to add to the list of add statuses.
	 */
	public void addAddStatus(EPPOrgStatus aStatus) {
		this.addStatuses.add(aStatus);
	}

	/**
	 * Gets the add statuses.
	 * 
	 * @return List of add statuses set.
	 */
	public List<EPPOrgStatus> getAddStatuses() {
		return this.addStatuses;
	}

	/**
	 * Sets the add statuses.
	 * 
	 * @param aStatuses
	 *           List of add statuses
	 */
	public void setAddStatuses(List<EPPOrgStatus> aStatuses) {
		this.addStatuses = aStatuses;
	}

	/**
	 * Is there any remove statuses set?
	 * 
	 * @return <code>true</code> if there is at least one remove
	 *         {@link EPPOrgStatus} set; <code>false</code> otherwise.
	 */
	public boolean hasRemStatuses() {
		if (this.remStatuses != null && !this.remStatuses.isEmpty()) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Adds a remove status to the list of remove statuses.
	 * 
	 * @param aStatus
	 *           Status to add to the list of remove statuses.
	 */
	public void addRemStatus(EPPOrgStatus aStatus) {
		this.remStatuses.add(aStatus);
	}

	/**
	 * Gets the remove statuses.
	 * 
	 * @return List of remove statuses set.
	 */
	public List<EPPOrgStatus> getRemStatuses() {
		return this.remStatuses;
	}

	/**
	 * Sets the remove statuses.
	 * 
	 * @param aStatuses
	 *           List of remove statuses
	 */
	public void setRemStatuses(List<EPPOrgStatus> aStatuses) {
		this.remStatuses = aStatuses;
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * <code>EPPOrgUpdateCmd</code> instance.
	 * 
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 * 
	 * @return Element Root DOM Element representing the EPPOrgUpdateCmd
	 *         instance.
	 * 
	 * @exception EPPEncodeException
	 *               Unable to encode EPPOrgUpdateCmd instance.
	 */
	protected Element doEncode(Document aDocument) throws EPPEncodeException {

		Element theElm = null;
		Text theVal = null;

		// Check required attributes
		if (this.orgId == null) {
			throw new EPPEncodeException("Undefined orgId in EPPOrgUpdateCmd");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPOrgMapFactory.NS, ELM_NAME);

		// Org Identifier
		EPPUtil.encodeString(aDocument, root, this.orgId, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_ORG_ID);

		// Add elements
		if (this.hasAddContacts() || this.hasAddRoles() || this.hasAddStatuses()) {
			Element addElm = aDocument.createElementNS(EPPOrgMapFactory.NS, EPPOrgMapFactory.NS_PREFIX + ":" + ELM_ADD);
			EPPUtil.encodeCompList(aDocument, addElm, this.addContacts);
			EPPUtil.encodeCompList(aDocument, addElm, this.addRoles);
			EPPUtil.encodeList(aDocument, addElm, this.addStatuses, EPPOrgMapFactory.NS,
			      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_STATUS);
			root.appendChild(addElm);
		}

		// Remove elements
		if (this.hasRemContacts() || this.hasRemRoles() || this.hasRemStatuses()) {
			Element remElm = aDocument.createElementNS(EPPOrgMapFactory.NS, EPPOrgMapFactory.NS_PREFIX + ":" + ELM_REM);
			EPPUtil.encodeCompList(aDocument, remElm, this.remContacts);
			EPPUtil.encodeCompList(aDocument, remElm, this.remRoles);
			EPPUtil.encodeList(aDocument, remElm, this.remStatuses, EPPOrgMapFactory.NS,
			      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_STATUS);
			root.appendChild(remElm);
		}

		// Change elements
		if (this.hasParentId() || this.hasPostalInfo() || this.hasVoice() || this.hasFax() || this.hasEmail()
		      || this.hasUrl()) {
			Element chgElm = aDocument.createElementNS(EPPOrgMapFactory.NS, EPPOrgMapFactory.NS_PREFIX + ":" + ELM_CHG);

			// Parent Id
			EPPUtil.encodeString(aDocument, chgElm, this.parentId, EPPOrgMapFactory.NS,
			      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_PARENT_ID);

			// Postal Information
			EPPUtil.encodeCompList(aDocument, chgElm, this.postalInfo);

			// Voice and Voice Extension
			if (this.voice != null) {
				theElm = aDocument.createElementNS(EPPOrgMapFactory.NS, EPPOrgMapFactory.NS_PREFIX + ":" + ELM_VOICE);
				theVal = aDocument.createTextNode(this.voice);
				theElm.appendChild(theVal);

				if (this.voiceExt != null) {
					theElm.setAttribute(ATTR_EXT, this.voiceExt);
				}

				chgElm.appendChild(theElm);
			}

			// Fax and Fax Extension
			if (this.fax != null) {
				theElm = aDocument.createElementNS(EPPOrgMapFactory.NS, EPPOrgMapFactory.NS_PREFIX + ":" + ELM_FAX);
				theVal = aDocument.createTextNode(this.fax);
				theElm.appendChild(theVal);

				if (this.faxExt != null) {
					theElm.setAttribute(ATTR_EXT, this.faxExt);
				}

				chgElm.appendChild(theElm);
			}

			// Email
			EPPUtil.encodeString(aDocument, chgElm, this.email, EPPOrgMapFactory.NS,
			      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_EMAIL);

			// URL
			EPPUtil.encodeString(aDocument, chgElm, this.url, EPPOrgMapFactory.NS,
			      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_URL);

			root.appendChild(chgElm);
		}

		return root;
	}

	/**
	 * Decode the <code>EPPOrgUpdateCmd</code> attributes from the aElement DOM
	 * Element tree.
	 * 
	 * @param aElement
	 *           Root DOM Element to decode <code>EPPOrgUpdateCmd</code> from.
	 * 
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	protected void doDecode(Element aElement) throws EPPDecodeException {

		Element theElm = null;

		// Org Identifier
		this.orgId = EPPUtil.decodeString(aElement, EPPOrgMapFactory.NS, ELM_ORG_ID);

		// Add element
		Element theAddElm = EPPUtil.getElementByTagNameNS(aElement, EPPOrgMapFactory.NS, ELM_ADD);
		if (theAddElm != null) {
			// Add Contacts
			this.addContacts = EPPUtil.decodeCompList(theAddElm, EPPOrgMapFactory.NS, EPPOrgContact.ELM_NAME,
			      EPPOrgContact.class);

			// Add Roles
			this.addRoles = EPPUtil.decodeCompList(theAddElm, EPPOrgMapFactory.NS, EPPOrgRole.ELM_NAME, EPPOrgRole.class);

			// Add Statuses
			this.addStatuses = EPPUtil.decodeEnumList(theAddElm, EPPOrgMapFactory.NS, ELM_STATUS, EPPOrgStatus.class);
		}

		// Remove element
		Element theRemElm = EPPUtil.getElementByTagNameNS(aElement, EPPOrgMapFactory.NS, ELM_REM);
		if (theRemElm != null) {
			// Remove Contacts
			this.remContacts = EPPUtil.decodeCompList(theRemElm, EPPOrgMapFactory.NS, EPPOrgContact.ELM_NAME,
			      EPPOrgContact.class);

			// Remove Roles
			this.remRoles = EPPUtil.decodeCompList(theRemElm, EPPOrgMapFactory.NS, EPPOrgRole.ELM_NAME, EPPOrgRole.class);

			// Remove Statuses
			this.remStatuses = EPPUtil.decodeEnumList(theRemElm, EPPOrgMapFactory.NS, ELM_STATUS, EPPOrgStatus.class);
		}

		// Change element
		Element theChgElm = EPPUtil.getElementByTagNameNS(aElement, EPPOrgMapFactory.NS, ELM_CHG);
		if (theChgElm != null) {

			// Parent Id
			this.parentId = EPPUtil.decodeString(theChgElm, EPPOrgMapFactory.NS, ELM_PARENT_ID);

			// Postal Information
			this.postalInfo = EPPUtil.decodeCompList(theChgElm, EPPOrgMapFactory.NS, EPPOrgPostalDefinition.ELM_NAME,
			      EPPOrgPostalDefinition.class);

			// Voice and Voice Extension
			theElm = EPPUtil.getElementByTagNameNS(theChgElm, EPPOrgMapFactory.NS, ELM_VOICE);
			if (theElm != null) {
				this.voice = EPPUtil.getTextContent(theElm);
				this.voiceExt = EPPUtil.decodeStringAttr(theElm, ATTR_EXT);
			}
			else {
				this.voice = null;
				this.voiceExt = null;
			}

			// Fax and Fax Extension
			theElm = EPPUtil.getElementByTagNameNS(theChgElm, EPPOrgMapFactory.NS, ELM_FAX);
			if (theElm != null) {
				this.fax = EPPUtil.getTextContent(theElm);
				this.faxExt = EPPUtil.decodeStringAttr(theElm, ATTR_EXT);
			}
			else {
				this.fax = null;
				this.faxExt = null;
			}

			// Email
			this.email = EPPUtil.decodeString(theChgElm, EPPOrgMapFactory.NS, ELM_EMAIL);

			// URL
			this.url = EPPUtil.decodeString(theChgElm, EPPOrgMapFactory.NS, ELM_URL);
		}

	}

	/**
	 * Clone <code>EPPOrgUpdateCmd</code>.
	 * 
	 * @return clone of <code>EPPOrgUpdateCmd</code>
	 * 
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPOrgUpdateCmd clone = (EPPOrgUpdateCmd) super.clone();

		if (this.hasPostalInfo()) {
			clone.postalInfo = (List) ((ArrayList) this.postalInfo).clone();
		}

		if (this.hasAddContacts()) {
			clone.addContacts = (List) ((ArrayList) this.addContacts).clone();
		}

		if (this.hasRemContacts()) {
			clone.remContacts = (List) ((ArrayList) this.remContacts).clone();
		}

		if (this.hasAddRoles()) {
			clone.addRoles = (List) ((ArrayList) this.addRoles).clone();
		}

		if (this.hasRemRoles()) {
			clone.remRoles = (List) ((ArrayList) this.remRoles).clone();
		}

		if (this.hasAddStatuses()) {
			clone.addStatuses = (List) ((ArrayList) this.addStatuses).clone();
		}

		if (this.hasRemStatuses()) {
			clone.remStatuses = (List) ((ArrayList) this.remStatuses).clone();
		}

		return clone;
	}

	/**
	 * Gets the EPP response type associated with <code>EPPOrgUpdateCmd</code>.
	 * 
	 * @return <code>EPPOrgUpdateCmd.ELM_NAME</code>
	 */
	public String getType() {
		return ELM_NAME;
	}

	/**
	 * Gets the EPP command namespace associated with
	 * <code>EPPOrgUpdateCmd</code>.
	 * 
	 * @return <code>EPPOrgMapFactory.NS</code>
	 */
	public String getNamespace() {
		return EPPOrgMapFactory.NS;
	}

	/**
	 * Compare an instance of <code>EPPOrgUpdateCmd</code> with this instance.
	 * 
	 * @param aObject
	 *           Object to compare with.
	 * 
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPOrgUpdateCmd)) {
			return false;
		}

		EPPOrgUpdateCmd other = (EPPOrgUpdateCmd) aObject;

		// Org Identifier
		if (!EqualityUtil.equals(this.orgId, other.orgId)) {
			cat.error("EPPOrgUpdateCmd.equals(): orgId not equal");
			return false;
		}

		// Parent Id
		if (!EqualityUtil.equals(this.parentId, other.parentId)) {
			cat.error("EPPOrgUpdateCmd.equals(): parentId not equal");
			return false;
		}

		// Postal Information
		if (!EPPUtil.equalLists(this.postalInfo, other.postalInfo)) {
			cat.error("EPPOrgUpdateCmd.equals(): postalInfo not equal");
			return false;
		}

		// Voice
		if (!EqualityUtil.equals(this.voice, other.voice)) {
			cat.error("EPPOrgUpdateCmd.equals(): voice not equal");
			return false;
		}

		// Voice Ext
		if (!EqualityUtil.equals(this.voiceExt, other.voiceExt)) {
			cat.error("EPPOrgUpdateCmd.equals(): voiceExt not equal");
			return false;
		}

		// Fax
		if (!EqualityUtil.equals(this.fax, other.fax)) {
			cat.error("EPPOrgUpdateCmd.equals(): fax not equal");
			return false;
		}

		// Fax Ext
		if (!EqualityUtil.equals(this.faxExt, other.faxExt)) {
			cat.error("EPPOrgUpdateCmd.equals(): faxExt not equal, this.faxExt = " + this.faxExt + ", other.faxExt = "
			      + other.faxExt);
			return false;
		}

		// Email
		if (!EqualityUtil.equals(this.email, other.email)) {
			cat.error("EPPOrgUpdateCmd.equals(): email not equal");
			return false;
		}

		// Url
		if (!EqualityUtil.equals(this.url, other.url)) {
			cat.error("EPPOrgUpdateCmd.equals(): url not equal");
			return false;
		}

		// Add Contacts
		if (!EPPUtil.equalLists(this.addContacts, other.addContacts)) {
			cat.error("EPPOrgUpdateCmd.equals(): addContacts not equal");
			return false;
		}

		// Remove Contacts
		if (!EPPUtil.equalLists(this.remContacts, other.remContacts)) {
			cat.error("EPPOrgUpdateCmd.equals(): remContacts not equal");
			return false;
		}

		// Add Roles
		if (!EPPUtil.equalLists(this.addRoles, other.addRoles)) {
			cat.error("EPPOrgUpdateCmd.equals(): addRoles not equal");
			return false;
		}

		// Remove Roles
		if (!EPPUtil.equalLists(this.remRoles, other.remRoles)) {
			cat.error("EPPOrgUpdateCmd.equals(): remRoles not equal");
			return false;
		}

		// Add Statuses
		if (!EPPUtil.equalLists(this.addStatuses, other.addStatuses)) {
			cat.error("EPPOrgUpdateCmd.equals(): addStatuses not equal");
			return false;
		}

		// Remove Statuses
		if (!EPPUtil.equalLists(this.remStatuses, other.remStatuses)) {
			cat.error("EPPOrgUpdateCmd.equals(): remStatuses not equal");
			return false;
		}

		return true;
	}

	/**
	 * Implementation of <code>Object.toString</code>, which will result in an
	 * indented XML <code>String</code> representation of the concrete
	 * <code>EPPCodecComponent</code>.
	 * 
	 * @return Indented XML <code>String</code> if successful; <code>ERROR</code>
	 *         otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

}
