/***********************************************************
Copyright (C) 2017 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.util;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPMessage;

/**
 * Interface for logging messages sent and received. A class can implement this
 * interface and configured to override the information logged. For example, a
 * secure logger can be created to mask certain attributes (.e.g, password, auth
 * info) from getting logged.
 */
public interface EPPSendReceiveLogger {

	/**
	 * Log the sending of a message. The message itself is immutable, but the
	 * message can be cloned to customize the attributes logged.
	 * 
	 * @param aPacket
	 *           Packet to send. This may be <code>null</code> if the packet has
	 *           not been encoded yet.
	 * @param aMessage
	 *           The message that is being sent, which could be any concrete
	 *           <code>EPPMessage</code>, including a
	 *           {@link com.verisign.epp.codec.gen.EPPCommand}, an
	 *           {@link com.verisign.epp.codec.gen.EPPGreeting}, an
	 *           {@link com.verisign.epp.codec.gen.EPPHello}, or an
	 *           {@link com.verisign.epp.codec.gen.EPPResponse}. This may be
	 *           <code>null</code> if the <code>EPPMessage</code> is not
	 *           available.
	 */
	void logSend(byte[] aPacket, final EPPMessage aMessage);

	/**
	 * Log the receiving of a message. The message itself is immutable, but the
	 * message can be cloned to customize the attributes logged.
	 * 
	 * @param aPacket
	 *           Packet received. This may be <code>null</code> if the packet is
	 *           not available.
	 * @param aMessage
	 *           The message received, which could be any concrete
	 *           <code>EPPMessage</code>, including a
	 *           {@link com.verisign.epp.codec.gen.EPPCommand}, an
	 *           {@link com.verisign.epp.codec.gen.EPPGreeting}, an
	 *           {@link com.verisign.epp.codec.gen.EPPHello}, or an
	 *           {@link com.verisign.epp.codec.gen.EPPResponse}. This may be
	 *           <code>null</code> if the <code>EPPMessage</code> is not
	 *           available.
	 */
	void logReceive(byte[] aPacket, final EPPMessage aMessage);

	/**
	 * Utility method used to optionally mask specific message attributes based
	 * on the concrete <code>EPPSendReceiveLogger</code>.
	 * 
	 * @param aMessage
	 *           Message to optionally mask. If <code>null</code>,
	 *           <code>null</code> will be returned.
	 * @return Masked message if instance is different from <code>aMessage</code>
	 *         . <code>aMessage</code> is returned if no filtering was done.
	 */
	EPPCodecComponent maskMessage(final EPPCodecComponent aMessage);

	/**
	 * Utility method used to optionally mask specific strings values 
	 * using the concrete <code>EPPSendReceiverLogger</code>.
	 * 
	 * @param aString <code>String</code> to scan for masking
	 * @return Masked <code>String</code>.
	 */
	String maskString(String aString);
}
