/***********************************************************
Copyright (C) 2017 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.? See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA? 02111-1307? USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.fee.v1_0;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * The extension to a response to a check command with the fee check extension.
 * 
 * @see com.verisign.epp.codec.fee.v1_0.EPPFeeCheck
 * @see com.verisign.epp.codec.fee.v1_0.EPPFeeCheckResult
 */
public class EPPFeeChkData implements EPPCodecComponent {

	/**
	 * Category for logging
	 */
	    private static Logger cat = LoggerFactory.getLogger(EPPFeeChkData.class);
	      

	/**
	 * XML local name for <code>EPPFeeChkData</code>.
	 */
	public static final String ELM_LOCALNAME = "chkData";

	/**
	 * XML root tag for <code>EPPFeeChkData</code>.
	 */
	public static final String ELM_NAME = EPPFeeExtFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML local name for the currency
	 */
	private static final String ELM_CURRENCY = "currency";

	/**
	 * Currency of fees.
	 */
	private String currency;

	/**
	 * List of claims check results
	 */
	private List<EPPFeeCheckResult> results = new ArrayList<EPPFeeCheckResult>();

	/**
	 * <code>EPPFeeChkData</code> default constructor. The results list will be
	 * empty.
	 */
	public EPPFeeChkData() {
	}

	/**
	 * <code>EPPFeeChkData</code> constructor that only sets the currency. At
	 * least one result must be set.
	 * 
	 * @param aCurrency
	 *           Currency code of fees using a three-character currency code from
	 *           ISO4217.
	 */
	public EPPFeeChkData(String aCurrency) {
		this.currency = aCurrency;
	}

	/**
	 * <code>EPPFeeChkData</code> constructor that sets a single result.
	 * 
	 * @param aCurrency
	 *           Currency code of fees using a three-character currency code from
	 *           ISO4217.
	 * 
	 * @param aResult
	 *           A single result
	 */
	public EPPFeeChkData(String aCurrency, EPPFeeCheckResult aResult) {
		this.currency = aCurrency;
		this.addCheckResult(aResult);
	}

	/**
	 * <code>EPPFeeChkData</code> constructor that sets a list of results.
	 * 
	 * @param aCurrency
	 *           Currency code of fees using a three-character currency code from
	 *           ISO4217.
	 * 
	 * @param aResults
	 *           List of check results
	 */
	public EPPFeeChkData(String aCurrency, List<EPPFeeCheckResult> aResults) {
		this.currency = aCurrency;
		this.setCheckResults(aResults);
	}

	/**
	 * Get the EPP command Namespace associated with <code>EPPFeeChkData</code>.
	 * 
	 * @return <code>EPPFeeExtFactory.NS</code>
	 */
	public String getNamespace() {
		return EPPFeeExtFactory.NS;
	}

	
	/**
	 * Gets the currency value.
	 * 
	 * @return Currency if defined; <code>null</code> otherwise.
	 */
	public String getCurrency() {
		return this.currency;
	}

	/**
	 * Sets the currency value.
	 * 
	 * @param aCurrency
	 *           Currency value
	 */
	public void setCurrency(String aCurrency) {
		this.currency = aCurrency;
	}
	

	/**
	 * Get the results of a <code>EPPFeeChkData</code> Response. There is one
	 * <code>EPPFeeDomainResult</code> instance in <code>this.results</code> for
	 * each domain requested in the Check Command.
	 * 
	 * @return List of results if defined; empty list otherwise.
	 */
	public List<EPPFeeCheckResult> getCheckResults() {
		return this.results;
	}

	/**
	 * Set the results of a <code>EPPFeeChkData</code> Response. There is one
	 * <code>EPPFeeDomainResult</code> instance in <code>this.results</code> for
	 * each domain requested in the Check Command.
	 * 
	 * @param aResults
	 *           List of claims check results
	 */
	public void setCheckResults(List<EPPFeeCheckResult> aResults) {
		if (aResults == null) {
			this.results = new ArrayList<EPPFeeCheckResult>();
		}
		else {
			this.results = aResults;
		}
	}

	/**
	 * Adds a check result to the list of results.
	 * 
	 * @param aResult
	 *           Check result to add to the list.
	 */
	public void addCheckResult(EPPFeeCheckResult aResult) {
		this.results.add(aResult);
	}

	/**
	 * Sets the result of an individual domain name.
	 * 
	 * @param aResult
	 *           Check result to set
	 */
	public void setCheckResult(EPPFeeCheckResult aResult) {
		this.results = new ArrayList<EPPFeeCheckResult>();
		this.results.add(aResult);
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * <code>EPPFeeChkData</code> instance.
	 * 
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 * 
	 * @return Element Root DOM Element representing the
	 *         <code>EPPFeeChkData</code> instance.
	 * 
	 * @exception EPPEncodeException
	 *               Unable to encode <code>EPPFeeChkData</code> instance.
	 */
	public Element encode(Document aDocument) throws EPPEncodeException {

		if (aDocument == null) {
			throw new EPPEncodeException("aDocument is null" + " in EPPFeeChkData.encode(Document)");
		}
		
		if (this.currency == null) {
			throw new EPPEncodeException("currency is required" + " in EPPFeeChkData.encode(Document)");
		}
		
		if (this.results == null || this.results.isEmpty()) {
			throw new EPPEncodeException("At least one result is required" + " in EPPFeeChkData.encode(Document)");
		}
		

		Element root = aDocument.createElementNS(EPPFeeExtFactory.NS, ELM_NAME);

		// Currency
		EPPUtil.encodeString(aDocument, root, this.currency, EPPFeeExtFactory.NS,
		      EPPFeeExtFactory.NS_PREFIX + ":" + ELM_CURRENCY);
		 
		// Results
		EPPUtil.encodeCompList(aDocument, root, this.results);

		return root;
	}

	/**
	 * Decode the <code>EPPFeeChkData</code> attributes from the aElement DOM
	 * Element tree.
	 * 
	 * @param aElement
	 *           Root DOM Element to decode <code>EPPFeeChkData</code> from.
	 * 
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	public void decode(Element aElement) throws EPPDecodeException {
		
		// Currency
		this.currency = EPPUtil.decodeString(aElement, EPPFeeExtFactory.NS, ELM_CURRENCY);		

		// Results
		this.results = EPPUtil.decodeCompList(aElement, EPPFeeExtFactory.NS, EPPFeeCheckResult.ELM_NAME,
		      EPPFeeCheckResult.class);
	}

	/**
	 * Implementation of <code>Object.toString</code>, which will result in an
	 * indented XML <code>String</code> representation of the concrete
	 * <code>EPPCodecComponent</code>.
	 * 
	 * @return Indented XML <code>String</code> if successful; <code>ERROR</code>
	 *         otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}
	
	/**
	 * Implements a deep <code>EPPFeeChkData</code> compare.
	 * 
	 * @param aObject
	 *           <code>EPPFeeChkData</code> instance to compare with
	 * 
	 * @return <code>true</code> if equal <code>false</code> otherwise.
	 */
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPFeeChkData)) {
			cat.error("EPPFeeChkData.equals(): aObject is not an EPPFeeChkData");
			return false;
		}

		EPPFeeChkData other = (EPPFeeChkData) aObject;

		// Currency
		if (!EqualityUtil.equals(this.currency, other.currency)) {
			cat.error("EPPFeeChkData.equals(): currency not equal");
			return false;
		}
		
		// Results
		if (!EqualityUtil.equals(this.results, other.results)) {
			cat.error("EPPFeeChkData.equals(): results not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone <code>EPPFeeChkData</code> instance.
	 * 
	 * @return clone of <code>EPPFeeChkData</code>
	 * 
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPFeeChkData clone = (EPPFeeChkData) super.clone();
		
		// Currency
		clone.currency = this.currency;
		
		// Results
		clone.results = new ArrayList<EPPFeeCheckResult>();
		for (EPPFeeCheckResult result : this.results) {
			clone.results.add((EPPFeeCheckResult) result.clone());
		}

		return clone;
	}
	
}
