/***********************************************************
Copyright (C) 2020 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.interfaces;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import com.verisign.epp.codec.gen.EPPGreeting;
import com.verisign.epp.codec.gen.EPPLoginCmd;
import com.verisign.epp.codec.gen.EPPService;

/**
 * Interface that can be implemented and configured to be called by
 * {@link EPPSession} to customize the {@link EPPLoginCmd} prior to sending it.
 * Both the server's {@link EPPGreeting} and the original {@link EPPLoginCmd} is
 * provided. The adapter can modify the {@link EPPLoginCmd} that will then be
 * sent. This adapter is used to customize the login services that is passed by
 * the client to trigger application of the unhandled namespaces defined in
 * draft-ietf-regext-unhandled-namespaces.
 */
public class EPPUnhandledNamespacesLoginAdapter implements EPPLoginAdapter {

	/**
	 * Namespace URIs to filter from login services.
	 */
	List<String> filterNamespaces = new ArrayList<>();

	/**
	 * Default constructor. Filters are set with
	 * {@link #setFilterNamespaces(List)} and
	 * {@link #addFilterNamespace(String)}.
	 */
	public EPPUnhandledNamespacesLoginAdapter() {
	}

	/**
	 * {@code EPPUnhandledNamespacesLoginAdapter} constructor that takes a single
	 * Namespace URI to filter from the login services.
	 *
	 * @param aFilterNamespace
	 *           Namespace URI to filter from the login services.
	 */
	public EPPUnhandledNamespacesLoginAdapter(String aFilterNamespace) {
		this.addFilterNamespace(aFilterNamespace);
	}

	/**
	 * Sets the Namespace URIs to filter from the login services.
	 *
	 * @param aFilterNamespaces
	 *           Namespace URIs to filter from the login services.
	 */
	public void setFilterNamespaces(List<String> aFilterNamespaces) {
		if (aFilterNamespaces == null) {
			this.filterNamespaces = new ArrayList<>();
		}
		else {
			this.filterNamespaces = aFilterNamespaces;
		}
	}

	/**
	 * Add a Namespace URI to the list of Namespace URIs to filter from the login
	 * services.
	 *
	 * @param aNamespace
	 *           Namespace URI to add to the Namespace URIs list
	 */
	public void addFilterNamespace(String aNamespace) {
		this.filterNamespaces.add(aNamespace);
	}

	/**
	 * Adapt the {@link EPPLoginCmd} <code>aLoginCmd</code> parameter based on
	 * the current state of the {@link EPPLoginCmd} and the contents of the
	 * {@link EPPGreeting} <code>aGreeting</code> parameter sent by the server.
	 * This method will filter the Namespace URIs from the login services set in
	 * the {@code aLoginCmd} parameter.
	 *
	 * @param aLoginCmd
	 *           Original {@link EPPLoginCmd} created by {@link EPPSession} that
	 *           will have the login services filtered by the Namespace URI list.
	 * @param aGreeting
	 *           {@link EPPGreeting} returned by the server that can be used to
	 *           drive the adapting of the {@link EPPLoginCmd}.
	 *
	 * @exception EPPCommandException
	 *               Error found with adapting the login
	 */
	@Override
	public void adaptLogin(EPPLoginCmd aLoginCmd, final EPPGreeting aGreeting) throws EPPCommandException {

		Vector<EPPService> theServices = aLoginCmd.getServices();
		Vector<EPPService> theExtServices = aLoginCmd.getExtensionServices();

		for (String filterNamespace : this.filterNamespaces) {

			// Filter from object services
			if (theServices != null) {
				Iterator<EPPService> theServicesIterator = theServices.iterator();
				while (theServicesIterator.hasNext()) {
					EPPService theService = theServicesIterator.next();
					if (theService.getNamespaceURI().equals(filterNamespace)) {
						theServicesIterator.remove();
					}
				}
				aLoginCmd.setServices(theServices);
			}

			// Filter from extension services
			if (theExtServices != null) {
				Iterator<EPPService> theExtServicesIterator = theExtServices.iterator();
				while (theExtServicesIterator.hasNext()) {
					EPPService theExtService = theExtServicesIterator.next();
					if (theExtService.getNamespaceURI().equals(filterNamespace)) {
						theExtServicesIterator.remove();
					}
				}
				aLoginCmd.setExtensionServices(theExtServices);
			}

		}

	}

}
