/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.serverstub;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.verisign.epp.codec.domain.EPPDomainCreateCmd;
import com.verisign.epp.codec.domain.EPPDomainInfoCmd;
import com.verisign.epp.codec.domain.EPPDomainUpdateCmd;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.orgext.EPPOrgExtCreate;
import com.verisign.epp.codec.orgext.EPPOrgExtFactory;
import com.verisign.epp.codec.orgext.EPPOrgExtId;
import com.verisign.epp.codec.orgext.EPPOrgExtInfData;
import com.verisign.epp.codec.orgext.EPPOrgExtUpdate;
import com.verisign.epp.framework.EPPEvent;
import com.verisign.epp.framework.EPPEventResponse;
import com.verisign.epp.framework.EPPHandleEventException;

/**
 * The <code>OrgExtDomainHandler</code> class extends <code>DomainHandler</code>
 * to include handling the Org Extension.
 */
public class OrgExtDomainHandler extends DomainHandler {

	/** Category for logging */
	private static Logger cat = LoggerFactory.getLogger(OrgExtDomainHandler.class);

	/**
	 * Constructs an instance of OrgExtDomainHandler
	 */
	public OrgExtDomainHandler() {
	}

	/**
	 * Ensure that there is an active session.
	 *
	 * @param aEvent
	 *           The <code>EPPEvent</code> that is being handled
	 * @param aData
	 *           Any data that a Server needs to send to this
	 *           <code>EPPDomainHandler</code> This is assumed to be an instance
	 *           of SessionData here.
	 *
	 * @exception EPPHandleEventException
	 *               Thrown if an error condition occurs. It must contain an
	 *               <code>EPPEventResponse</code>
	 */
	protected void preHandleEvent(EPPEvent aEvent, Object aData) throws EPPHandleEventException {

		super.preHandleEvent(aEvent, aData);

	}

	/**
	 * Override base handler <code>doDomainInfo</code> method and add handling of
	 * the Org Extension.
	 * 
	 * @param aEvent
	 *           The <code>EPPEvent</code> that is being handled
	 * @param aData
	 *           Any data that a Server needs to send to this
	 *           <code>EPPDomaindHandler</code>
	 *
	 * @return EPPEventResponse The response that should be sent back to the
	 *         client.
	 */
	protected EPPEventResponse doDomainInfo(EPPEvent aEvent, Object aData) {

		EPPDomainInfoCmd theCommand = (EPPDomainInfoCmd) aEvent.getMessage();

		EPPEventResponse theEventResponse = super.doDomainInfo(aEvent, aData);

		SessionData theSessionData = (SessionData) aData;

		// Should the org extension be added?
		if (theSessionData.getLoginCmd().hasExtensionService(EPPOrgExtFactory.NS)
		      && theCommand.getName().equalsIgnoreCase("org.example")) {
			EPPResponse theResponse = (EPPResponse) theEventResponse.getResponse();

			// Add extension
			EPPOrgExtInfData theExt = new EPPOrgExtInfData();
			theExt.addOrgId(new EPPOrgExtId("reseller", "myreseller"));
			theExt.addOrgId(new EPPOrgExtId("privacyproxy", "myproxy"));
			theResponse.addExtension(theExt);
		}

		cat.info("doDomainInfo: EPPOrgExt added to response");
		System.out.println("doDomainInfo: EPPOrgExt added to response");

		return theEventResponse;
	}

	/**
	 * Override base handler <code>doDomainCreate</code> method and add handling
	 * of the Org Extension.
	 * 
	 * @param aEvent
	 *           The <code>EPPEvent</code> that is being handled
	 * @param aData
	 *           Any data that a Server needs to send to this
	 *           <code>EPPDomaindHandler</code>
	 *
	 * @return EPPEventResponse The response that should be sent back to the
	 *         client.
	 */
	protected EPPEventResponse doDomainCreate(EPPEvent aEvent, Object aData) {

		EPPDomainCreateCmd theCommand = (EPPDomainCreateCmd) aEvent.getMessage();

		if (theCommand.hasExtension(EPPOrgExtCreate.class)) {
			EPPOrgExtCreate theExt = (EPPOrgExtCreate) theCommand.getExtension(EPPOrgExtCreate.class);

			if (theExt.hasOrgIds()) {
				for (EPPOrgExtId theOrgId : theExt.getOrgIds()) {
					System.out.println("doDomainCreate: org role = " + theOrgId.getRole());
					cat.info("doDomainCreate: org role = " + theOrgId.getRole());
					System.out.println("doDomainCreate: org identifer = " + theOrgId.getOrgId());
					cat.info("doDomainCreate: org identifer = " + theOrgId.getOrgId());
				}
			}

		}
		else {
			cat.info("doDomainCreate: Org Extension = null");
			System.out.println("doDomainCreate: Org Extension = null");
		}

		EPPEventResponse theEventResponse = super.doDomainCreate(aEvent, aData);

		return theEventResponse;
	}

	/**
	 * Override base handler <code>doDomainUpdate</code> method and add handling
	 * of the Org Extension.
	 * 
	 * @param aEvent
	 *           The <code>EPPEvent</code> that is being handled
	 * @param aData
	 *           Any data that a Server needs to send to this
	 *           <code>EPPDomaindHandler</code>
	 *
	 * @return EPPEventResponse The response that should be sent back to the
	 *         client.
	 */
	protected EPPEventResponse doDomainUpdate(EPPEvent aEvent, Object aData) {

		EPPDomainUpdateCmd theCommand = (EPPDomainUpdateCmd) aEvent.getMessage();

		if (theCommand.hasExtension(EPPOrgExtUpdate.class)) {
			EPPOrgExtUpdate theExt = (EPPOrgExtUpdate) theCommand.getExtension(EPPOrgExtUpdate.class);
			if (theExt.hasAddOrgs()) {
				cat.info("doDomainUpdate: Add organizations = " + theExt.getAddOrgs());
				System.out.println("doDomainUpdate: Add organizations = " + theExt.getAddOrgs());
			}
			if (theExt.hasRemOrgs()) {
				cat.info("doDomainUpdate: Remove organizations = " + theExt.getRemOrgs());
				System.out.println("doDomainUpdate: Remove organizations = " + theExt.getRemOrgs());
			}
			if (theExt.hasChgOrgs()) {
				cat.info("doDomainUpdate: Change organizations = " + theExt.getChgOrgs());
				System.out.println("doDomainUpdate: Change organizations = " + theExt.getChgOrgs());
			}

		}
		else {
			cat.info("doDomainUpdate: Org Extension = null");
			System.out.println("doDomainUpdate: Org Extension = null");
		}

		EPPEventResponse theEventResponse = super.doDomainUpdate(aEvent, aData);

		return theEventResponse;
	}

}
