/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.org;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCheckCmd;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;


/**
 * Represents a org &lt;check&gt; command that enables checking on the
 * availability of a set of client-specified but server-unique org identifiers.
 * 
 * @see com.verisign.epp.codec.org.EPPOrgCheckResp
 */
public class EPPOrgCheckCmd extends EPPCheckCmd {

	/**
	 * Category for logging
	 */
	    private static Logger cat = LoggerFactory.getLogger(EPPOrgCheckCmd.class);
	      

	/**
	 * XML local name for <code>EPPOrgCheckCmd</code>.
	 */
	public static final String ELM_LOCALNAME = "check";

	/**
	 * XML root tag for <code>EPPOrgCheckCmd</code>.
	 */
	public static final String ELM_NAME = EPPOrgMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML local name for the org identifier element
	 */
	private static final String ELM_ORG_ID = "id";

	/**
	 * Org identifiers to check for availability.
	 */
	private List<String> orgIds = new ArrayList<String>();

	/**
	 * <code>EPPOrgCheckCmd</code> default constructor.
	 */
	public EPPOrgCheckCmd() {
	}

	/**
	 * <code>EPPOrgCheckCmd</code> constructor that only takes the client
	 * transaction identifier
	 * 
	 * @param aTransId
	 *           Transaction Id associated with command.
	 */
	public EPPOrgCheckCmd(String aTransId) {
		super(aTransId);
	}

	/**
	 * <code>EPPOrgCheckCmd</code> constructor for setting an individual org
	 * identifier to check.
	 * 
	 * @param aTransId
	 *           Transaction Id associated with command.
	 * @param aOrgId
	 *           Org identifier to check
	 */
	public EPPOrgCheckCmd(String aTransId, String aOrgId) {
		super(aTransId);
		this.addOrgId(aOrgId);
	}

	/**
	 * <code>EPPOrgCheckCmd</code> constructor for setting the list of org
	 * identifiers to check.
	 * 
	 * @param aTransId
	 *           Transaction Id associated with command.
	 * @param aOrgIds
	 *           List of Org identifiers to check
	 */
	public EPPOrgCheckCmd(String aTransId, List<String> aOrgIds) {
		super(aTransId);
		this.setOrgIds(aOrgIds);
	}

	/**
	 * Gets the EPP command Namespace associated with <code>EPPOrgCheckCmd</code>
	 * .
	 * 
	 * @return <code>EPPOrgMapFactory.NS</code>
	 */
	public String getNamespace() {
		return EPPOrgMapFactory.NS;
	}

	/**
	 * Are any org identifiers defined in the list of org identifiers?
	 * 
	 * @return <code>true</code> if there is at least one org identifier defined;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasOrgIds() {
		if (this.orgIds != null && !this.orgIds.isEmpty()) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Get the list of org identifiers to check.
	 * 
	 * @return {@code List} of org identifier {@code String}'s
	 */
	public List<String> getOrgIds() {
		return this.orgIds;
	}

	/**
	 * Set the list of org identifiers to check.
	 * 
	 * @param aOrgIds
	 *           {@code List} of org identifier {@code String}'s
	 */
	public void setOrgIds(List<String> aOrgIds) {
		if (aOrgIds == null) {
			this.orgIds = new ArrayList<String>();
		}
		else {
			this.orgIds = aOrgIds;
		}
	}

	/**
	 * Set an individual org identifier to check. This method clears the existing
	 * list of org identifiers.
	 * 
	 * @param aOrgId
	 *           Org identifier to check
	 */
	public void setOrgId(String aOrgId) {
		this.orgIds = new ArrayList<String>();
		this.orgIds.add(aOrgId);
	}

	/**
	 * Append a org identifier to the list of org identifiers to check. This
	 * method does NOT clear the existing list of org identifiers.
	 * 
	 * @param aOrgId
	 *           Org identifier to check
	 */
	public void addOrgId(String aOrgId) {
		this.orgIds.add(aOrgId);
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * <code>EPPOrgCheckCmd</code> instance.
	 * 
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 * 
	 * @return Element Root DOM Element representing the
	 *         <code>EPPOrgCheckCmd</code> instance.
	 * 
	 * @exception EPPEncodeException
	 *               Unable to encode <code>EPPOrgCheckCmd</code> instance.
	 */
	protected Element doEncode(Document aDocument) throws EPPEncodeException {

		// Check required attributes
		if (!this.hasOrgIds()) {
			throw new EPPEncodeException("At least one org identifier is needed with EPPOrgCheckCmd");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPOrgMapFactory.NS, ELM_NAME);

		// Org Identifiers
		EPPUtil.encodeList(aDocument, root, this.orgIds, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_ORG_ID);

		return root;
	}

	/**
	 * Decode the <code>EPPOrgCheckCmd</code> attributes from the aElement DOM
	 * Element tree.
	 * 
	 * @param aElement
	 *           Root DOM Element to decode <code>EPPOrgCheckCmd</code> from.
	 * 
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	protected void doDecode(Element aElement) throws EPPDecodeException {

		// Org Identifiers
		this.orgIds = EPPUtil.decodeList(aElement, EPPOrgMapFactory.NS, ELM_ORG_ID);
	}

	/**
	 * Compare an instance of <code>EPPOrgCheckCmd</code> with this instance.
	 * 
	 * @param aObject
	 *           Object to compare with.
	 * 
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise.
	 */
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPOrgCheckCmd)) {
			return false;
		}

		if (!super.equals(aObject)) {
			return false;
		}

		EPPOrgCheckCmd theMap = (EPPOrgCheckCmd) aObject;

		// Org Identifiers
		if (!EPPUtil.equalLists(this.orgIds, theMap.orgIds)) {
			cat.error("EPPOrgCheckCmd.equals(): orgIds not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone <code>EPPOrgCheckCmd</code>.
	 * 
	 * @return Deep copy clone of <code>EPPOrgCheckCmd</code>
	 * 
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPOrgCheckCmd clone = (EPPOrgCheckCmd) super.clone();

		// Org Identifiers
		clone.orgIds = (List) ((ArrayList) this.orgIds).clone();

		return clone;
	}

	/**
	 * Implementation of <code>Object.toString</code>, which will result in an
	 * indented XML <code>String</code> representation of the concrete
	 * <code>EPPCodecComponent</code>.
	 * 
	 * @return Indented XML <code>String</code> if successful; <code>ERROR</code>
	 *         otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

}
