/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/

package com.verisign.epp.codec.namestoreext;

import java.util.Hashtable;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;


/**
 * Namestore &lt;nsExtErrData&gt; extension element to an error EPP Response. The
 * error code and message is currently associated with an EPP response code of
 * 2306 "Parameter value policy error", and can have one of the
 * {@code EPPNamestoreExtNSExtErrData}{@code ERROR} constant values.
 * Optionally, a "lang" attribute can be provide to indicate the language. The
 * default value for "lang" is "en" (English). <br><br>
 *
 * @author $Author: jim $
 * @version $Revision: 1.3 $
 */
public class EPPNamestoreExtNSExtErrData implements EPPCodecComponent {
	/** The default language of the error message "en". */
	public static final String DEFAULT_LANG = "en";

	/** Specified sub-product does not exist */
	public static final int ERROR_SUB_PRODUCT_NOT_EXISTS = 1;

	/** XML root tag for {@code EPPNamestoreExtNSExtErrData}. */
	public static final String ELM_NAME = "namestoreExt:nsExtErrData";

	/**
	 * Hash that maps the pre-defined result codes to their associated message
	 * in     the default language of "en".
	 */
	private static Hashtable _defaultMsg;

	// Static Initializer for EPPNamestoreExtNSExtErrData
	static {
		_defaultMsg = new Hashtable();
		_defaultMsg.put(
						Integer.valueOf(ERROR_SUB_PRODUCT_NOT_EXISTS),
						"Specified sub-product does not exist");
	}

	/** XML tag name for the {@code _message} element. */
	private final static String ELM_MSG = "namestoreExt:msg";

	/** XML attribute name for the result {@code _code} attribute. */
	private final static String ATTR_CODE = "code";

	/**
	 * XML attribute name for the optional result {@code _lang} attribute.
	 */
	private final static String ATTR_LANG = "lang";

	/** Error code */
	private int _code = 0;

	/**
	 * Error message associated with _code.  A default English message is set
	 * by default.
	 */
	private String _message = "";

	/** Language of the {@code _message} attribute. */
	private String _lang = DEFAULT_LANG;

	/**
	 * Default constructor.  The error code is set to -1.
	 */
	public EPPNamestoreExtNSExtErrData() {
		// Do nothing
	}


	/**
	 * Constructor that sets the error code.  Use of the the {@code ERROR}
	 * constants for the error code.
	 *
	 * @param aCode Error code
	 */
	public EPPNamestoreExtNSExtErrData(int aCode) {
		_code		 = aCode;
		_message     = (String) _defaultMsg.get(Integer.valueOf(aCode));
	}


	/**
	 * Gets the error code.
	 *
	 * @return Error code that should be one of the {@code ERROR} constant
	 * 		   values.
	 */
	public int getCode() {
		return _code;
	}


	/**
	 * Sets the error code.
	 *
	 * @param aCode Error code that should be one of the {@code ERROR}
	 * 		  constant values.
	 */
	public void setCode(int aCode) {
		_code = aCode;
	}


	/**
	 * Sets the error code and the default "en" message associated with the
	 * error code if {@code aUserDefaultMessage} is set to
	 * {@code true}.
	 *
	 * @param aCode Error code that should be one of the {@code ERROR}
	 * 		  constant values.
	 * @param aUseDefaultMessage Use the default en message associated with
	 * 		  aCode?
	 */
	public void setCode(int aCode, boolean aUseDefaultMessage) {
		_code = aCode;

		if (aUseDefaultMessage) {
			_message = (String) _defaultMsg.get(Integer.valueOf(aCode));
		}
	}


	/**
	 * Gets the error message.
	 *
	 * @return Error message associated with the error code in the specified
	 * 		   language.
	 */
	public String getMessage() {
		return _message;
	}


	/**
	 * Sets the error message.  This should only be called if the default "en"
	 * language message is not valid.
	 *
	 * @param aMessage Error message associated with the error code in the
	 * 		  specified language.
	 */
	public void setMessage(String aMessage) {
		_message = aMessage;
	}


	/**
	 * Gets the error message language.  The Language must be structured as
	 * documented in <a
	 * href="http://www.ietf.org/rfc/rfc1766.txt?number=1766">[RFC1766]</a>.
	 *
	 * @return Error message language.
	 */
	public String getLang() {
		return _lang;
	}


	/**
	 * Sets the error message language.  The Language must be structured as
	 * documented in <a
	 * href="http://www.ietf.org/rfc/rfc1766.txt?number=1766">[RFC1766]</a>.
	 *
	 * @param aLang DOCUMENT ME!
	 */
	public void setLang(String aLang) {
		_lang = aLang;
	}


	/**
	 * encode instance into a DOM element tree. A DOM Document is passed as an
	 * argument and functions as a factory for DOM objects.  The root element
	 * associated with the instance is created and each instance attributeis
	 * appended as a child node.
	 *
	 * @param aDocument DOM Document, which acts is an Element factory
	 *
	 * @return Element Root element associated with the object
	 *
	 * @exception EPPEncodeException Error encoding
	 * 			  {@code EPPNamestoreExtNSExtErrData}
	 */
	public Element encode(Document aDocument) throws EPPEncodeException {
		Element currElm;
		Text    currVal;

		// Validate state
		if (_code == -1) {
			throw new EPPEncodeException("code required attribute is not set");
		}

		if (_message == null) {
			throw new EPPEncodeException("message required attribute is not set");
		}

		Element root =
			aDocument.createElementNS(EPPNamestoreExtExtFactory.NS, ELM_NAME);
		root.setAttribute("xmlns:namestoreExt", EPPNamestoreExtExtFactory.NS);

		// Message
		currElm =
			aDocument.createElementNS(EPPNamestoreExtExtFactory.NS, ELM_MSG);
		currVal = aDocument.createTextNode(_message);

		// Code
		currElm.setAttribute(ATTR_CODE, _code + "");

		// Lang
		if (!_lang.equals(DEFAULT_LANG)) {
			currElm.setAttribute(ATTR_LANG, _lang);
		}

		currElm.appendChild(currVal);
		root.appendChild(currElm);

		return root;
	}


	/**
	 * decode a DOM element tree to initialize the instance attributes.  The
	 * {@code aElement} argument represents the root DOM element and is
	 * used to traverse the DOM nodes for instance attribute values.
	 *
	 * @param aElement {@code Element} to decode
	 *
	 * @exception EPPDecodeException Error decoding {@code Element}
	 */
	public void decode(Element aElement) throws EPPDecodeException {
		Element  currElm;

		// Message
		currElm = EPPUtil.getElementByTagNameNS(aElement,
				EPPNamestoreExtExtFactory.NS, ELM_MSG);
		
		if (currElm == null) {
			throw new EPPDecodeException("Required EPPNamestoreExtNSExtErrData element "
										 + ELM_MSG + " not found");
		}

		_message = currElm.getFirstChild().getNodeValue();

		if (_message == null) {
			throw new EPPDecodeException("Required message value of EPPNamestoreExtNSExtErrData element "
										 + ELM_MSG + " not found");
		}

		// Lang
		setLang(currElm.getAttribute(ATTR_LANG));

		// Code
		_code = Integer.parseInt(currElm.getAttribute(ATTR_CODE));
	}


	/**
	 * Compare an instance of {@code EPPNamestoreExtNSExtErrData} with
	 * this instance.
	 *
	 * @param aObject Object to compare with.
	 *
	 * @return {@code true} if equal; {@code false} otherwise.
	 */
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPNamestoreExtNSExtErrData)) {
			return false;
		}

		EPPNamestoreExtNSExtErrData theComp =
			(EPPNamestoreExtNSExtErrData) aObject;

		// _code
		if (_code != theComp._code) {
			return false;
		}

		// _message
		if (
			!(
					(_message == null) ? (theComp._message == null)
										   : _message.equals(theComp._message)
				)) {
			return false;
		}

		// _lang
		if (!_lang.equals(theComp._lang)) {
			return false;
		}

		return true;
	}


	/**
	 * clone an {@code EPPCodecComponent}.
	 *
	 * @return clone of concrete {@code EPPNamestoreExtNSExtErrData}
	 *
	 * @exception CloneNotSupportedException standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPNamestoreExtNSExtErrData clone =
			(EPPNamestoreExtNSExtErrData) super.clone();

		return clone;
	}


	/**
	 * Gets the root element name.
	 *
	 * @return "persReg:creErrData"
	 */
	protected String getRootElm() {
		return ELM_NAME;
	}

 	/**
 	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
 	 * 
 	 * @return XML namespace for the {@code EPPCodecComponent}. 
 	 */
 	@Override
 	public String getNamespace() {
 		return EPPNamestoreExtExtFactory.NS;
 	}
}


