/***********************************************************
Copyright (C) 2021 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.maintenance.v1_0;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPMaintenanceSystem} indicates what impacted system with the
 * following attributes:<br>
 * <ul>
 * <li>name - Name of the affected system, such as 'EPP', 'WHOIS', 'DNS',
 * 'Portal', etc..</li>
 * <li>host - Indicates the affected maintained system based on host name or
 * host address.</li>
 * <li>impact - Indicates the impact level, which is either 'full' or
 * 'partial'.
 * </ul>
 */
public class EPPMaintenanceSystem implements EPPCodecComponent {

	/**
	 * Impact enumerated values.
	 */
	public enum Impact {
		full, partial, none;
	}

	/**
	 * Category for logging
	 */
	    private static Logger cat = LoggerFactory.getLogger(EPPMaintenanceSystem.class);
	      

	/**
	 * XML local name for {@code EPPMaintenanceSystem}.
	 */
	public static final String ELM_LOCALNAME = "system";

	/**
	 * XML root tag for {@code EPPMaintenance}.
	 */
	public static final String ELM_NAME = EPPMaintenanceMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML local name for the name element
	 */
	private final static String ELM_SYSTEM_NAME = "name";

	/**
	 * XML local name for the host element
	 */
	private final static String ELM_HOST = "host";

	/**
	 * XML local name for the impact element
	 */
	private final static String ELM_IMPACT = "impact";

	/**
	 * Name of the effected system, such as 'EPP', 'WHOIS', 'DNS', 'Portal', etc.
	 */
	private String name;

	/**
	 * Affected maintained system
	 */
	private String host;

	/**
	 * Impact level with the default value of {@link Impact#none}.
	 */
	private Impact impact = Impact.none;

	/**
	 * Default constructor for {@code EPPMaintenanceSystem}. The required
	 * attributes need to be set. The {@code impact} attribute defaults to
	 * {@link Impact#partial}.
	 */
	public EPPMaintenanceSystem() {
	}

	/**
	 * Constructor for {@code EPPMaintenanceSystem} with all of the attributes as
	 * parameters.
	 *
	 * @param aName
	 *           Name of the affected system, such as 'EPP', 'WHOIS', 'DNS',
	 *           'PORTAL', etc..
	 * @param aHost
	 *           Host of the system
	 * @param aImpact
	 *           Impact of system maintenance
	 */
	public EPPMaintenanceSystem(String aName, String aHost, Impact aImpact) {
		this.name = aName;
		this.host = aHost;
		this.impact = aImpact;
	}

	/**
	 * Gets the name of the affected system, such as 'EPP', 'WHOIS', 'DNS',
	 * 'PORTAL', etc..
	 *
	 * @return Name of the affected system if set; {@code null} otherwise.
	 */
	public String getName() {
		return this.name;
	}

	/**
	 * Sets the name of the affected system, such as 'EPP', 'WHOIS', 'DNS',
	 * 'PORTAL', etc..
	 *
	 * @param aName
	 *           Name of the affected system
	 */
	public void setName(String aName) {
		this.name = aName;
	}

	/**
	 * Is the system host defined?
	 *
	 * @return {@code true} if the system host is defined; {@code false}
	 *         otherwise.
	 */
	public boolean hasHost() {
		return (this.host != null ? true : false);
	}
	
	
	/**
	 * Gets the OPTIONAL affected maintained system host.
	 *
	 * @return The affected maintained system host if defined; {@code null}
	 *         otherwise.
	 */
	public String getHost() {
		return this.host;
	}

	/**
	 * Sets the OPTIONAL affected maintained system host.
	 *
	 * @param aHost
	 *           affected maintained system host.
	 */
	public void setHost(String aHost) {
		this.host = aHost;
	}

	/**
	 * Gets the impact level.
	 *
	 * @return Returns the impact level with the default set to
	 *         {@link Impact#partial}.
	 */
	public Impact getImpact() {
		return this.impact;
	}

	/**
	 * Sets the impact level.
	 *
	 * @param impact
	 *           level of the system maintenance
	 */
	public void setImpact(Impact impact) {
		this.impact = impact;
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPMaintenanceSystem} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Root DOM Element representing the {@code EPPMaintenanceSystem}
	 *         instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPMaintenanceSystem} instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {

		// Validate state
		if ((this.name == null) || (this.impact == null)) {
			throw new EPPEncodeException("required attribute is not set");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPMaintenanceMapFactory.NS, ELM_NAME);

		// Name
		EPPUtil.encodeString(aDocument, root, this.name, EPPMaintenanceMapFactory.NS,
		      EPPMaintenanceMapFactory.NS_PREFIX + ":" + ELM_SYSTEM_NAME);

		// Host
		EPPUtil.encodeString(aDocument, root, this.host, EPPMaintenanceMapFactory.NS,
		      EPPMaintenanceMapFactory.NS_PREFIX + ":" + ELM_HOST);

		// Impact
		EPPUtil.encodeString(aDocument, root, this.impact.toString(), EPPMaintenanceMapFactory.NS,
		      EPPMaintenanceMapFactory.NS_PREFIX + ":" + ELM_IMPACT);

		return root;
	}

	/**
	 * Decode the {@code EPPMaintenanceSystem} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPMaintenanceSystem} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement.
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {

		// Name
		this.name = EPPUtil.decodeString(aElement, EPPMaintenanceMapFactory.NS, ELM_SYSTEM_NAME);

		// Host
		this.host = EPPUtil.decodeString(aElement, EPPMaintenanceMapFactory.NS, ELM_HOST);

		// Impact
		this.impact = Impact.valueOf(EPPUtil.decodeString(aElement, EPPMaintenanceMapFactory.NS, ELM_IMPACT));
	}

	/**
	 * implements a deep {@code EPPMaintenanceSystem} compare.
	 *
	 * @param aObject
	 *           {@code EPPMaintenanceSystem} instance to compare with
	 *
	 * @return {@code true} of {@code aObject} is equal to instance;
	 *         {@code false} otherwise.
	 */
	@Override
	public boolean equals(Object aObject) {

		if (!(aObject instanceof EPPMaintenanceSystem)) {
			return false;
		}

		EPPMaintenanceSystem other = (EPPMaintenanceSystem) aObject;

		// Name
		if (!EqualityUtil.equals(this.name, other.name)) {
			cat.error("EPPMaintenanceSystem.equals(): name not equal");
			return false;
		}

		// Host
		if (!EqualityUtil.equals(this.host, other.host)) {
			cat.error("EPPMaintenanceSystem.equals(): host not equal");
			return false;
		}

		// Impact
		if (!EqualityUtil.equals(this.impact, other.impact)) {
			cat.error("EPPMaintenanceSystem.equals(): impact not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone {@code EPPMaintenanceSystem}.
	 *
	 * @return clone of {@code EPPMaintenanceSystem}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPMaintenanceSystem clone = null;

		clone = (EPPMaintenanceSystem) super.clone();

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPMaintenanceMapFactory.NS;
	}

}
