/***********************************************************
Copyright (C) 2021 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.maintenance.v1_0;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPMaintenanceIntervention} indicates what impact the maintenance will
 * have to the client with the two boolean attributes:<br>
 * <ul>
 * <li>connection - to indicate if a client needs to do something that is
 * connection-related, such as a reconnect.</li>
 * <li>implementation - To indicate if a client needs to do something that is
 * implementation-related, such as a code change.</li>
 * </ul>
 */
public class EPPMaintenanceIntervention implements EPPCodecComponent {

	/**
	 * Category for logging
	 */
	    private static Logger cat = LoggerFactory.getLogger(EPPMaintenanceIntervention.class);
	      

	/**
	 * XML local name for {@code EPPMaintenanceIntervention}.
	 */
	public static final String ELM_LOCALNAME = "intervention";

	/**
	 * XML root tag for {@code EPPMaintenanceIntervention}.
	 */
	public static final String ELM_NAME = EPPMaintenanceMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML local name for the connection element
	 */
	private final static String ELM_CONNECTION = "connection";

	/**
	 * XML local name for the implementation element
	 */
	private final static String ELM_IMPLEMENTATION = "implementation";

	/**
	 * Connection attribute that indicates if a client needs to do something that
	 * is connection-related, such as a reconnect. The default value is
	 * {@code true}.
	 */
	private boolean connection = true;

	/**
	 * Connection attribute that indicates if a client needs to do something that
	 * is implementation-related, such as a code change. The default value is
	 * {@code true}.
	 */
	private boolean implementation = true;

	/**
	 * Default constructor for {@code EPPMaintenanceIntervention}. The attributes
	 * {@code connection} and {@code implementation} are both set to
	 * {@code true}.
	 */
	public EPPMaintenanceIntervention() {
	}

	/**
	 * Constructor for {@code EPPMaintenanceIntervention} with all of the attributes as
	 * parameters.
	 *
	 * @param aConnection
	 *           Indicates if a client needs to do something that is
	 *           connection-related, such as a reconnect.
	 * @param aImplementation
	 *           indicates if a client needs to do something that is
	 *           implementation-related, such as a code change.
	 */
	public EPPMaintenanceIntervention(boolean aConnection, boolean aImplementation) {
		this.connection = aConnection;
		this.implementation = aImplementation;
	}

	/**
	 * Gets whether the client needs to do something that is connection-related,
	 * such as a reconnect.
	 *
	 * @return {@code true} if the client does need to take a connection-related
	 *         action; {@code false} otherwise.
	 */
	public boolean isConnection() {
		return this.connection;
	}

	/**
	 * Sets whether the client needs to do something that is connection-related,
	 * such as a reconnect.
	 *
	 * @param aConnection
	 *           {@code true} if the client does need to take a
	 *           connection-related action; {@code false} otherwise.
	 */
	public void setConnection(boolean aConnection) {
		this.connection = aConnection;
	}

	/**
	 * Gets whether the client needs to do something that is
	 * implementation-related, such as a code change.
	 *
	 * @return {@code true} if the client does need to take a
	 *         implementation-related action; {@code false} otherwise.
	 */
	public boolean isImplementation() {
		return this.implementation;
	}

	/**
	 * Sets whether the client needs to do something that is
	 * implementation-related, such as a code change.
	 *
	 * @param aImplementation
	 *           {@code true} if the client does need to take a
	 *           implementation-related action; {@code false} otherwise.
	 */
	public void setImplementation(boolean aImplementation) {
		this.implementation = aImplementation;
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPMaintenanceIntervention} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Root DOM Element representing the {@code EPPMaintenanceIntervention} instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPMaintenanceIntervention} instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {

		// Create root element
		Element root = aDocument.createElementNS(EPPMaintenanceMapFactory.NS, ELM_NAME);

		// Connection
		EPPUtil.encodeBoolean(aDocument, root, this.connection, EPPMaintenanceMapFactory.NS,
		      EPPMaintenanceMapFactory.NS_PREFIX + ":" + ELM_CONNECTION);

		// Implementation
		EPPUtil.encodeBoolean(aDocument, root, this.implementation, EPPMaintenanceMapFactory.NS,
		      EPPMaintenanceMapFactory.NS_PREFIX + ":" + ELM_IMPLEMENTATION);

		return root;
	}

	/**
	 * Decode the {@code EPPMaintenanceIntervention} attributes from the aElement DOM Element
	 * tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPMaintenanceIntervention} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement.
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {

		Boolean theVal;

		// Connection
		theVal = EPPUtil.decodeBoolean(aElement, EPPMaintenanceMapFactory.NS, ELM_CONNECTION);
		if (theVal == null) {
			throw new EPPDecodeException("Required element " + ELM_CONNECTION + " does not exist");
		}
		else {
			this.connection = theVal;
		}

		// Implementation
		theVal = EPPUtil.decodeBoolean(aElement, EPPMaintenanceMapFactory.NS, ELM_IMPLEMENTATION);
		if (theVal == null) {
			throw new EPPDecodeException("Required element " + ELM_IMPLEMENTATION + " does not exist");
		}
		else {
			this.implementation = theVal;
		}
	}

	/**
	 * implements a deep {@code EPPMaintenanceIntervention} compare.
	 *
	 * @param aObject
	 *           {@code EPPMaintenanceIntervention} instance to compare with
	 *
	 * @return {@code true} of {@code aObject} is equal to instance;
	 *         {@code false} otherwise.
	 */
	@Override
	public boolean equals(Object aObject) {

		if (!(aObject instanceof EPPMaintenanceIntervention)) {
			return false;
		}

		EPPMaintenanceIntervention other = (EPPMaintenanceIntervention) aObject;

		// Connection
		if (!EqualityUtil.equals(this.connection, other.connection)) {
			cat.error("EPPMaintenanceIntervention.equals(): connection not equal");
			return false;
		}

		// Implementation
		if (!EqualityUtil.equals(this.implementation, other.implementation)) {
			cat.error("EPPMaintenanceIntervention.equals(): implementation not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone {@code EPPMaintenanceIntervention}.
	 *
	 * @return clone of {@code EPPMaintenanceIntervention}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPMaintenanceIntervention clone = null;

		clone = (EPPMaintenanceIntervention) super.clone();

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPMaintenanceMapFactory.NS;
	}

}
