/***********************************************************
Copyright (C) 2021 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.maintenance.v1_0;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPMaintenanceDescription} is a free-form descriptions of the
 * maintenance without having to create and traverse an external resource
 * defined by the &lt;maint:detail&gt; element. The OPTIONAL "lang" attribute
 * MAY be present to identify the language if the negotiated value is something
 * other then the default value of "en" (English). The OPTIONAL "type" attribute
 * MAY be present to identify the format of the description. It MUST either be
 * "plain" for plain text or "html" HTML text that is defined in [W3C-HTML5] and
 * XML-escaped, with a default value of "plain".
 */
public class EPPMaintenanceDescription implements EPPCodecComponent {

	/**
	 * Description type enumerated values.
	 */
	public enum Type {
		plain, html;
	}

	/**
	 * Default language
	 */
	public static final String DEFAULT_LANG = "en";

	/**
	 * Category for logging
	 */
	    private static Logger cat = LoggerFactory.getLogger(EPPMaintenanceDescription.class);
	      

	/**
	 * XML local name for {@code EPPMaintenanceDescription}.
	 */
	public static final String ELM_LOCALNAME = "description";

	/**
	 * XML root tag for {@code EPPMaintenance}.
	 */
	public static final String ELM_NAME = EPPMaintenanceMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML attribute for the description lang attribute.
	 */
	private static final String ATTR_LANG = "lang";

	/**
	 * XML attribute for the description type attribute.
	 */
	private static final String ATTR_TYPE = "type";

	/**
	 * OPTIONAL language of the description.
	 */
	private String lang = DEFAULT_LANG;

	/**
	 * OPTIONAL description type with the default of {@link Type#plain}.
	 */
	private Type type = Type.plain;

	/**
	 * Description value
	 */
	private String description;

	/**
	 * Default constructor for {@code EPPMaintenanceDescription}. The required
	 * attributes need to be set. The {@code type} attribute defaults to
	 * {@link Type#plain} and the {@code lang} attribute defaults to
	 * {@link DEFAULT_LANG}.
	 */
	public EPPMaintenanceDescription() {
	}

	/**
	 * Constructor for {@code EPPMaintenanceDescription} with the required
	 * attribute as a parameter.
	 *
	 * @param aDescription
	 *           Description value
	 */
	public EPPMaintenanceDescription(String aDescription) {
		this.description = aDescription;
	}

	/**
	 * Constructor for {@code EPPMaintenanceDescription} with all of the
	 * attributes as parameters.
	 *
	 * @param aDescription
	 *           Description value
	 * @param aType
	 *           Type of the description
	 * @param aLang
	 *           Language of the description
	 */
	public EPPMaintenanceDescription(String aDescription, Type aType, String aLang) {
		this.description = aDescription;
		this.type = aType;
		this.lang = aLang;
	}

	/**
	 * Gets the free-form description of the maintenance without having to create
	 * an external resource.
	 *
	 * @return The free-form description of the maintenance if defined;
	 *         {@code null} otherwise.
	 */
	public String getDescription() {
		return this.description;
	}

	/**
	 * Sets the free-form description of the maintenance without having to create
	 * an external resource.
	 *
	 * @param aDescription
	 *           Free-form description of the maintenance without having to
	 *           create an external resource.
	 */
	public void setDescription(String aDescription) {
		this.description = aDescription;
	}

	/**
	 * Is the description language defined with a non-default value?
	 *
	 * @return <code>true</code> if the description language is defined;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasLang() {
		return (!this.lang.equals(DEFAULT_LANG) ? true : false);
	}

	/**
	 * Gets the description language value.
	 *
	 * @return Description language if defined; <code>DEFAULT_LANG</code>
	 *         otherwise.
	 */
	public String getLang() {
		return this.lang;
	}

	/**
	 * Sets the description language value.
	 *
	 * @param aLang
	 *           Description language for value.
	 */
	public void setLang(String aLang) {
		if (aLang == null) {
			this.lang = DEFAULT_LANG;
		}
		else {
			this.lang = aLang;
		}
	}

	/**
	 * Gets the description type value.
	 *
	 * @return Description type with a default value of {@link Type#plain}.
	 */
	public Type getType() {
		return this.type;
	}

	/**
	 * Sets the description type value.
	 *
	 * @param aType
	 *           Description type value.
	 */
	public void setType(Type aType) {
		if (aType == null) {
			this.type = Type.plain;
		}
		else {
			this.type = aType;
		}
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPMaintenanceDescription} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Root DOM Element representing the
	 *         {@code EPPMaintenanceDescription} instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPMaintenanceDescription} instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {

		// Validate state
		if ((this.description == null) || (this.type == null) || (this.lang == null)) {
			throw new EPPEncodeException("required attribute is not set");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPMaintenanceMapFactory.NS, ELM_NAME);

		// Description value
		Text currVal = aDocument.createTextNode(this.description);
		root.appendChild(currVal);

		// Type
		root.setAttribute(ATTR_TYPE, this.type.toString());

		// Lang
		root.setAttribute(ATTR_LANG, this.lang);

		return root;
	}

	/**
	 * Decode the {@code EPPMaintenanceDescription} attributes from the aElement
	 * DOM Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPMaintenanceDescription}
	 *           from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement.
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {

		// Description value
		this.description = EPPUtil.decodeStringValue(aElement);

		// Type
		this.setType(Type.valueOf(aElement.getAttribute(ATTR_TYPE)));

		// Lang
		this.setLang(EPPUtil.decodeStringAttr(aElement, ATTR_LANG));
	}

	/**
	 * implements a deep {@code EPPMaintenanceDescription} compare.
	 *
	 * @param aObject
	 *           {@code EPPMaintenanceDescription} instance to compare with
	 *
	 * @return {@code true} of {@code aObject} is equal to instance;
	 *         {@code false} otherwise.
	 */
	@Override
	public boolean equals(Object aObject) {

		if (!(aObject instanceof EPPMaintenanceDescription)) {
			return false;
		}

		EPPMaintenanceDescription other = (EPPMaintenanceDescription) aObject;

		// Description
		if (!EqualityUtil.equals(this.description, other.description)) {
			cat.error("EPPMaintenanceDescription.equals(): description not equal");
			return false;
		}

		// Type
		if (!EqualityUtil.equals(this.type, other.type)) {
			cat.error("EPPMaintenanceDescription.equals(): type not equal");
			return false;
		}

		// Lang
		if (!EqualityUtil.equals(this.lang, other.lang)) {
			cat.error("EPPMaintenanceDescription.equals(): lang not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone {@code EPPMaintenanceDescription}.
	 *
	 * @return clone of {@code EPPMaintenanceDescription}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPMaintenanceDescription clone = null;

		clone = (EPPMaintenanceDescription) super.clone();

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPMaintenanceMapFactory.NS;
	}

}
