/***********************************************************
Copyright (C) 2019 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.loginsecpolicy.v04;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.framework.EPPRegistryPolicyAdapter;


/**
 * {@code EPPLaunchPolicyAdapter} is used to create the login security policy
 * extension information that will be called by the EPP SDK Stub Server
 * RegistryHandler.
 */
public class EPPLoginSecPolicyAdapter extends EPPRegistryPolicyAdapter {

	    private static Logger cat = LoggerFactory.getLogger(EPPLoginSecPolicyAdapter.class);
	      

	EPPLoginSecPolicyInfData systemExt;

	/**
	 * Default constructor that pre-defines a login security policy extension for
	 * use as an extension to the registry info response.
	 */
	public EPPLoginSecPolicyAdapter() {
		EPPLoginSecPolicySystem theSystem = new EPPLoginSecPolicySystem();

		EPPLoginSecPolicyPassword thePassword = new EPPLoginSecPolicyPassword();
		thePassword.setExpression(
		      "(?=.*\\d)(?=.*[a-zA-Z])(?=.*[\\x21-\\x2F\\x3A-\\x40\\x5B-\\x60\\x7B-\\x7E])(?!^\\s+)(?!.*\\s+$)(?!.*\\s{2,})^[\\x20-\\x7e]{16,32}$");
		thePassword.setDescription(
		      "16 to 32 printable characters (alphanumeric, space, and special characters) with at least one number, letter, and special character, with no leading or trailing whitespace, and with no consecutive spaces.");
		thePassword.setSpecialRules(Boolean.FALSE);
		thePassword.setRestrictedWords(Boolean.TRUE);
		thePassword.setRestrictedWordsUrl("http://example.com/restrictedwords");
		theSystem.setPassword(thePassword);

		theSystem.setUserAgentSupport(Boolean.TRUE);

		EPPLoginSecPolicyEvent theEvent;

		// Type "newPW"
		theEvent = new EPPLoginSecPolicyEvent();
		theEvent.setType(EPPLoginSecPolicyEvent.EventType.newPW);
		theEvent.addLevel(EPPLoginSecPolicyEvent.EventLevel.error);
		theEvent.setErrorAction(EPPLoginSecPolicyEvent.EventErrorAction.login);
		theSystem.addEvent(theEvent);

		// Type "password"
		theEvent = new EPPLoginSecPolicyEvent();
		theEvent.setType(EPPLoginSecPolicyEvent.EventType.password);
		theEvent.addLevel(EPPLoginSecPolicyEvent.EventLevel.error);
		theEvent.addLevel(EPPLoginSecPolicyEvent.EventLevel.warning);
		theEvent.setExDate(Boolean.TRUE);
		theEvent.setExPeriod("P90D");
		theEvent.setWarningPeriod("P15D");
		theEvent.setErrorAction(EPPLoginSecPolicyEvent.EventErrorAction.login);
		theSystem.addEvent(theEvent);

		// Type "certificate"
		theEvent = new EPPLoginSecPolicyEvent();
		theEvent.setType(EPPLoginSecPolicyEvent.EventType.certificate);
		theEvent.addLevel(EPPLoginSecPolicyEvent.EventLevel.error);
		theEvent.addLevel(EPPLoginSecPolicyEvent.EventLevel.warning);
		theEvent.setExDate(Boolean.TRUE);
		theEvent.setWarningPeriod("P15D");
		theEvent.setErrorAction(EPPLoginSecPolicyEvent.EventErrorAction.connect);
		theSystem.addEvent(theEvent);

		// Type "cipher"
		theEvent = new EPPLoginSecPolicyEvent();
		theEvent.setType(EPPLoginSecPolicyEvent.EventType.cipher);
		theEvent.addLevel(EPPLoginSecPolicyEvent.EventLevel.warning);
		theEvent.setExDate(Boolean.FALSE);
		theSystem.addEvent(theEvent);

		// Type "tlsProtocol"
		theEvent = new EPPLoginSecPolicyEvent();
		theEvent.setType(EPPLoginSecPolicyEvent.EventType.tlsProtocol);
		theEvent.addLevel(EPPLoginSecPolicyEvent.EventLevel.warning);
		theEvent.setExDate(Boolean.FALSE);
		theSystem.addEvent(theEvent);

		// Type "stat" "failedLogins"
		theEvent = new EPPLoginSecPolicyEvent();
		theEvent.setType(EPPLoginSecPolicyEvent.EventType.stat);
		theEvent.setName("failedLogins");
		theEvent.addLevel(EPPLoginSecPolicyEvent.EventLevel.warning);
		theEvent.setExDate(Boolean.FALSE);
		theEvent.setThreshold(Integer.valueOf(100));
		theEvent.setPeriod("P1D");
		theSystem.addEvent(theEvent);

		// Type "custom" "myCustomEvent"
		theEvent = new EPPLoginSecPolicyEvent();
		theEvent.setType(EPPLoginSecPolicyEvent.EventType.custom);
		theEvent.setName("myCustomEvent");
		theEvent.addLevel(EPPLoginSecPolicyEvent.EventLevel.warning);
		theEvent.setExDate(Boolean.FALSE);
		theSystem.addEvent(theEvent);

		this.systemExt = new EPPLoginSecPolicyInfData(theSystem);
	}

	/**
	 * Gets the login security system policy extension to use in the registry
	 * info response.
	 *
	 * @return {@link EPPLoginSecPolicyInfData} that represents the login
	 *         security policy extension information.
	 */
	@Override
	public EPPCodecComponent getSystemExt() {
		return this.systemExt;
	}

}
