/***********************************************************
Copyright (C) 2019 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.loginsec.v1_0;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * Login Security Extension that extends an
 * {@link com.verisign.epp.codec.gen.EPPLoginCmd} to support passing a longer
 * password (<code>pw</code>) or new password (<code>newPW</code>).
 */
public class EPPLoginSec implements EPPCodecComponent {

	/**
	 * Password set in the <code>pw</code> or <code>newPW</code> fields of the
	 * login command if overridden the password with the password in the
	 * extension.
	 */
	public static final String LOGIN_SECURITY_PASSWORD = "[LOGIN-SECURITY]";

	/**
	 * Category for logging
	 */
	    private static Logger cat = LoggerFactory.getLogger(EPPLoginSec.class);

	/**
	 * XML local name for <code>EPPLoginSec</code>.
	 */
	public static final String ELM_LOCALNAME = "loginSec";

	/**
	 * XML root tag for <code>EPPLoginSec</code>.
	 */
	public static final String ELM_NAME = EPPLoginSecExtFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML tag name for the <code>password</code> attribute.
	 */
	private final static String ELM_PASSWORD = "pw";

	/**
	 * XML tag name for the <code>newPassword</code> attribute.
	 */
	private final static String ELM_NEW_PASSWORD = "newPW";

	/**
	 * User agent
	 */
	private EPPLoginSecUserAgent userAgent = null;

	/**
	 * Client password
	 */
	private String password = null;

	/**
	 * New client password
	 */
	private String newPassword = null;

	/**
	 * Default constructor for <code>EPPLoginSec</code>.
	 */
	public EPPLoginSec() {
	}

	/**
	 * <code>EPPLoginSec</code> constructor that takes the client user agent.
	 * 
	 * @param aUserAgent
	 *           Client user agent
	 */
	public EPPLoginSec(EPPLoginSecUserAgent aUserAgent) {
		this.userAgent = aUserAgent;
	}

	/**
	 * <code>EPPLoginSec</code> constructor that takes the password.
	 * 
	 * @param aUserAgent
	 *           Client user agent. Set to {@code null} if undefined.
	 * @param aPassword
	 *           Client password
	 */
	public EPPLoginSec(EPPLoginSecUserAgent aUserAgent, String aPassword) {
		this.userAgent = aUserAgent;
		this.password = aPassword;
	}

	/**
	 * <code>EPPLoginSec</code> constructor that takes all attributes (user
	 * agent, password, and new password).
	 * 
	 * @param aUserAgent
	 *           Client user agent. Set to {@code null} if undefined.
	 * @param aPassword
	 *           Client password
	 * @param aNewPassword
	 *           New client password
	 */
	public EPPLoginSec(EPPLoginSecUserAgent aUserAgent, String aPassword, String aNewPassword) {
		this.userAgent = aUserAgent;
		this.password = aPassword;
		this.newPassword = aNewPassword;
	}

	/**
	 * Is a user agent defined?
	 *
	 * @return <code>true</code> if the user agent is defined; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasUserAgent() {
		if (this.userAgent != null) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Gets the user agent.
	 *
	 * @return User agent if defined; {@code null} otherwise.
	 */
	public EPPLoginSecUserAgent getUserAgent() {
		return this.userAgent;
	}

	/**
	 * Sets the user agent.
	 *
	 * @param aUserAgent
	 *           User agent. Set to {@code null} to unset the user agent.
	 */
	public void setUserAgent(EPPLoginSecUserAgent aUserAgent) {
		this.userAgent = aUserAgent;
	}

	/**
	 * Is a password defined?
	 *
	 * @return <code>true</code> if the password is defined; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasPassword() {
		if (this.password != null) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Gets the client password.
	 *
	 * @return Client password if defined; {@code null} otherwise.
	 */
	public String getPassword() {
		return this.password;
	}

	/**
	 * Sets the client password.
	 *
	 * @param aPassword
	 *           Client password.
	 */
	public void setPassword(String aPassword) {
		this.password = aPassword;
	}

	/**
	 * Is a new password defined?
	 *
	 * @return <code>true</code> if the new password is defined;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasNewPassword() {
		if (this.newPassword != null) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Gets the new client password.
	 *
	 * @return New client password if defined; {@code null} otherwise.
	 */
	public String getNewPassword() {
		return this.newPassword;
	}

	/**
	 * Sets the new client password.
	 *
	 * @param aNewPassword
	 *           New client password.
	 */
	public void setNewPassword(String aNewPassword) {
		this.newPassword = aNewPassword;
	}

	/**
	 * Encode instance into a DOM element tree. A DOM Document is passed as an
	 * argument and functions as a factory for DOM objects. The root element
	 * associated with the instance is created and each instance attribute is
	 * appended as a child node.
	 *
	 * @param aDocument
	 *           DOM Document, which acts is an Element factory
	 *
	 * @return Element Root element associated with the object
	 *
	 * @exception EPPEncodeException
	 *               Error encoding <code>EPPLoginSec</code>
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {

		if (aDocument == null) {
			throw new EPPEncodeException("aDocument is null" + " on in EPPLoginSec.encode(Document)");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPLoginSecExtFactory.NS, ELM_NAME);

		// User Agent
		EPPUtil.encodeComp(aDocument, root, this.userAgent);

		// Password
		EPPUtil.encodeString(aDocument, root, this.password, EPPLoginSecExtFactory.NS,
		      EPPLoginSecExtFactory.NS_PREFIX + ":" + ELM_PASSWORD);

		// New Password
		EPPUtil.encodeString(aDocument, root, this.newPassword, EPPLoginSecExtFactory.NS,
		      EPPLoginSecExtFactory.NS_PREFIX + ":" + ELM_NEW_PASSWORD);

		return root;
	}

	/**
	 * Decode a DOM element tree to initialize the instance attributes. The
	 * <code>aElement</code> argument represents the root DOM element and is used
	 * to traverse the DOM nodes for instance attribute values.
	 *
	 * @param aElement
	 *           <code>Element</code> to decode
	 *
	 * @exception EPPDecodeException
	 *               Error decoding <code>Element</code>
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {

		// User Agent
		this.userAgent = (EPPLoginSecUserAgent) EPPUtil.decodeComp(aElement, EPPLoginSecExtFactory.NS,
		      EPPLoginSecUserAgent.ELM_NAME, EPPLoginSecUserAgent.class);

		// Password
		this.password = EPPUtil.decodeString(aElement, EPPLoginSecExtFactory.NS, ELM_PASSWORD);

		// New Password
		this.newPassword = EPPUtil.decodeString(aElement, EPPLoginSecExtFactory.NS, ELM_NEW_PASSWORD);
	}

	/**
	 * Compare an instance of <code>EPPLoginSec</code> with this instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return <code>true</code> if equal; <code>false</code> otherwise.
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPLoginSec)) {
			cat.error("EPPLoginSec.equals(): " + aObject.getClass().getName() + " not EPPLoginSec instance");

			return false;
		}

		EPPLoginSec other = (EPPLoginSec) aObject;

		// User Agent
		if (!EqualityUtil.equals(this.userAgent, other.userAgent)) {
			cat.error("EPPLoginSec.equals(): userAgent not equal");
			return false;
		}

		// Password
		if (!EqualityUtil.equals(this.password, other.password)) {
			cat.error("EPPLoginSec.equals(): password not equal");
			return false;
		}

		// New Password
		if (!EqualityUtil.equals(this.newPassword, other.newPassword)) {
			cat.error("EPPLoginSec.equals(): newPassword not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone an <code>EPPCodecComponent</code> instance.
	 *
	 * @return clone of concrete <code>EPPLoginSec</code>
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {

		EPPLoginSec clone = (EPPLoginSec) super.clone();

		// Period
		if (this.userAgent != null) {
			clone.userAgent = (EPPLoginSecUserAgent) this.userAgent.clone();
		}

		return clone;
	}

	/**
	 * Implementation of <code>Object.toString</code>, which will result in an
	 * indented XML <code>String</code> representation of the concrete
	 * <code>EPPCodecComponent</code>.
	 *
	 * @return Indented XML <code>String</code> if successful; <code>ERROR</code>
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Returns the XML namespace associated with the
	 * <code>EPPCodecComponent</code>.
	 * 
	 * @return XML namespace for the <code>EPPCodecComponent</code>.
	 */
	@Override
	public String getNamespace() {
		return EPPLoginSecExtFactory.NS;
	}

}
