/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.launchpolicy.v01;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.codec.registry.policy.EPPRegistryZoneInterface;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPLaunchPolicyZone} contains all of the policy information associated
 * with the zone.
 */
public class EPPLaunchPolicyZone implements EPPCodecComponent, EPPRegistryZoneInterface {

	/**
	 * XML local name for {@code EPPLaunchPolicyZone}.
	 */
	public static final String ELM_LOCALNAME = "zone";

	/**
	 * XML root tag for {@code EPPLaunchPolicyZone}.
	 */
	public static final String ELM_NAME = EPPLaunchPolicyExtFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * Category for logging
	 */
	    private static Logger cat = LoggerFactory.getLogger(EPPLaunchPolicyZone.class);
	      

	/**
	 * Zone phases
	 */
	private List<EPPLaunchPolicyPhase> phases = new ArrayList<EPPLaunchPolicyPhase>();

	/**
	 * Default constructor for {@code EPPLaunchPolicyZone}.
	 */
	public EPPLaunchPolicyZone() {
	}

	/**
	 * Constructor for {@code EPPLaunchPolicyZone} that takes a single phase.
	 *
	 * @param aPhase
	 *           Zone phase
	 */
	public EPPLaunchPolicyZone(EPPLaunchPolicyPhase aPhase) {
		this.addPhase(aPhase);
	}

	/**
	 * Constructor for {@code EPPLaunchPolicyZone} that takes a list of phases.
	 *
	 * @param aPhases
	 *           List of phases
	 */
	public EPPLaunchPolicyZone(List<EPPLaunchPolicyPhase> aPhases) {
		this.setPhases(aPhases);
	}

	/**
	 * Are the phases defined?
	 *
	 * @return {@code true} if the phases are defined; {@code false} otherwise.
	 */
	public boolean hasPhases() {
		return !this.phases.isEmpty();
	}

	/**
	 * Gets the list of phases if defined.
	 *
	 * @return List of phases if defined; empty list otherwise.
	 */
	public List<EPPLaunchPolicyPhase> getPhases() {
		return this.phases;
	}

	/**
	 * Adds a phase to the list of phases.
	 *
	 * @param aPhase
	 *           The phase to add.
	 */
	public void addPhase(EPPLaunchPolicyPhase aPhase) {
		if (aPhase == null) {
			return;
		}

		this.phases.add(aPhase);
	}

	/**
	 * Sets the list of phases.
	 *
	 * @param aPhases
	 *           The phases to set.
	 */
	public void setPhases(List<EPPLaunchPolicyPhase> aPhases) {
		if (aPhases == null) {
			this.phases = new ArrayList<EPPLaunchPolicyPhase>();
		}
		else {
			this.phases = aPhases;
		}
	}

	/**
	 * Encode instance into a DOM element tree. A DOM Document is passed as an
	 * argument and functions as a factory for DOM objects. The root element
	 * associated with the instance is created and each instance attribute is
	 * appended as a child node.
	 *
	 * @param aDocument
	 *           DOM Document, which acts is an Element factory
	 *
	 * @return Element Root element associated with the object
	 *
	 * @exception EPPEncodeException
	 *               Error encoding {@code EPPLaunchPolicyZone}
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {

		if (aDocument == null) {
			throw new EPPEncodeException("aDocument is null" + " in EPPLaunchPolicyZone.encode(Document)");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPLaunchPolicyExtFactory.NS, ELM_NAME);

		// Phases
		EPPUtil.encodeCompList(aDocument, root, this.phases);

		return root;
	}

	/**
	 * Decode a DOM element tree to initialize the instance attributes. The
	 * {@code aElement} argument represents the root DOM element and is used to
	 * traverse the DOM nodes for instance attribute values.
	 *
	 * @param aElement
	 *           {@code Element} to decode
	 *
	 * @exception EPPDecodeException
	 *               Error decoding {@code Element}
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {

		// Phases
		this.phases = EPPUtil.decodeCompList(aElement, EPPLaunchPolicyExtFactory.NS, EPPLaunchPolicyPhase.ELM_LOCALNAME,
		      EPPLaunchPolicyPhase.class);
	}

	/**
	 * Clone an {@code EPPCodecComponent} instance.
	 *
	 * @return clone of concrete {@code EPPLaunchPolicyZone}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {

		EPPLaunchPolicyZone clone = (EPPLaunchPolicyZone) super.clone();

		// Phases
		clone.phases = new ArrayList<EPPLaunchPolicyPhase>();
		for (EPPLaunchPolicyPhase phase : this.phases) {
			clone.phases.add((EPPLaunchPolicyPhase) phase.clone());
		}

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Compare an instance of {@code EPPLaunchPolicyZone} with this instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return {@code true} if equal; {@code false} otherwise.
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPLaunchPolicyZone)) {
			cat.error(
			      "EPPLaunchPolicyZone.equals(): " + aObject.getClass().getName() + " not EPPLaunchPolicyZone instance");

			return false;
		}

		EPPLaunchPolicyZone other = (EPPLaunchPolicyZone) aObject;

		// Phases
		if (!EqualityUtil.equals(this.phases, other.phases)) {
			cat.error("EPPLaunchPolicyZone.equals(): phases not equal");
			return false;
		}

		return true;
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPLaunchPolicyExtFactory.NS;
	}

}
