/***********************************************************
Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.gen;


import java.util.Set;

import org.w3c.dom.Element;


/**
 * Represents an abstract class, used by {@code EPPFactory}, and extended
 * by concrete EPP Command Mappings for the creation of concrete
 * {@code EPPCommand} and     {@code EPPResponse} instances from a
 * command extension XML element.  Each {@code EPPMapFactory}     is
 * associated with an XML namespace, and is delegated the responsibility by
 * {@code EPPFactory}     of creating concrete {@code EPPCommand}
 * and {@code EPPResponse} objects.<br>
 * <br>
 * For example, the Domain Command Mapping has an XML namespace of
 * "urn:iana:xmlns:domain"     and has a concrete {@code EPPMapFactory}
 * named {@code EPPDomainMapFactory} that will create     all of the
 * domain package objects.
 *
 * @see com.verisign.epp.codec.gen.EPPFactory
 */
public abstract class EPPMapFactory {
	/** Command type constant associated with the &lt;create&gt; command. */
	public static final String EPP_CREATE = "create";

	/** Command type constant associated with the &lt;delete&gt; command. */
	public static final String EPP_DELETE = "delete";

	/** Command type constant associated with the &lt;info&gt; command. */
	public static final String EPP_INFO = "info";

	/** Command type constant associated with the &lt;login&gt; command. */
	public static final String EPP_LOGIN = "login";

	/** Command type constant associated with the &lt;logout&gt; command. */
	public static final String EPP_LOGOUT = "logout";

	/** Command type constant associated with the &lt;ping&gt; command. */
	public static final String EPP_CHECK = "check";

	/** Command type constant associated with the &lt;renew&gt; command. */
	public static final String EPP_RENEW = "renew";

	/** Command type constant associated with the &lt;transfer&gt; command. */
	public static final String EPP_TRANSFER = "transfer";

	/** Command type constant associated with the &lt;update&gt; command. */
	public static final String EPP_UPDATE = "update";

	/**
	 * Abstract method that creates a concrete {@code EPPCommand} from an
	 * XML element.      For example, a {@code EPPDomainCreateCmd} will
	 * be created by {@code EPPDomainMapFactory}     given the
	 * &lt;domain:create&gt; XML element.
	 *
	 * @param aMapElement command extension XML element.  For example
	 * 		  &lt;domain:create&gt;.
	 *
	 * @return Concrete {@code EPPCommand} associated with command
	 * 		   extension XML element.      For example,
	 * 		   {@code EPPDomainCreateCmd}.
	 *
	 * @exception EPPCodecException Error creating the concrete
	 * 			  {@code EPPCommand}
	 */
	public abstract EPPCommand createCommand(Element aMapElement)
									  throws EPPCodecException;

	/**
	 * Abstract method that creates a concrete {@code EPPResponse} from an
	 * XML element.      For example, a {@code EPPDomainInfoResp} will be
	 * created by {@code EPPDomainMapFactory}     given the
	 * &lt;domain:info-data&gt; XML element.
	 *
	 * @param aMapElement command extension XML element.  For example
	 * 		  &lt;domain:info-data&gt;.
	 *
	 * @return Concrete {@code EPPResponse} associated with command
	 * 		   extension XML element.      For example,
	 * 		   {@code EPPDomainInfoResp}.
	 *
	 * @exception EPPCodecException Error creating the concrete
	 * 			  {@code EPPResponse}
	 */
	public abstract EPPResponse createResponse(Element aMapElement)
										throws EPPCodecException;

	/**
	 * Abstract method that gets the service information associated with the
	 * concrete {@code EPPMapFactory}.  The service information is used
	 * by     {@code EPPFactory} for extracting the XML namespace
	 * associated with the service.
	 *
	 * @return service description associated with the concret
	 * 		   {@code EPPMapFactory}.
	 */
	abstract public EPPService getService();
	
	
	/**
	 * Gets the list of XML schemas that need to be pre-loaded into the 
	 * XML Parser.	 
	 * 
	 * @return {@code Set} of {@code String} XML Schema names that 
	 * should be pre-loaded in the XML Parser.  Defaults to returning {@code null} 
	 * to indicate that no XML schemas need to be loaded.
	 */
	public Set getXmlSchemas() {
		return null;
	}
}
