/***********************************************************
Copyright (C) 2017 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.fee.v1_0;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EqualityUtil;

/**
 * <code>EPPFeeCheckResult</code> represents a fee result to a check.
 */
public class EPPFeeCheckResult implements EPPCodecComponent {

	/**
	 * XML local name for <code>EPPFeeCheckResult</code>.
	 */
	public static final String ELM_LOCALNAME = "cd";

	/**
	 * XML root tag for <code>EPPFeeCheckResult</code>.
	 */
	public static final String ELM_NAME = EPPFeeExtFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * Category for logging
	 */
	    private static Logger cat = LoggerFactory.getLogger(EPPFeeCheckResult.class);
	      

	/**
	 * XML local name for the object
	 */
	private static final String ELM_OBJ_ID = "objID";

	/**
	 * XML local name for the classification
	 */
	private static final String ELM_CLASSIFICATION = "class";
	
	/**
	 * XML local name for the reason
	 */
	private static final String ELM_REASON = "reason";

	/**
	 * &quot;avail&quot; attribute constant.
	 */
	private final static String ATTR_AVAIL = "avail";

	/**
	 * XML attribute name for the <code>fee:reason</code> attribute.
	 */
	private final static String ATTR_LANG = "lang";

	/**
	 * Default XML attribute value for the reason language.
	 */
	private final static String DEFAULT_LANG = "en";

	/**
	 * The string that identifies the object, which most likely corresponds to
	 * the &lt;domain:name&gt; element.
	 */
	private String objID;

	/**
	 * OPTIONAL classification of the domain name.
	 */
	private String classification;
	
	/**
	 * Is the fee information available?
	 */
	private boolean available;

	/**
	 * Commands fee data
	 */
	private List<EPPFeeCommandData> commands = new ArrayList<EPPFeeCommandData>();

	/**
	 * XML attribute value for the <code>lang</code> attribute.
	 */
	private String language = DEFAULT_LANG;

	/**
	 * Reason for a &quot;false&quot; availability.
	 */
	private String reason;

	/**
	 * Default constructor for <code>EPPFeeCheckResult</code>.
	 */
	public EPPFeeCheckResult() {
	}

	/**
	 * Constructor for <code>EPPFeeCheckResult</code> that only takes the
	 * identifier (domain name) and the available flag.
	 * 
	 * @param aObjID
	 *           Object identifier. An example is the domain name for extension
	 *           of a domain check response.
	 * @param aAvailable
	 *           Fee available flag. If <code>true</code>, at least one fee
	 *           command data element must be set and if <code>false</code> a
	 *           reason should be set.
	 */
	public EPPFeeCheckResult(String aObjID, boolean aAvailable) {
		this.setObjID(aObjID);
		this.setAvailable(aAvailable);
	}

	/**
	 * Constructor for <code>EPPFeeCheckResult</code> that takes the identifier
	 * (domain name) and a single command for an available fee.
	 * 
	 * @param aObjID
	 *           Object identifier. An example is the domain name for extension
	 *           of a domain check response.
	 * @param aCommand
	 *           Command fee data
	 */
	public EPPFeeCheckResult(String aObjID, EPPFeeCommandData aCommand) {
		this.setObjID(aObjID);
		this.setAvailable(true);
		this.addCommand(aCommand);
	}

	/**
	 * Constructor for <code>EPPFeeCheckResult</code> that takes the object
	 * identifier (domain name) and a list of commands for an available fee.
	 * 
	 * @param aObjID
	 *           Object identifier. An example is the domain name for extension
	 *           of a domain check response.
	 * @param aCommands
	 *           Command fee data
	 */
	public EPPFeeCheckResult(String aObjID, List<EPPFeeCommandData> aCommands) {
		this.setObjID(aObjID);
		this.setAvailable(true);
		this.setCommands(aCommands);
	}

	/**
	 * Constructor for <code>EPPFeeCheckResult</code> that takes the object
	 * identifier (domain name) and the reason that the fee data is unavailable.
	 * 
	 * @param aObjID
	 *           Object identifier. An example is the domain name for extension
	 *           of a domain check response.
	 * @param aReason
	 *           Reason that the fee data is unavailable.
	 */
	public EPPFeeCheckResult(String aObjID, String aReason) {
		this.setObjID(aObjID);
		this.setAvailable(false);
		this.setReason(aReason);
	}

	/**
	 * Constructor for <code>EPPFeeCheckResult</code> that takes the object
	 * identifier (domain name), the reason that the fee data is unavailable, and
	 * the language of the reason.
	 * 
	 * @param aObjID
	 *           Object identifier. An example is the domain name for extension
	 *           of a domain check response.
	 * @param aReason
	 *           Reason that the fee data is unavailable.
	 * @param aLanguage
	 *           Language of the <code>aReason</code> value.
	 */
	public EPPFeeCheckResult(String aObjID, String aReason, String aLanguage) {
		this(aObjID, aReason);
		this.setLanguage(aLanguage);
	}
	
	/**
	 * Constructor for <code>EPPFeeCheckResult</code> that takes all attributes.
	 * 
	 * @param aObjID
	 *           Object identifier. An example is the domain name for extension
	 *           of a domain check response.
	 * @param aClassification Fee classification for object.  Set to <code>null</code> for no classification.           
	 * @param aReason
	 *           Reason that the fee data is unavailable.
	 * @param aLanguage
	 *           Language of the <code>aReason</code> value.
	 */
	public EPPFeeCheckResult(String aObjID, String aClassification, String aReason, String aLanguage) {
		this(aObjID, aReason);
		this.classification = aClassification;		
		this.setLanguage(aLanguage);
	}
	

	/**
	 * Gets the object identifier value.
	 * 
	 * @return Object identifier value if defined; <code>null</code> otherwise.
	 */
	public String getObjID() {
		return this.objID;
	}

	/**
	 * Sets the name value.
	 * 
	 * @param aObjID
	 *           Object identifier value
	 */
	public void setObjID(String aObjID) {
		this.objID = aObjID;
	}

	/**
	 * Is the classification defined?
	 * 
	 * @return <code>true</code> if the classification is defined;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasClassification() {
		return (this.classification != null ? true : false);
	}

	/**
	 * Gets the classification.
	 * 
	 * @return The classification value if defined; <code>null</code> otherwise.
	 */
	public String getClassification() {
		return this.classification;
	}

	/**
	 * Sets the classification.
	 * 
	 * @param aClassification
	 *           the classification to set
	 */
	public void setClassification(String aClassification) {
		this.classification = aClassification;
	}
	
	/**
	 * Is the fee information available? See {@link #getReason()} to get the
	 * optional reason when available is <code>false</code>.
	 * 
	 * @return <code>true</code> if the fee information is available;
	 *         <code>false</code> otherwise.
	 */
	public boolean isAvailable() {
		return this.available;
	}

	/**
	 * Sets the fee information available value. See {@link #setReason(String)}
	 * to set the optional reason when available is <code>false</code>.
	 * 
	 * @param aAvailable
	 *           <code>true</code> if the fee is available; <code>false</code>
	 *           otherwise.
	 */
	public void setAvailable(boolean aAvailable) {
		this.available = aAvailable;
	}

	/**
	 * Are the commands defined?
	 * 
	 * @return <code>true</code> if the commands are defined; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasCommands() {
		return !this.commands.isEmpty();
	}

	/**
	 * Gets the list of commands if defined.
	 * 
	 * @return List of commands if defined; empty list otherwise.
	 */
	public List<EPPFeeCommandData> getCommands() {
		return this.commands;
	}

	/**
	 * Adds a command to the list of commands.
	 * 
	 * @param aCommand
	 *           The command to add.
	 */
	public void addCommand(EPPFeeCommandData aCommand) {
		if (aCommand == null)
			return;

		this.commands.add(aCommand);
	}

	/**
	 * Sets the list of commands.
	 * 
	 * @param aCommands
	 *           The commands to set.
	 */
	public void setCommands(List<EPPFeeCommandData> aCommands) {
		if (aCommands == null)
			this.commands = new ArrayList<EPPFeeCommandData>();
		else
			this.commands = aCommands;
	}

	/**
	 * Is the reason defined if the available attribute is <code>false</code>?
	 * 
	 * @return <code>true</code> if the reason is defined; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasReason() {
		return (this.reason != null ? true : false);
	}

	/**
	 * Gets the reason value.
	 * 
	 * @return Reason if defined; <code>null</code> otherwise.
	 */
	public String getReason() {
		return this.reason;
	}

	/**
	 * Sets the reason value.
	 * 
	 * @param aReason
	 *           reason value
	 */
	public void setReason(String aReason) {
		this.reason = aReason;
	}

	/**
	 * Gets reason language
	 *
	 * @return Reason language
	 */
	public String getLanguage() {
		return this.language;
	}

	/**
	 * Sets language attribute.
	 *
	 * @param aLanguage
	 *           Sets reason language attribute.
	 */
	public void setLanguage(String aLanguage) {
		if (aLanguage == null) {
			this.language = DEFAULT_LANG;
		}
		else {
			this.language = aLanguage;			
		}
	}

	/**
	 * Encode instance into a DOM element tree. A DOM Document is passed as an
	 * argument and functions as a factory for DOM objects. The root element
	 * associated with the instance is created and each instance attribute is
	 * appended as a child node.
	 * 
	 * @param aDocument
	 *           DOM Document, which acts is an Element factory
	 * 
	 * @return Element Root element associated with the object
	 * 
	 * @exception EPPEncodeException
	 *               Error encoding <code>EPPFeeCheckResult</code>
	 */
	public Element encode(Document aDocument) throws EPPEncodeException {

		if (aDocument == null) {
			throw new EPPEncodeException("aDocument is null" + " on in EPPFeeCheckResult.encode(Document)");
		}

		// Check for required attributes
		if (this.objID == null) {
			throw new EPPEncodeException("Required objID attribute is null in EPPFeeCheckResult.encode(Document).");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPFeeExtFactory.NS, ELM_NAME);

		// Available
		EPPUtil.encodeBooleanAttr(root, ATTR_AVAIL, this.available);

		// Object ID
		EPPUtil.encodeString(aDocument, root, this.objID, EPPFeeExtFactory.NS,
		      EPPFeeExtFactory.NS_PREFIX + ":" + ELM_OBJ_ID);
		
		// Classification
		EPPUtil.encodeString(aDocument, root, this.classification, EPPFeeExtFactory.NS,
		      EPPFeeExtFactory.NS_PREFIX + ":" + ELM_CLASSIFICATION);		

		// Commands
		EPPUtil.encodeCompList(aDocument, root, this.commands);

		// Reason
		if (this.hasReason()) {
			Element reasonElm = aDocument.createElementNS(EPPFeeExtFactory.NS,
			      EPPFeeExtFactory.NS_PREFIX + ":" + ELM_REASON);
			root.appendChild(reasonElm);

			// Language
			if (!language.equals(DEFAULT_LANG)) {
				reasonElm.setAttribute(ATTR_LANG, this.language);
			}

			// Reason
			Text aReason = aDocument.createTextNode(this.reason);
			reasonElm.appendChild(aReason);
		}

		return root;
	}

	/**
	 * Decode a DOM element tree to initialize the instance attributes. The
	 * <code>aElement</code> argument represents the root DOM element and is used
	 * to traverse the DOM nodes for instance attribute values.
	 * 
	 * @param aElement
	 *           <code>Element</code> to decode
	 * 
	 * @exception EPPDecodeException
	 *               Error decoding <code>Element</code>
	 */
	public void decode(Element aElement) throws EPPDecodeException {

		// Available
		this.available = EPPUtil.decodeBooleanAttr(aElement, ATTR_AVAIL);

		// Object ID
		this.objID = EPPUtil.decodeString(aElement, EPPFeeExtFactory.NS, ELM_OBJ_ID);

		// Classification
		this.classification = EPPUtil.decodeString(aElement, EPPFeeExtFactory.NS, ELM_CLASSIFICATION);
		
		// Commands
		this.commands = EPPUtil.decodeCompList(aElement, EPPFeeExtFactory.NS, EPPFeeCommandData.ELM_LOCALNAME,
		      EPPFeeCommandData.class);

		// Reason
		Element theReasonElm = EPPUtil.getElementByTagNameNS(aElement, EPPFeeExtFactory.NS, ELM_REASON);

		if (theReasonElm != null) {
			this.reason = theReasonElm.getFirstChild().getNodeValue();

			String theLang = theReasonElm.getAttribute(ATTR_LANG);

			if (theLang.length() > 0) {
				this.language = theLang;
			}
		}

		// Reason
		this.reason = EPPUtil.decodeString(aElement, EPPFeeExtFactory.NS, ELM_REASON);

	}

	/**
	 * Clone an <code>EPPCodecComponent</code> instance.
	 * 
	 * @return clone of concrete <code>EPPFeeCheckResult</code>
	 * 
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPFeeCheckResult clone = (EPPFeeCheckResult) super.clone();

		// Available
		clone.available = this.available;

		// Object ID
		clone.objID = this.objID;

		// Classification
		clone.classification = this.classification;
		
		// Commands
		clone.commands = new ArrayList<EPPFeeCommandData>();
		for (EPPFeeCommandData command : this.commands) {
			clone.commands.add((EPPFeeCommandData) command.clone());
		}

		// Reason
		clone.reason = this.reason;

		// Language
		clone.language = this.language;

		return clone;
	}

	/**
	 * Implementation of <code>Object.toString</code>, which will result in an
	 * indented XML <code>String</code> representation of the concrete
	 * <code>EPPCodecComponent</code>.
	 * 
	 * @return Indented XML <code>String</code> if successful; <code>ERROR</code>
	 *         otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Compare an instance of <code>EPPFeeCheckResult</code> with this instance.
	 * 
	 * @param aObject
	 *           Object to compare with.
	 * 
	 * @return <code>true</code> if equal; <code>false</code> otherwise.
	 */
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPFeeCheckResult)) {
			cat.error("EPPFeeCheckResult.equals(): " + aObject.getClass().getName() + " not EPPFeeCheckResult instance");

			return false;
		}

		EPPFeeCheckResult other = (EPPFeeCheckResult) aObject;

		// Available
		if (!EqualityUtil.equals(this.available, other.available)) {
			cat.error("EPPFeeCheckResult.equals(): available not equal");
			return false;
		}

		// Object ID
		if (!EqualityUtil.equals(this.objID, other.objID)) {
			cat.error("EPPFeeCheckResult.equals(): objID not equal");
			return false;
		}

		// Classification
		if (!EqualityUtil.equals(this.classification, other.classification)) {
			cat.error("EPPFeeCheckResult.equals(): classification not equal");
			return false;
		}
		
		// Commands
		if (!EqualityUtil.equals(this.commands, other.commands)) {
			cat.error("EPPFeeCheckResult.equals(): commands not equal");
			return false;
		}

		// Reason
		if (!EqualityUtil.equals(this.reason, other.reason)) {
			cat.error("EPPFeeCheckResult.equals(): reason not equal");
			return false;
		}

		// Language
		if (!EqualityUtil.equals(this.language, other.language)) {
			cat.error("EPPFeeCheckResult.equals(): language not equal");
			return false;
		}

		return true;
	}

	/**
	 * Returns the XML namespace associated with the <code>EPPCodecComponent</code>.
	 * 
	 * @return XML namespace for the <code>EPPCodecComponent</code>. 
	 */
	@Override
	public String getNamespace() {
		return EPPFeeExtFactory.NS;
	}

}
