/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * The default, minimum, and maximum period length for the command type. The
 * &lt;registry:length&gt; element contains the following child elements, where
 * all of the child elements require the "unit" attribute with possible values
 * of "y" for year and "m" for month
 */
public class EPPRegistryMinMaxPeriod implements EPPCodecComponent {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryMinMaxPeriod.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for {@code EPPRegistryMinMaxPeriod}.
	 */
	public static final String ELM_LOCALNAME = "length";

	/**
	 * XML root tag for {@code EPPRegistryMinMaxPeriod}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * The minimum supported period length.
	 */
	private EPPRegistryMinPeriodType min;

	/**
	 * The maximum supported period length.
	 */
	private EPPRegistryMaxPeriodType max;

	/**
	 * The default period length if not defined by the client.
	 */
	private EPPRegistryDefaultPeriodType defaultLength;

	public EPPRegistryMinMaxPeriod() {
	}

	/**
	 * Constructor for {@code EPPRegistryMinMaxPeriod} that takes all of the
	 * attribute values as parameters.
	 *
	 * @param aMinNumber
	 *           The minimum supported period length number.
	 * @param aMinUnit
	 *           The minimum supported period length unit.
	 * @param aMaxNumber
	 *           The maximum supported period length number.
	 * @param aMaxUnit
	 *           The maximum supported period length unit.
	 * @param aDefaultNumber
	 *           The default period length number if not defined by the client.
	 * @param aDefaultUnit
	 *           The default period length unit if not defined by the client.
	 */
	public EPPRegistryMinMaxPeriod(Integer aMinNumber, EPPRegistryPeriodType.Unit aMinUnit, Integer aMaxNumber,
	      EPPRegistryPeriodType.Unit aMaxUnit, Integer aDefaultNumber, EPPRegistryPeriodType.Unit aDefaultUnit) {
		this.min = new EPPRegistryMinPeriodType(aMinNumber, aMinUnit);
		this.max = new EPPRegistryMaxPeriodType(aMaxNumber, aMaxUnit);
		this.defaultLength = new EPPRegistryDefaultPeriodType(aDefaultNumber, aDefaultUnit);
	}

	/**
	 * Constructor for {@code EPPRegistryMinMaxPeriod} that takes all of the
	 * attribute values as parameters.
	 *
	 * @param aMinNumber
	 *           The minimum supported period length number.
	 * @param aMinUnit
	 *           The minimum supported period length unit.
	 * @param aMaxNumber
	 *           The maximum supported period length number.
	 * @param aMaxUnit
	 *           The maximum supported period length unit.
	 * @param aDefaultNumber
	 *           The default period length number if not defined by the client.
	 * @param aDefaultUnit
	 *           The default period length unit if not defined by the client.
	 */
	public EPPRegistryMinMaxPeriod(int aMinNumber, EPPRegistryPeriodType.Unit aMinUnit, int aMaxNumber,
	      EPPRegistryPeriodType.Unit aMaxUnit, int aDefaultNumber, EPPRegistryPeriodType.Unit aDefaultUnit) {
		this.min = new EPPRegistryMinPeriodType(aMinNumber, aMinUnit);
		this.max = new EPPRegistryMaxPeriodType(aMaxNumber, aMaxUnit);
		this.defaultLength = new EPPRegistryDefaultPeriodType(aDefaultNumber, aDefaultUnit);
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryMinMaxPeriod} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPRegistryMinMaxPeriod} instance.
	 *
	 * @exception EPPEncodeException
	 *               - Unable to encode {@code EPPRegistryMinMaxPeriod} instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		try {
			validateState();
		}
		catch (EPPCodecException e) {
			throw new EPPEncodeException("Invalid state on EPPRegistryMinMaxPeriod.encode: " + e);
		}

		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

		// Min
		EPPUtil.encodeComp(aDocument, root, this.min);

		// Max
		EPPUtil.encodeComp(aDocument, root, this.max);

		// Default
		EPPUtil.encodeComp(aDocument, root, this.defaultLength);

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryMinMaxPeriod} attributes from the aElement
	 * DOM Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryMinMaxPeriod} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {
		// Min
		this.min = (EPPRegistryMinPeriodType) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS,
		      EPPRegistryMinPeriodType.ELM_LOCALNAME, EPPRegistryMinPeriodType.class);

		// Max
		this.max = (EPPRegistryMaxPeriodType) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS,
		      EPPRegistryMaxPeriodType.ELM_LOCALNAME, EPPRegistryMaxPeriodType.class);

		// Default
		this.defaultLength = (EPPRegistryDefaultPeriodType) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS,
		      EPPRegistryDefaultPeriodType.ELM_LOCALNAME, EPPRegistryDefaultPeriodType.class);
	}

	/**
	 * Clone {@code EPPRegistryMinMaxPeriod}.
	 *
	 * @return clone of {@code EPPRegistryMinMaxPeriod}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPRegistryMinMaxPeriod clone = (EPPRegistryMinMaxPeriod) super.clone();

		// Min
		if (this.min != null) {
			clone.min = (EPPRegistryMinPeriodType) this.min.clone();
		}

		// Max
		if (this.max != null) {
			clone.max = (EPPRegistryMaxPeriodType) this.max.clone();
		}

		// Default
		if (this.defaultLength != null) {
			clone.defaultLength = (EPPRegistryDefaultPeriodType) this.defaultLength.clone();
		}

		return clone;
	}

	/**
	 * implements a deep {@code EPPRegistryMinMaxPeriod} compare.
	 *
	 * @param aObject
	 *           {@code EPPRegistryMinMaxPeriod} instance to compare with
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryMinMaxPeriod)) {
			return false;
		}

		EPPRegistryMinMaxPeriod theComp = (EPPRegistryMinMaxPeriod) aObject;

		// Min
		if (!EqualityUtil.equals(this.min, theComp.min)) {
			cat.error("EPPRegistryMinMaxPeriod.equals(): min not equal");
			return false;
		}

		// Max
		if (!EqualityUtil.equals(this.max, theComp.max)) {
			cat.error("EPPRegistryMinMaxPeriod.equals(): max not equal");
			return false;
		}

		// Default
		if (!EqualityUtil.equals(this.defaultLength, theComp.defaultLength)) {
			cat.error("EPPRegistryMinMaxPeriod.equals(): defaultLength not equal");
			return false;
		}

		return true;
	}

	void validateState() throws EPPCodecException {
		// Min
		if (this.min == null) {
			throw new EPPCodecException("min element is not set");
		}
		this.min.validateState();

		// Max
		if (this.max == null) {
			throw new EPPCodecException("max element is not set");
		}
		this.max.validateState();

		// Default
		if (this.defaultLength == null) {
			throw new EPPCodecException("defaultLength element is not set");
		}
		this.defaultLength.validateState();
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Gets the minimum supported period length.
	 *
	 * @return Minimum supported period length if defined; {@code null}
	 *         otherwise.
	 */
	public EPPRegistryMinPeriodType getMin() {
		return this.min;
	}

	/**
	 * Sets the minimum supported period length.
	 *
	 * @param aMin
	 *           Minimum supported period length
	 */
	public void setMin(EPPRegistryMinPeriodType aMin) {
		this.min = aMin;
	}

	/**
	 * Gets the maximum supported period length.
	 *
	 * @return Maximum supported period length if defined; {@code null}
	 *         otherwise.
	 */
	public EPPRegistryMaxPeriodType getMax() {
		return this.max;
	}

	/**
	 * Sets the maximum supported period length.
	 *
	 * @param aMax
	 *           Maximum supported period length
	 */
	public void setMax(EPPRegistryMaxPeriodType aMax) {
		this.max = aMax;
	}

	/**
	 * Gets the default supported period length.
	 *
	 * @return Default supported period length if defined; {@code null}
	 *         otherwise.
	 */
	public EPPRegistryDefaultPeriodType getDefaultLength() {
		return this.defaultLength;
	}

	/**
	 * Sets the default supported period length.
	 *
	 * @param aDefault
	 *           Default supported period length
	 */
	public void setDefaultLength(EPPRegistryDefaultPeriodType aDefault) {
		this.defaultLength = aDefault;
	}

	/**
	 * Returns the XML namespace associated with the
	 * <code>EPPCodecComponent</code>.
	 *
	 * @return XML namespace for the <code>EPPCodecComponent</code>.
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

}
