/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v02;

import java.util.HashSet;
import java.util.Set;

import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPCommand;
import com.verisign.epp.codec.gen.EPPMapFactory;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPService;
import com.verisign.epp.codec.gen.EPPUtil;

/**
 * Concrete {@link EPPMapFactory} used to create {@link EPPCommand} and
 * {@link EPPResponse} objects from DOM elements for the
 * {@link EPPRegistryMapFactory#NS}.
 */
public class EPPRegistryMapFactory extends EPPMapFactory {
	/**
	 * Namespace URI associated with {@code EPPRegistryMapFactory}.
	 */
	public static final String NS = "urn:ietf:params:xml:ns:epp:registry-0.2";

	/**
	 * Namespace prefix associated with {@code EPPRegistryMapFactory}.
	 */
	public static final String NS_PREFIX = "registry";

	/**
	 * Service entry for {@code EPPRegistryMapFactory}.
	 */
	private EPPService service = null;

	/**
	 * Default constructor for {@code EPPRegistryMapFactory} that initializes the
	 * service attribute.
	 */
	public EPPRegistryMapFactory() {
		service = new EPPService(NS_PREFIX, NS);
	}

	/**
	 * Creates a concrete {@link EPPCommand} for the DOM {@link Element}.
	 * 
	 * @param aMapElement
	 *           DOM {@link Element} used to identify the concrete
	 *           {@link EPPCommand} associated with the
	 *           {@code EPPRegistryMapFactory}.
	 * 
	 * @exception EPPCodecException
	 *               Error creating concrete {@link EPPCommand} for the DOM
	 *               {@link Element}.
	 */
	public EPPCommand createCommand(Element aMapElement) throws EPPCodecException {

		String name = aMapElement.getLocalName();

		if (!aMapElement.getNamespaceURI().equals(NS)) {
			throw new EPPCodecException("Invalid mapping type " + name);
		}

		if (name.equals(EPPUtil.getLocalName(EPPRegistryInfoCmd.ELM_NAME))) {
			return new EPPRegistryInfoCmd();
		}
		else if (name.equals(EPPUtil.getLocalName(EPPRegistryCheckCmd.ELM_NAME))) {
			return new EPPRegistryCheckCmd();
		}
		else if (name.equals(EPPUtil.getLocalName(EPPRegistryCreateCmd.ELM_NAME))) {
			return new EPPRegistryCreateCmd();
		}
		else if (name.equals(EPPUtil.getLocalName(EPPRegistryUpdateCmd.ELM_NAME))) {
			return new EPPRegistryUpdateCmd();
		}
		else if (name.equals(EPPUtil.getLocalName(EPPRegistryDeleteCmd.ELM_NAME))) {
			return new EPPRegistryDeleteCmd();
		}
		else {
			throw new EPPCodecException("Invalid command element " + name);
		}
	}

	/**
	 * Creates a concrete {@link EPPResponse} for the DOM {@link Element}.
	 * 
	 * @param aMapElement
	 *           DOM {@link Element} used to identify the concrete
	 *           {@link EPPResponse} associated with the
	 *           {@code EPPRegistryMapFactory}.
	 * 
	 * @exception EPPCodecException
	 *               Error creating concrete {@link EPPResponse} for the DOM
	 *               {@link Element}.
	 */
	public EPPResponse createResponse(Element aMapElement) throws EPPCodecException {
		String name = aMapElement.getLocalName();

		if (!aMapElement.getNamespaceURI().equals(NS)) {
			throw new EPPCodecException("Invalid mapping type " + name);
		}

		if (name.equals(EPPUtil.getLocalName(EPPRegistryInfoResp.ELM_NAME))) {
			return new EPPRegistryInfoResp();
		}
		else if (name.equals(EPPUtil.getLocalName(EPPRegistryCheckResp.ELM_NAME))) {
			return new EPPRegistryCheckResp();
		}
		else if (name.equals(EPPUtil.getLocalName(EPPRegistryCreateResp.ELM_NAME))) {
			return new EPPRegistryCreateResp();
		}
		else {
			throw new EPPCodecException("Invalid response element " + name);
		}
	}

	/**
	 * Gets the <code>EPPService</code> associated with
	 * <code>EPPRegistryMapFactory</code>. The <code>EPPService</code> is used by
	 * <code>EPPFactory</code> for distributing the responsibility of creating
	 * concrete <code>EPPCommand</code> and <code>EPPResponse</code> objects by
	 * XML namespace. The XML namespace is defined in the returned
	 * <code>EPPService</code>.
	 * 
	 * @return service description for the Registry Command Mapping.
	 */
	public EPPService getService() {
		return service;
	}

	/**
	 * Gets the list of XML schemas that need to be pre-loaded into the XML
	 * Parser.
	 * 
	 * @return <code>Set</code> of <code>String</code> XML Schema names that
	 *         should be pre-loaded in the XML Parser.
	 * 
	 * @see com.verisign.epp.codec.gen.EPPMapFactory#getXmlSchemas()
	 */
	public Set getXmlSchemas() {
		Set theSchemas = new HashSet();
		theSchemas.add("registry-0.2.xsd");
		return theSchemas;
	}
}
