/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v01;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * The zone name is an element that includes an optional "form" attribute that
 * defines the form of the zone name as either "aLabel" or "uLabel", with the
 * default value of "aLabel". The "aLabel" form of a zone name contains all
 * ASCII name labels that conform to [RFC0952] and [RFC1123]. The "uLabel" form
 * of a zone name that includes one or more non-ASCII name labels that can be
 * represented as ASCII labels using [RFC5890].
 */
public class EPPRegistryZoneName implements EPPCodecComponent {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryZoneName.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * Possible values for the {@code form} attribute.
	 */
	public static enum Form {

		/**
		 * The "aLabel" form of a zone name contains all ASCII name labels that
		 * conform to [RFC0952] and [RFC1123].
		 */
		aLabel,

		/**
		 * The "uLabel" form of a zone name that includes one or more non-ASCII
		 * name labels that can be represented as ASCII labels using [RFC5890].
		 */
		uLabel
	}

	/**
	 * XML local name that must be set prior to calling
	 * {@link #encode(org.w3c.dom.Document)}.
	 */
	private String elmRootLocalName = ELM_ZONE_NAME;

	/**
	 * Default XML Element Name for {@code name} attribute.
	 */
	public static final String ELM_ZONE_NAME = "name";

	/**
	 * Optional XML Element Name of {@code name} attribute.
	 */
	public static final String ELM_ZONE = "zone";
	
	/**
	 * XML attribute name for the {@code form} attribute.
	 */
	public static final String ATTR_FORM = "form";
	

	/**
	 * Fully qualified name of the zone object
	 */
	private String name;

	/**
	 * The form of the zone name as either {@link Form#aLabel} or
	 * {@link Form#uLabel}, with the default value of {@link Form#aLabel}.
	 */
	private Form form = Form.aLabel;

	/**
	 * Default constructor. Attributes are set to:
	 * <ul>
	 * <li>{@code elmRootLocalName} - {@link #ELM_ZONE_NAME}</li>
	 * <li>{@code name} - null</li>
	 * <li>{@code form} - {@link Form#aLabel}</li>
	 * </ul>
	 *
	 * Use {@link #setName(String)} before calling {@link #encode(Document)}
	 */
	public EPPRegistryZoneName() {
	}

	/**
	 * Construct {@code EPPRegistryZoneName} with name using the default root
	 * label of {@link #ELM_ZONE_NAME} and the default form of
	 * {@link Form#aLabel}.
	 *
	 * @param aName
	 *           fully qualified name of zone object
	 */
	public EPPRegistryZoneName(String aName) {
		this.name = aName;
	}

	/**
	 * Construct {@code EPPRegistryZoneName} with name and form using the default
	 * root label of {@link #ELM_ZONE_NAME}.
	 *
	 * @param aName
	 *           fully qualified name of zone object
	 * @param aForm
	 *           The form of the zone name as either {@link Form#aLabel} or
	 *           {@link Form#uLabel},
	 */
	public EPPRegistryZoneName(String aName, Form aForm) {
		this.name = aName;
		this.form = aForm;
	}

	/**
	 * Validate the state of the {@code EPPRegistryZoneName} instance. A valid
	 * state means that all of the required attributes have been set. If
	 * validateState returns without an exception, the state is valid. If the
	 * state is not valid, the EPPCodecException will contain a description of
	 * the error. throws EPPCodecException State error. This will contain the
	 * name of the attribute that is not valid.
	 *
	 * @throws EPPCodecException
	 *            Invalid state
	 */
	void validateState() throws EPPCodecException {
		if (this.elmRootLocalName == null) {
			throw new EPPCodecException("elmRootLocalName required attribute is not set");
		}
		if (this.name == null) {
			throw new EPPCodecException("name required attribute is not set");
		}
		if (this.form == null) {
			throw new EPPCodecException("form required attribute is not set");
		}
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryZone} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the {@code EPPRegistryZone}
	 *         instance.
	 *
	 * @exception EPPEncodeException
	 *               - Unable to encode {@code EPPRegistryZone} instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		try {
			validateState();
		}
		catch (EPPCodecException e) {
			cat.error(this.getClass().getName() + ".encode(): Invalid state on encode: " + e);
			throw new EPPEncodeException("Invalid state on " + this.getClass().getName() + ".encode(): " + e);
		}

		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS,
		      EPPRegistryMapFactory.NS_PREFIX + ":" + this.elmRootLocalName);
		
		// Name
		Text theNameText = aDocument.createTextNode(this.name);
		root.appendChild(theNameText);
		
		// Form
		root.setAttribute(ATTR_FORM, this.form.toString());
		
		return root;
	}

	/**
	 * Decode the {@code EPPRegistryZone} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryZone} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {
		// Root local name
		this.elmRootLocalName = aElement.getLocalName();
		
		// Name
		this.name = EPPUtil.decodeStringValue(aElement);
		
		// Form
		String theFormStr = aElement.getAttribute(ATTR_FORM);
		if (theFormStr == null) {
			this.form = Form.aLabel;
		}
		else {
			this.form = Form.valueOf(theFormStr);
		}
	}

	/**
	 * implements a deep {@code EPPRegistryZone} compare.
	 *
	 * @param aObject
	 *           {@code EPPRegistryZone} instance to compare with
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryZoneName)) {
			return false;
		}

		EPPRegistryZoneName theComp = (EPPRegistryZoneName) aObject;

		// Root Local Name
		if (!EqualityUtil.equals(this.elmRootLocalName, theComp.elmRootLocalName)) {
			cat.error("EPPRegistryZoneName.equals(): elmRootLocalName not equal");
			return false;
		}
		
		// Name
		if (!EqualityUtil.equals(this.name, theComp.name)) {
			cat.error("EPPRegistryZoneName.equals(): name not equal");
			return false;
		}
		
		// Name
		if (!EqualityUtil.equals(this.form, theComp.form)) {
			cat.error("EPPRegistryZoneName.equals(): form not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone {@code EPPRegistryZone}.
	 *
	 * @return clone of {@code EPPRegistryZone}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPRegistryZoneName clone = (EPPRegistryZoneName) super.clone();

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Gets the root element local name, which should be either
	 * {@link #ELM_ZONE_NAME} or {@link #ELM_ZONE}.
	 *
	 * @return {@link #ELM_ZONE_NAME} or {@link #ELM_ZONE}
	 */
	public String getRootName() {
		return this.elmRootLocalName;
	}

	/**
	 * Sets the root element local name, which should be either
	 * {@link #ELM_ZONE_NAME} or {@link #ELM_ZONE}.
	 * 
	 * @param aRootName
	 *           {@link #ELM_ZONE_NAME} or {@link #ELM_ZONE}
	 */
	public void setRootName(String aRootName) {
		if (aRootName == null) {
			this.elmRootLocalName = ELM_ZONE_NAME;
		}
		else {
			this.elmRootLocalName = aRootName;
		}
	}

	/**
	 * Gets the name of the zone.
	 *
	 * @return fully qualified name of the zone object
	 */
	public String getName() {
		return this.name;
	}

	/**
	 * Sets the name of the zone.
	 *
	 * @param aName
	 *           fully qualified name of the zone object
	 */
	public void setName(String aName) {
		this.name = aName;
	}

	/**
	 * Gets the form of the zone name.
	 *
	 * @return Form of the zone name with the default of {@link Form#aLabel}.
	 */
	public Form getForm() {
		return this.form;
	}

	/**
	 * Sets the form of the zone name.
	 * 
	 * @param aForm
	 *           Form of the zone name.
	 */
	public void setForm(Form aForm) {
		this.form = aForm;
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

}
