/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v01;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPCodecTst;
import com.verisign.epp.codec.gen.EPPEncodeDecodeStats;
import com.verisign.epp.codec.gen.EPPFactory;
import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPTransId;
import com.verisign.epp.codec.registry.v01.EPPRegistryServices.EPPRegistryObjURI;
import com.verisign.epp.codec.registry.v01.EPPRegistryServicesExt.EPPRegistryExtURI;
import com.verisign.epp.codec.registry.v01.EPPRegistrySupportedStatus.Status;
import com.verisign.epp.codec.registry.v01.EPPRegistryZoneName.Form;
import com.verisign.epp.util.TestThread;

import junit.framework.Assert;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

public class EPPRegistryTst extends TestCase {
	/**
	 * Number of unit test iterations to run. This is set in
	 * <code>EPPCodecTst.main</code>
	 */
	static private long numIterations = 1;

	public EPPRegistryTst(String name) {
		super(name);
	}

	/**
	 * Test the registry info command and response for the zone list.
	 */
	public void testRegistryInfoZoneList() {
		EPPCodecTst.printStart("testRegistryInfoZoneList");

		// Zone List Info Command
		EPPRegistryInfoCmd theCommand = new EPPRegistryInfoCmd("ABC-12349", true);

		EPPEncodeDecodeStats commandStats = EPPCodecTst.testEncodeDecode(theCommand);
		System.out.println(commandStats);

		EPPRegistryZoneList zoneList = new EPPRegistryZoneList();
		EPPTransId respTransId = new EPPTransId("ABC-12345", "54322-XYZ");
		EPPRegistryInfoResp theResponse = new EPPRegistryInfoResp(respTransId, zoneList);
		zoneList.addZone(new EPPRegistryZoneSummary("EXAMPLE1", new GregorianCalendar(2012, 10, 1).getTime(),
		      new GregorianCalendar(2012, 10, 15).getTime()));
		zoneList.addZone(new EPPRegistryZoneSummary("EXAMPLE2", new GregorianCalendar(2012, 9, 1).getTime(),
		      new GregorianCalendar(2012, 9, 19).getTime()));

		EPPEncodeDecodeStats responseStats = EPPCodecTst.testEncodeDecode(theResponse);
		System.out.println(responseStats);

		theCommand = new EPPRegistryInfoCmd("ABC-12349", "com");

		commandStats = EPPCodecTst.testEncodeDecode(theCommand);
		System.out.println(commandStats);

		// Test with contact policy information
		respTransId = new EPPTransId(theCommand.getTransId(), "54321-XYZ");
		EPPRegistryZone zoneInfo = createZoneInfo();

		theResponse = new EPPRegistryInfoResp(respTransId, zoneInfo);
		responseStats = EPPCodecTst.testEncodeDecode(theResponse);
		System.out.println(responseStats);

		// Test without contact policy information
		zoneInfo.setContact(null);
		responseStats = EPPCodecTst.testEncodeDecode(theResponse);
		System.out.println(responseStats);

		EPPCodecTst.printEnd("testRegistryInfoZoneList");
	}

	/**
	 * Test the registry info command and response for the zone info.
	 */
	public void testRegistryInfoZoneInfo() {
		EPPCodecTst.printStart("testRegistryInfoZoneInfo");

		EPPRegistryInfoCmd theCommand = new EPPRegistryInfoCmd("ABC-12345", "zone1");

		EPPEncodeDecodeStats commandStats = EPPCodecTst.testEncodeDecode(theCommand);
		System.out.println(commandStats);

		// Test with contact policy information
		EPPTransId theTransId = new EPPTransId("ABC-12345", "54321-XYZ");
		EPPRegistryZone zoneInfo = createZoneInfo();

		EPPRegistryInfoResp theResponse = new EPPRegistryInfoResp(theTransId, zoneInfo);
		EPPEncodeDecodeStats responseStats = EPPCodecTst.testEncodeDecode(theResponse);
		System.out.println(responseStats);

		// Test without contact policy information
		zoneInfo.setContact(null);
		responseStats = EPPCodecTst.testEncodeDecode(theResponse);
		System.out.println(responseStats);

		EPPCodecTst.printEnd("testRegistryInfoZoneInfo");
	}

	/**
	 * Test the registry info command and response for the system info.
	 */
	public void testRegistryInfoSystemInfo() {
		EPPCodecTst.printStart("testRegistryInfoSystemInfo");

		EPPRegistryInfoCmd theCommand = new EPPRegistryInfoCmd();
		theCommand.setTransId("ABC-12345");
		theCommand.setSystem(true);

		EPPEncodeDecodeStats commandStats = EPPCodecTst.testEncodeDecode(theCommand);
		System.out.println(commandStats);

		EPPTransId theTransId = new EPPTransId("ABC-12345", "54322-XYZ");
		EPPRegistrySystemInfo theSystemInfo = new EPPRegistrySystemInfo(Integer.valueOf(200), Integer.valueOf(600000),
		      Integer.valueOf(86400000), Integer.valueOf(10000), Integer.valueOf(10), Integer.valueOf(1000));
		EPPRegistryInfoResp theResponse = new EPPRegistryInfoResp(theTransId, theSystemInfo);
		EPPEncodeDecodeStats responseStats = EPPCodecTst.testEncodeDecode(theResponse);
		System.out.println(responseStats);

		EPPCodecTst.printEnd("testRegistryInfoSystemInfo");
	}

	public void testRegistryCheck() {
		EPPCodecTst.printStart("testRegistryCheck");

		EPPRegistryCheckCmd theCommand = new EPPRegistryCheckCmd("ABC-12345", "com");

		EPPEncodeDecodeStats commandStats = EPPCodecTst.testEncodeDecode(theCommand);
		System.out.println(commandStats);

		theCommand = new EPPRegistryCheckCmd("ABC-12345", "com");
		theCommand.addName("weirdname");

		commandStats = EPPCodecTst.testEncodeDecode(theCommand);
		System.out.println(commandStats);

		// Encode EPPRegistryInfo Response
		EPPRegistryCheckResp theResponse;
		EPPEncodeDecodeStats responseStats;

		EPPTransId respTransId = new EPPTransId(theCommand.getTransId(), "54321-XYZ");
		EPPRegistryCheckResult result = new EPPRegistryCheckResult("com", Boolean.FALSE);
		result.setReason("Already taken");
		theResponse = new EPPRegistryCheckResp(respTransId, result);

		responseStats = EPPCodecTst.testEncodeDecode(theResponse);
		System.out.println(responseStats);

		respTransId = new EPPTransId(theCommand.getTransId(), "54321-XYZ");
		List results = new ArrayList();
		result = new EPPRegistryCheckResult("com", Boolean.FALSE);
		result.setReason("Already taken");
		results.add(result);
		results.add(new EPPRegistryCheckResult("availtld", Boolean.FALSE));

		theResponse = new EPPRegistryCheckResp(respTransId, results);

		responseStats = EPPCodecTst.testEncodeDecode(theResponse);
		System.out.println(responseStats);

		EPPCodecTst.printEnd("testRegistryCheck");
	}

	public void testRegistryCreate() {
		EPPCodecTst.printStart("testRegistryCreate");

		EPPRegistryZone zone = createZoneInfo();
		zone.setName("newtld");

		EPPRegistryCreateCmd theCommand = null;
		theCommand = new EPPRegistryCreateCmd("ABC-12349", zone);

		EPPEncodeDecodeStats commandStats = EPPCodecTst.testEncodeDecode(theCommand);
		System.out.println(commandStats);

		commandStats = EPPCodecTst.testEncodeDecode(theCommand);
		System.out.println(commandStats);

		EPPTransId respTransId = new EPPTransId(theCommand.getTransId(), "54321-XYZ");

		EPPRegistryCreateResp theResponse = new EPPRegistryCreateResp(respTransId, "newtld");
		Calendar baseCal = Calendar.getInstance();
		Calendar endCal = (Calendar) baseCal.clone();
		theResponse.setCreateDate(baseCal.getTime());
		endCal = (Calendar) baseCal.clone();
		endCal.add(Calendar.MONTH, 9);
		endCal.add(Calendar.YEAR, 10);

		EPPEncodeDecodeStats responseStats = EPPCodecTst.testEncodeDecode(theResponse);
		System.out.println(responseStats);

		EPPCodecTst.printEnd("testRegistryCreate");
	}

	public void testRegistryUpdate() {
		EPPCodecTst.printStart("testRegistryUpdate");

		EPPRegistryZone zone = new EPPRegistryZone();
		zone.setName("newtld");

		EPPRegistryUpdateCmd theCommand = null;
		theCommand = new EPPRegistryUpdateCmd("ABC-12349", zone);

		zone.setDomain(this.buildInfoDomain());
		zone.setHost(this.buildInfoHost());
		zone.setContact(this.buildContact());

		EPPRegistryServices services = new EPPRegistryServices();
		services.addObjURI(new EPPRegistryObjURI(EPPRegistryMapFactory.NS, Boolean.TRUE));
		EPPRegistryServicesExt svcExt = new EPPRegistryServicesExt();
		services.setExtension(svcExt);
		svcExt.addExtURI(new EPPRegistryExtURI(EPPRegistryMapFactory.NS, Boolean.TRUE));
		zone.setServices(services);

		zone.setCreatedBy("crId");
		zone.setCreatedDate(new Date());

		EPPEncodeDecodeStats commandStats = EPPCodecTst.testEncodeDecode(theCommand);
		System.out.println(commandStats);

		commandStats = EPPCodecTst.testEncodeDecode(theCommand);
		System.out.println(commandStats);

		EPPTransId respTransId = new EPPTransId(theCommand.getTransId(), "54321-XYZ");

		EPPResponse theResponse = new EPPResponse(respTransId);

		EPPEncodeDecodeStats responseStats = EPPCodecTst.testEncodeDecode(theResponse);
		System.out.println(responseStats);

		EPPCodecTst.printEnd("testRegistryUpdate");
	}

	public void testRegistryDelete() {
		EPPCodecTst.printStart("testRegistryDelete");

		EPPRegistryDeleteCmd theCommand = null;
		theCommand = new EPPRegistryDeleteCmd("ABC-12349", "newetld");

		EPPEncodeDecodeStats commandStats = EPPCodecTst.testEncodeDecode(theCommand);
		System.out.println(commandStats);

		EPPTransId respTransId = new EPPTransId(theCommand.getTransId(), "54321-XYZ");

		EPPResponse theResponse = new EPPResponse(respTransId);

		EPPEncodeDecodeStats responseStats = EPPCodecTst.testEncodeDecode(theResponse);
		System.out.println(responseStats);

		EPPCodecTst.printEnd("testRegistryDelete");
	}

	/**
	 * Creates a {@link EPPRegistryZone} test instance. This can be used to
	 * bootstrap testing of the Registry Mapping or a Registry Mapping extension.
	 * 
	 * @return {@link EPPRegistryZone} instance that is populated with an
	 *         initial set of .COM zone information.
	 */
	public static EPPRegistryZone createZoneInfo() {
		EPPRegistryZone zoneInfo = new EPPRegistryZone("com");
		zoneInfo.setGroup("1");

		EPPRegistryServices services = new EPPRegistryServices();
		services.addObjURI(new EPPRegistryObjURI("http://www.verisign.com/epp/rgp-poll-1.0", Boolean.TRUE));
		services.addObjURI(new EPPRegistryObjURI("urn:ietf:params:xml:ns:host-1.0", Boolean.TRUE));
		services.addObjURI(new EPPRegistryObjURI("urn:ietf:params:xml:ns:contact-1.0", Boolean.TRUE));
		services.addObjURI(new EPPRegistryObjURI("urn:ietf:params:xml:ns:domain-1.0", Boolean.TRUE));
		services.addObjURI(new EPPRegistryObjURI("http://www.verisign.com/epp/lowbalance-poll-1.0", Boolean.FALSE));
		EPPRegistryServicesExt svcExt = new EPPRegistryServicesExt();
		services.setExtension(svcExt);
		svcExt.addExtURI(new EPPRegistryExtURI("http://www.verisign-grs.com/epp/namestoreExt-1.1", Boolean.TRUE));
		svcExt.addExtURI(new EPPRegistryExtURI("urn:ietf:params:xml:ns:rgp-1.0", Boolean.TRUE));
		svcExt.addExtURI(new EPPRegistryExtURI("http://www.verisign.com/epp/sync-1.0", Boolean.TRUE));
		svcExt.addExtURI(new EPPRegistryExtURI("http://www.verisign.com/epp/idnLang-1.0", Boolean.TRUE));
		svcExt.addExtURI(new EPPRegistryExtURI("http://www.verisign.com/epp/jobsContact-1.0", Boolean.TRUE));
		svcExt.addExtURI(new EPPRegistryExtURI("http://www.verisign.com/epp/premiumdomain-1.0", Boolean.TRUE));
		svcExt.addExtURI(new EPPRegistryExtURI("urn:ietf:params:xml:ns:secDNS-1.1", Boolean.FALSE));
		zoneInfo.setServices(services);

		zoneInfo.setCreatedBy("crId");
		zoneInfo.setCreatedDate(new Date());
		zoneInfo.setLastUpdatedBy("upId");
		zoneInfo.setLastUpdatedDate(new Date());
		zoneInfo.addBatchJob(new EPPRegistryBatchJob("pendingDelete", "0 14 * * *", "Pending Delete Batch", "EDT5EDT"));
		zoneInfo.addSystemZone(new EPPRegistryZoneName("EXAMPLE"));
		zoneInfo.addSystemZone(new EPPRegistryZoneName("EXAMPLE2", Form.aLabel));

		zoneInfo.setDomain(buildInfoDomain());

		zoneInfo.setHost(buildInfoHost());

		zoneInfo.setContact(buildContact());
		return zoneInfo;
	}

	/**
	 * Creates a {@link EPPRegistrySystemInfo} test instance. This can be used to
	 * bootstrap testing of the Registry Mapping or a Registry Mapping extension.
	 * 
	 * @return {@link EPPRegistrySystemInfo} instance that is populated system
	 *         data.
	 */
	public static EPPRegistrySystemInfo createSystemInfo() {
		return new EPPRegistrySystemInfo(Integer.valueOf(200), Integer.valueOf(600000), Integer.valueOf(86400000), Integer.valueOf(10000),
		      Integer.valueOf(10), Integer.valueOf(1000));
	}

	/**
	 * Builds a {@link EPPRegistryDomain} instance for use in Registry Mapping
	 * tests or Registry Mapping extension tests.
	 * 
	 * @return {@link EPPRegistryDomain} instance populated with an initial set of
	 *               domain policy information.
	 */
	public static EPPRegistryDomain buildInfoDomain() {
		EPPRegistryDomain domain = new EPPRegistryDomain();

		List domainNames = new ArrayList();
		EPPRegistryDomainName domainName = new EPPRegistryDomainName();
		domainName.setLevel(Integer.valueOf(2));
		domainName.setMinLength(Integer.valueOf(5));
		domainName.setMaxLength(Integer.valueOf(50));
		domainName.setAlphaNumStart(Boolean.valueOf(true));
		domainName.setAlphaNumEnd(Boolean.valueOf(false));
		domainName.setALabelSupported(Boolean.valueOf(true));

		List regex = new ArrayList();
		EPPRegistryRegex r = new EPPRegistryRegex("^\\w+.*$", "test regex");
		regex.add(r);

		r = new EPPRegistryRegex("^\\d+.*$");
		regex.add(r);
		domainName.setRegex(regex);

		EPPRegistryReservedNames reservedNames = new EPPRegistryReservedNames();
		List rNames = new ArrayList();
		reservedNames.setReservedNames(rNames);
		rNames.add("reserved1");
		rNames.add("reserved2");
		// reservedNames.setReservedNameURI("http://example.com/reservedNames");

		domainName.setReservedNames(reservedNames);
		domainNames.add(domainName);

		try {
			domainName = (EPPRegistryDomainName) domainName.clone();
			domainName.setLevel(Integer.valueOf(3));
			domainName.getReservedNames().setReservedNames(new ArrayList());
			domainName.getReservedNames().setReservedNameURI("http://testrn.vrsn.com");
			domainNames.add(domainName);
		}
		catch (CloneNotSupportedException e) {
			fail("failed to clone domainName");
		}

		domain.setDomainNames(domainNames);

		EPPRegistryIDN idn = new EPPRegistryIDN();
		idn.setIdnVersion("1.1");
		idn.setIdnaVersion("2008");
		idn.setUnicodeVersion("6.0");
		idn.addLanguage(new EPPRegistryLanguage("CHI", "http://www.iana.org/domains/idn-tables/tables/com_zh_1.1.txt",
		      EPPRegistryLanguage.VariantStrategy.restricted));
		idn.addLanguage(new EPPRegistryLanguage("LATN", "http://www.iana.org/domains/idn-tables/tables/eu_latn_1.0.html",
		      EPPRegistryLanguage.VariantStrategy.blocked));
		idn.setCommingleAllowed(Boolean.TRUE);
		domain.setIdn(idn);

		domain.setPremiumSupport(Boolean.valueOf(true));
		domain.setContactsSupported(Boolean.valueOf(false));

		domain.addContact(new EPPRegistryDomainContact(EPPRegistryDomainContact.Type.admin, 1, 4));
		domain.addContact(new EPPRegistryDomainContact(EPPRegistryDomainContact.Type.billing, 2, 5));
		domain.addContact(new EPPRegistryDomainContact(EPPRegistryDomainContact.Type.tech, 3, 6));

		domain.setNameServerLimit(new EPPRegistryDomainNSLimit(1, 16));

		domain.setChildHostLimit(new EPPRegistryDomainHostLimit(2, 32));

		domain.addPeriod(new EPPRegistryDomainPeriod("create", Boolean.TRUE));
		domain.addPeriod(new EPPRegistryDomainPeriod("renew", 1, EPPRegistryPeriodType.Unit.y, 60,
		      EPPRegistryPeriodType.Unit.m, 5, EPPRegistryPeriodType.Unit.y));
		domain.addPeriod(new EPPRegistryDomainPeriod("transfer", 1, EPPRegistryPeriodType.Unit.y, 8,
		      EPPRegistryPeriodType.Unit.y, 3, EPPRegistryPeriodType.Unit.y));

		domain.setTransferHoldPeriod(new EPPRegistryTransferHoldPeriodType(5, EPPRegistryPeriodType.Unit.d));

		domain.addGracePeriod(new EPPRegistryGracePeriod("create", 1, EPPRegistryPeriodType.Unit.d));
		domain.addGracePeriod(new EPPRegistryGracePeriod("renew", 2, EPPRegistryPeriodType.Unit.h));
		domain.addGracePeriod(new EPPRegistryGracePeriod("transfer", 3, EPPRegistryPeriodType.Unit.m));

		EPPRegistryRGP rgp = new EPPRegistryRGP();
		rgp.setPendingDeletePeriod(new EPPRegistryPendingDeletePeriodType(1, EPPRegistryPeriodType.Unit.m));
		rgp.setRedemptionPeriod(new EPPRegistryRedemptionPeriodType(1, EPPRegistryPeriodType.Unit.m));
		rgp.setPendingRestorePeriod(new EPPRegistryPendingRestorePeriodType(1, EPPRegistryPeriodType.Unit.m));
		domain.setRgp(rgp);

		EPPRegistryDNSSEC dnssec = new EPPRegistryDNSSEC();
		EPPRegistryDS ds = new EPPRegistryDS(1, 3, null, null);
		ds.addAlgorithm("algDS1");
		ds.addAlgorithm("algDS2");
		ds.addDigestType("digest1");
		ds.addDigestType("digest2");
		dnssec.setDs(ds);

		// EPPRegistryKey key = new EPPRegistryKey(2, 4, null);
		// key.addAlgorithm("algKey1");
		// key.addAlgorithm("algKey2");
		// key.addAlgorithm("algKey3");
		// dnssec.setKey(key);

		dnssec.setMaxSigLife(new EPPRegistryMaxSig(true, 1, 2, 3));
		dnssec.setUrgent(Boolean.TRUE);

		domain.setDnssec(dnssec);
		domain.setMaxCheckDomain(Integer.valueOf(12));
		domain.setAuthInfoRegex(new EPPRegistryRegex("^.*$", "exp"));

		EPPRegistrySupportedStatus supportedStatus = new EPPRegistrySupportedStatus();
		supportedStatus.addStatus(Status.DOMAIN_CLIENTDELETEPROHIBITED);
		supportedStatus.addStatus(Status.DOMAIN_SERVERDELETEPROHIBITED);
		supportedStatus.addStatus(Status.DOMAIN_CLIENTHOLD);
		supportedStatus.addStatus(Status.DOMAIN_SERVERHOLD);
		supportedStatus.addStatus(Status.DOMAIN_CLIENTRENEWPROHIBITED);
		supportedStatus.addStatus(Status.DOMAIN_SERVERRENEWPROHIBITED);
		supportedStatus.addStatus(Status.DOMAIN_CLIENTTRANSFERPROHIBITED);
		supportedStatus.addStatus(Status.DOMAIN_SERVERTRANSFERPROHIBITED);
		supportedStatus.addStatus(Status.DOMAIN_CLIENTUPDATEPROHIBITED);
		supportedStatus.addStatus(Status.DOMAIN_SERVERUPDATEPROHIBITED);
		supportedStatus.addStatus(Status.DOMAIN_INACTIVE);
		supportedStatus.addStatus(Status.DOMAIN_OK);
		supportedStatus.addStatus(Status.DOMAIN_PENDINGCREATE);
		supportedStatus.addStatus(Status.DOMAIN_PENDINGDELETE);
		supportedStatus.addStatus(Status.DOMAIN_PENDINGRENEW);
		supportedStatus.addStatus(Status.DOMAIN_PENDINGTRANSFER);
		supportedStatus.addStatus(Status.DOMAIN_PENDINGUPDATE);
		domain.setSupportedStatus(supportedStatus);

		return domain;
	}

	/**
	 * Builds a {@link EPPRegistryHost} instance for use in Registry Mapping
	 * tests or Registry Mapping extension tests.
	 * 
	 * @return {@link EPPRegistryHost} instance populated with an initial set of
	 *               host policy information.
	 */
	public static EPPRegistryHost buildInfoHost() {
		EPPRegistryHost host = new EPPRegistryHost();

		host.setInternal(new EPPRegistryInternalHost(5, 15, EPPRegistryInternalHost.SharePolicy.perZone, Boolean.FALSE));
		host.setExternal(new EPPRegistryExternalHost(2, 12, EPPRegistryExternalHost.SharePolicy.perZone, Boolean.FALSE));
		host.addNameRegex(new EPPRegistryRegex("^.*$", "exp1"));
		host.addNameRegex(new EPPRegistryRegex("^.*$", "exp2"));
		host.setMaxCheckHost(Integer.valueOf(15));

		EPPRegistrySupportedStatus supportedStatus = new EPPRegistrySupportedStatus();
		supportedStatus.addStatus(Status.HOST_CLIENTDELETEPROHIBITED);
		supportedStatus.addStatus(Status.HOST_SERVERDELETEPROHIBITED);
		supportedStatus.addStatus(Status.HOST_CLIENTUPDATEPROHIBITED);
		supportedStatus.addStatus(Status.HOST_SERVERUPDATEPROHIBITED);
		supportedStatus.addStatus(Status.HOST_LINKED);
		supportedStatus.addStatus(Status.HOST_OK);
		supportedStatus.addStatus(Status.HOST_PENDINGCREATE);
		supportedStatus.addStatus(Status.HOST_PENDINGDELETE);
		supportedStatus.addStatus(Status.HOST_PENDINGTRANSFER);
		supportedStatus.addStatus(Status.HOST_PENDINGUPDATE);
		host.setSupportedStatus(supportedStatus);

		return host;
	}

	/**
	 * Builds a {@link EPPRegistryContact} instance for use in Registry Mapping
	 * tests or Registry Mapping extension tests.
	 * 
	 * @return {@link EPPRegistryContact} instance populated with an initial set
	 *               of contact policy information.
	 */
	public static EPPRegistryContact buildContact() {
		EPPRegistryContact contact = new EPPRegistryContact();

		contact.setIntPostalInfoTypeSupport(EPPRegistryContact.PostalInfoTypeSupport.locSupport);

		contact.setContactIdRegex(new EPPRegistryRegex("^.*$"));
		contact.setSharePolicy(EPPRegistryContact.SharePolicy.perZone);
		contact.setAuthInfoRegex(new EPPRegistryRegex("^.*$", "exp"));

		contact.setMaxCheckContact(Integer.valueOf(15));

		EPPRegistryPostal postalInfo = new EPPRegistryPostal();
		postalInfo.setName(new EPPRegistryContactName(5, 15));
		postalInfo.setOrg(new EPPRegistryContactOrg(2, 12));
		postalInfo.setVoiceRequired(Boolean.TRUE);
		postalInfo.setVoiceExt(new EPPRegistryMinMaxLength(5, 15));
		postalInfo.setFaxExt(new EPPRegistryMinMaxLength(5, 15));
		List emailRegex = new ArrayList();
		postalInfo.setEmailRegex(new EPPRegistryRegex("^.+\\..+$"));

		EPPRegistryContactAddress address = new EPPRegistryContactAddress();
		address.setStreet(new EPPRegistryContactStreet(2, 12, 0, 3));
		address.setCity(new EPPRegistryContactCity(5, 15));
		address.setStateProvince(new EPPRegistryContactStateProvince(1, 11));
		address.setPostalCode(new EPPRegistryContactPostalCode(2, 12));

		postalInfo.setAddress(address);

		contact.setMaxCheckContact(Integer.valueOf(5));

		contact.setPostalInfo(postalInfo);

		contact.setClientDisclosureSupported(Boolean.TRUE);

		EPPRegistrySupportedStatus supportedStatus = new EPPRegistrySupportedStatus();
		supportedStatus.addStatus(Status.CONTACT_CLIENTDELETEPROHIBITED);
		supportedStatus.addStatus(Status.CONTACT_SERVERDELETEPROHIBITED);
		supportedStatus.addStatus(Status.CONTACT_CLIENTTRANSFERPROHIBITED);
		supportedStatus.addStatus(Status.CONTACT_SERVERTRANSFERPROHIBITED);
		supportedStatus.addStatus(Status.CONTACT_CLIENTUPDATEPROHIBITED);
		supportedStatus.addStatus(Status.CONTACT_SERVERUPDATEPROHIBITED);
		supportedStatus.addStatus(Status.CONTACT_LINKED);
		supportedStatus.addStatus(Status.CONTACT_OK);
		supportedStatus.addStatus(Status.CONTACT_PENDINGCREATE);
		supportedStatus.addStatus(Status.CONTACT_PENDINGDELETE);
		supportedStatus.addStatus(Status.CONTACT_PENDINGTRANSFER);
		supportedStatus.addStatus(Status.CONTACT_PENDINGUPDATE);
		contact.setSupportedStatus(supportedStatus);

		contact.setTransferHoldPeriod(new EPPRegistryTransferHoldPeriodType(5, EPPRegistryPeriodType.Unit.d));

		return contact;
	}

	/**
	 * Unit test main, which accepts the following system property options: <br>
	 * 
	 * <ul>
	 * <li>iterations Number of unit test iterations to run</li>
	 * <li>validate Turn XML validation on (<code>true</code>) or off (
	 * <code>false</code>). If validate is not specified, validation will be off.
	 * </li>
	 * </ul>
	 * 
	 * 
	 * @param args
	 *           DOCUMENT ME!
	 */
	public static void main(String[] args) {
		// Number of Threads
		int numThreads = 1;
		String threadsStr = System.getProperty("threads");

		if (threadsStr != null) {
			numThreads = Integer.parseInt(threadsStr);
		}

		// Run test suite in multiple threads?
		if (numThreads > 1) {
			// Spawn each thread passing in the Test Suite
			for (int i = 0; i < numThreads; i++) {
				TestThread thread = new TestThread("EPPRegistryTst Thread " + i, EPPRegistryTst.suite());
				thread.start();
			}
		}
		else { // Single threaded mode.
			junit.textui.TestRunner.run(EPPRegistryTst.suite());
		}
	}

	/**
	 * JUNIT <code>suite</code> static method, which returns the tests associated
	 * with <code>EPPRegistryTst</code>.
	 * 
	 * @return DOCUMENT ME!
	 */
	public static Test suite() {
		EPPCodecTst.initEnvironment();

		TestSuite suite = new TestSuite(EPPRegistryTst.class);

		// iterations Property
		String numIterProp = System.getProperty("iterations");

		if (numIterProp != null) {
			numIterations = Integer.parseInt(numIterProp);
		}

		// Add the EPPRegistryMapFactory to the EPPCodec.
		try {
			EPPFactory.getInstance().addMapFactory("com.verisign.epp.codec.registry.v01.EPPRegistryMapFactory");
		}
		catch (EPPCodecException e) {
			Assert.fail("EPPCodecException adding EPPRegistryMapFactory to EPPCodec: " + e);
		}

		return suite;
	}

	/**
	 * JUNIT <code>setUp</code>, which currently does nothing.
	 */
	protected void setUp() {
	}

	/**
	 * JUNIT <code>tearDown</code>, which currently does nothing.
	 */
	protected void tearDown() {
	}

	/**
	 * Sets the number of iterations to run per test.
	 * 
	 * @param aNumIterations
	 *           number of iterations to run per test
	 */
	public static void setNumIterations(long aNumIterations) {
		numIterations = aNumIterations;
	}
}
