/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v01;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * Defines the Key Data Interface, as defined in RFC 5910, policies. The
 * &lt;registry:keyDataInterface&gt; element contains the following child
 * elements: <br>
 * <ul>
 * <li>&lt;registry:min&gt; - the minimum number of keys associated with the
 * domain object. Use {@link #getMin()} and {@link #setMin(Integer)} to get and
 * set the element.</li>
 * <li>&lt;registry:max&gt; - the maximum number of keys associated with the
 * domain object. Use {@link #getMax()} and {@link #setMax(Integer)} to get and
 * set the element.</li>
 * <li>&lt;registry:alg&gt; - Zero or more &lt;registry:alg&gt; elements that
 * define the supported algorithms as described in section 2.1.3 of RFC 4034.
 * Use {@link #getAlgorithms()} and {@link #setAlgorithms(List)} to get and set
 * the element. Use {@link #addAlgorithm(String)} to add an algorithm
 * {@code String} to an existing {@code List}.</li>
 * </ul>
 *
 * @see com.verisign.epp.codec.registry.v01.EPPRegistryDNSSEC
 */
public class EPPRegistryKey implements EPPCodecComponent {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryKey.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for {@code EPPRegistryKey}.
	 */
	public static final String ELM_LOCALNAME = "keyDataInterface";

	/**
	 * XML root tag for {@code EPPRegistryKey}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML Element Name of {@code min} attribute.
	 */
	public static final String ELM_MIN = "min";

	/**
	 * XML Element Name of {@code max} attribute.
	 */
	public static final String ELM_MAX = "max";

	/**
	 * XML Element Name of {@code algorithms} attribute.
	 */
	public static final String ELM_ALGORITHM = "alg";

	/**
	 * Minimum number of keys
	 */
	private Integer min = null;

	/**
	 * Maximum number of keys
	 */
	private Integer max = null;

	/**
	 * {@code List} of algorithms in {@code String}
	 */
	private List<String> algorithms = new ArrayList<String>();

	/**
	 * Default constructor. Must call {@link #setMin(Integer)} and
	 * {@link #setMax(Integer)} before calling {@link #encode(Document)} method.
	 */
	public EPPRegistryKey() {
	}

	/**
	 * Constructs an instance with {@code min}, {@code max} and {@code List} of
	 * {@code algorithms}.
	 *
	 * @param aMin
	 *           minimum number of keys associated with the domain object
	 * @param aMax
	 *           maximum number of keys associated with the domain object
	 * @param aAlgorithms
	 *           supported algorithms as described in section 2.1.3 of RFC 4034
	 */
	public EPPRegistryKey(Integer aMin, Integer aMax, List<String> aAlgorithms) {
		this.min = aMin;
		this.max = aMax;
		this.algorithms = aAlgorithms;
	}

	/**
	 * Constructs an instance with {@code min}, {@code max} and {@code List} of
	 * {@code algorithms}.
	 *
	 * @param aMin
	 *           minimum number of keys associated with the domain object
	 * @param aMax
	 *           maximum number of keys associated with the domain object
	 * @param aAlgorithms
	 *           supported algorithms as described in section 2.1.3 of RFC 4034
	 */
	public EPPRegistryKey(int aMin, int aMax, List<String> aAlgorithms) {
		this.min = aMin;
		this.max = aMax;
		this.algorithms = aAlgorithms;
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryKey} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the {@code EPPRegistryKey}
	 *         instance.
	 *
	 * @exception EPPEncodeException
	 *               - Unable to encode {@code EPPRegistryKey} instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		try {
			validateState();
		}
		catch (EPPCodecException e) {
			throw new EPPEncodeException("Invalid state on EPPRegistryKey.encode: " + e);
		}

		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS,
		      EPPRegistryMapFactory.NS_PREFIX + ":" + getRootName());

		// Min
		EPPUtil.encodeString(aDocument, root, this.min.toString(), EPPRegistryMapFactory.NS,
		      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_MIN);

		// Max
		EPPUtil.encodeString(aDocument, root, this.max.toString(), EPPRegistryMapFactory.NS,
		      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_MAX);

		// Algorithms
		EPPUtil.encodeList(aDocument, root, this.algorithms, EPPRegistryMapFactory.NS,
		      EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_ALGORITHM);

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryKey} attributes from the aElement DOM Element
	 * tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryKey} from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {
		// Min
		this.min = EPPUtil.decodeInteger(aElement, EPPRegistryMapFactory.NS, ELM_MIN);

		// Max
		this.max = EPPUtil.decodeInteger(aElement, EPPRegistryMapFactory.NS, ELM_MAX);

		// Algorithms
		this.algorithms = EPPUtil.decodeList(aElement, EPPRegistryMapFactory.NS, ELM_ALGORITHM);
	}

	/**
	 * Validate the state of the {@code EPPRegistryKey} instance. A valid state
	 * means that all of the required attributes have been set. If validateState
	 * returns without an exception, the state is valid. If the state is not
	 * valid, the EPPCodecException will contain a description of the error.
	 * throws EPPCodecException State error. This will contain the name of the
	 * attribute that is not valid.
	 *
	 * @throws EPPCodecException
	 *            Validation error
	 */
	void validateState() throws EPPCodecException {
		// Min
		if (this.min == null || this.min.intValue() < 0) {
			throw new EPPCodecException("min is required and should be greater than or equal to 0");
		}

		// Max
		if (this.max == null || this.max.intValue() < this.min.intValue()) {
			throw new EPPCodecException("max is required and should be greater than or equal to min");
		}
	}

	/**
	 * Clone {@code EPPRegistryKey}.
	 *
	 * @return clone of {@code EPPRegistryKey}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPRegistryKey clone = (EPPRegistryKey) super.clone();

		// Algorithms
		if (this.algorithms != null) {
			clone.algorithms = (List) ((ArrayList) this.algorithms).clone();
		}
		return clone;
	}

	/**
	 * implements a deep {@code EPPRegistryKey} compare.
	 *
	 * @param aObject
	 *           {@code EPPRegistryKey} instance to compare with
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryKey)) {
			return false;
		}

		EPPRegistryKey theComp = (EPPRegistryKey) aObject;

		// Min
		if (!EqualityUtil.equals(this.min, theComp.min)) {
			cat.error("EPPRegistryKey.equals(): min not equal");
			return false;
		}

		// Max
		if (!EqualityUtil.equals(this.max, theComp.max)) {
			cat.error("EPPRegistryKey.equals(): max not equal");
			return false;
		}

		// Algorithms
		if (!((this.algorithms == null) ? (theComp.algorithms == null) :
		      EPPUtil.equalLists(this.algorithms, theComp.algorithms))) {
			cat.error("EPPRegistryKey.equals(): algorithms not equal");
			return false;
		}

		return true;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Gets the minimum number of keys.
	 *
	 * @return minimum number of keys associated with the domain object.
	 */
	public Integer getMin() {
		return this.min;
	}

	/**
	 * Sets the minimum number of keys.
	 *
	 * @param aMin
	 *           minimum number of keys associated with the domain object.
	 */
	public void setMin(Integer aMin) {
		this.min = aMin;
	}

	/**
	 * Gets the maximum number of keys.
	 *
	 * @return maximum number of keys associated with the domain object.
	 */
	public Integer getMax() {
		return this.max;
	}

	/**
	 * Sets the maximum number of keys.
	 *
	 * @param aMax
	 *           maximum number of keys associated with the domain object.
	 */
	public void setMax(Integer aMax) {
		this.max = aMax;
	}

	/**
	 * Is the algorithms defined?
	 *
	 * @return {@code true} if the algorithms is defined; {@code false}
	 *         otherwise.
	 */
	public boolean hasAlgorithms() {
		return (this.algorithms != null && !this.algorithms.isEmpty() ? true : false);
	}

	/**
	 * Gets the supported algorithms.
	 *
	 * @return {@code List} of supported algorithms as described in section 2.1.3
	 *         of RFC 4034
	 */
	public List<String> getAlgorithms() {
		return this.algorithms;
	}

	/**
	 * Sets the supported algorithms.
	 *
	 * @param aAlgorithms
	 *           {@code List} of supported algorithms as described in section
	 *           2.1.3 of RFC 4034
	 */
	public void setAlgorithms(List<String> aAlgorithms) {
		if (aAlgorithms == null) {
			this.algorithms = new ArrayList<String>();
		}
		else {
			this.algorithms = aAlgorithms;
		}
	}

	/**
	 * Adds one algorithm to existing supported algorithms.
	 *
	 * @param aAltorithm
	 *           supported algorithms as described in section 2.1.3 of RFC 4034
	 */
	public void addAlgorithm(String aAltorithm) {
		if (this.algorithms == null) {
			this.algorithms = new ArrayList<String>();
		}
		this.algorithms.add(aAltorithm);
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 *
	 * @return XML namespace for the {@code EPPCodecComponent}.
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

	/**
	 * Gets the root element local name.
	 * 
	 * @return Root element local name.
	 */
	protected String getRootName() {
		return ELM_LOCALNAME;
	}

}
