/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.registry.v01;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecException;
import com.verisign.epp.codec.gen.EPPCreateCmd;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * Represents an EPP Registry &lt;create&gt; command, which provides a transform
 * operation that allows a client to create a registry object. In addition to
 * the standard EPP command elements, the &lt;create&gt; command MUST contain a
 * &lt;registry:create&gt; element that identifies the registry namespace and
 * the location of the registry schema. The &lt;registry:create&gt; element MUST
 * contain the following child elements: <br>
 * <br>
 *
 * <ul>
 * <li>A &lt;registry:zone&gt; element that contains the detailed registry
 * information of the object to be created. Use <code>getZone</code> and
 * <code>setZone</code> to get and set the element.</li>
 * </ul>
 *
 * <code>EPPRegistryCreateResp</code> is the concrete <code>EPPReponse</code>
 * associated with <code>EPPRegistryCreateCmd</code>.<br>
 * <br>
 *
 * @see com.verisign.epp.codec.registry.v01.EPPRegistryCreateResp
 */
public class EPPRegistryCreateCmd extends EPPCreateCmd {

	/**
	 * Logger
	 */
	private static Logger cat = Logger.getLogger(EPPRegistryCreateCmd.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for {@code EPPRegistryCreateCmd}.
	 */
	public static final String ELM_LOCALNAME = "create";

	/**
	 * XML root tag for {@code EPPRegistryCreateCmd}.
	 */
	public static final String ELM_NAME = EPPRegistryMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * Instance of {@link EPPRegistryZone} to create
	 */
	private EPPRegistryZone zone = null;

	/**
	 * Creates an empty {@code EPPRegistryCreateCmd} object. {@code zone} is set
	 * to {@code null}. Use {@code setZone} to set {@code zone} attribute before
	 * sending the create command.
	 */
	public EPPRegistryCreateCmd() {
		this.zone = null;
	}

	/**
	 * Creates a new {@code EPPRegistryCreateCmd} object that will create a
	 * registry object based on the info in {@code zone}.
	 *
	 * @param aTransId
	 *           Transaction Id associated with command.
	 * @param aZone
	 *           Instance of {@link EPPRegistryZone} to create
	 */
	public EPPRegistryCreateCmd(String aTransId, EPPRegistryZone aZone) {
		super(aTransId);
		this.zone = aZone;
	}

	/**
	 * Get the EPP command Namespace associated with {@code EPPRegistryCreateCmd}
	 * .
	 *
	 * @return {@link EPPRegistryMapFactory}.NS
	 */
	@Override
	public String getNamespace() {
		return EPPRegistryMapFactory.NS;
	}

	/**
	 * Validate the state of the <code>EPPRegistryCreateCmd</code> instance. A
	 * valid state means that all of the required attributes have been set. If
	 * validateState returns without an exception, the state is valid. If the
	 * state is not valid, the <code>EPPCodecException</code> will contain a
	 * description of the error. throws EPPCodecException State error. This will
	 * contain the name of the attribute that is not valid.
	 *
	 * @throws EPPCodecException
	 */
	void validateState() throws EPPCodecException {
		if (this.zone == null) {
			throw new EPPCodecException("zone required element is not set");
		}
	}

	/**
	 * Compare an instance of <code>EPPRegistryCreateCmd</code> with this
	 * instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPRegistryCreateCmd)) {
			cat.error(
			      "EPPRegistryCreateCmd.equals(): " + aObject.getClass().getName() + " not EPPRegistryCreateCmd instance");

			return false;
		}

		if (!super.equals(aObject)) {
			cat.error("EPPRegistryCreateCmd.equals(): super class not equal");

			return false;
		}

		EPPRegistryCreateCmd theComp = (EPPRegistryCreateCmd) aObject;

		// Zone
		if (!EqualityUtil.equals(this.zone, theComp.zone)) {
			cat.error("EPPRegistryCreateCmd.equals(): zone not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone <code>EPPRegistryCreateCmd</code>.
	 *
	 * @return clone of <code>EPPRegistryCreateCmd</code>
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPRegistryCreateCmd clone = (EPPRegistryCreateCmd) super.clone();

		if (this.zone != null) {
			clone.zone = (EPPRegistryZone) this.zone.clone();
		}

		return clone;
	}

	/**
	 * Implementation of <code>Object.toString</code>, which will result in an
	 * indented XML <code>String</code> representation of the concrete
	 * <code>EPPCodecComponent</code>.
	 *
	 * @return Indented XML <code>String</code> if successful; <code>ERROR</code>
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPRegistryCreateCmd} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Root DOM Element representing the {@code EPPRegistryCreateCmd}
	 *         instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPRegistryCreateCmd} instance.
	 */
	@Override
	protected Element doEncode(Document aDocument) throws EPPEncodeException {
		try {
			// Validate States
			validateState();
		}
		catch (EPPCodecException e) {
			cat.error("EPPRegistryCreateCmd.doEncode(): Invalid state on encode: " + e);
			throw new EPPEncodeException("EPPRegistryCreateCmd invalid state: " + e);
		}

		Element root = aDocument.createElementNS(EPPRegistryMapFactory.NS, ELM_NAME);

		// Zone
		EPPUtil.encodeComp(aDocument, root, this.zone);

		return root;
	}

	/**
	 * Decode the {@code EPPRegistryCreateCmd} attributes from the aElement DOM
	 * Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPRegistryCreateCmd} from.
	 *
	 * @throws EPPDecodeException
	 *            Unable to decode aElement
	 */
	@Override
	protected void doDecode(Element aElement) throws EPPDecodeException {
		// Zone
		this.zone = (EPPRegistryZone) EPPUtil.decodeComp(aElement, EPPRegistryMapFactory.NS, EPPRegistryZone.ELM_NAME,
		      EPPRegistryZone.class);
	}

	/**
	 * Get zone to create
	 *
	 * @return Instance of {@link EPPRegistryZone} to create
	 */
	public EPPRegistryZone getZone() {
		return this.zone;
	}

	/**
	 * Set zone to create
	 *
	 * @param aZone
	 *           Instance of {@link EPPRegistryZone} to create
	 */
	public void setZone(EPPRegistryZone aZone) {
		this.zone = aZone;
	}
}
