/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.interfaces;

import java.util.Random;

import org.apache.log4j.Logger;

import com.verisign.epp.codec.gen.EPPResponse;
import com.verisign.epp.codec.gen.EPPResult;
import com.verisign.epp.codec.org.EPPOrgAddress;
import com.verisign.epp.codec.org.EPPOrgCheckResp;
import com.verisign.epp.codec.org.EPPOrgContact;
import com.verisign.epp.codec.org.EPPOrgCreateResp;
import com.verisign.epp.codec.org.EPPOrgInfoResp;
import com.verisign.epp.codec.org.EPPOrgPostalDefinition;
import com.verisign.epp.codec.org.EPPOrgRole;
import com.verisign.epp.codec.org.EPPOrgRoleStatus;
import com.verisign.epp.codec.org.EPPOrgStatus;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.Environment;
import com.verisign.epp.util.TestThread;

import junit.framework.Assert;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

/**
 * Is a unit test of the <code>EPPOrg</code> class. The unit test will
 * initialize a session with an EPP Server, will invoke <code>EPPOrg</code>
 * operations, and will end a session with an EPP Server. The configuration file
 * used by the unit test defaults to epp.config, but can be changed by passing
 * the file path as the first command line argument. The unit test can be run in
 * multiple threads by setting the "threads" system property. For example, the
 * unit test can be run in 2 threads with the configuration file
 * ../../epp.config with the following command: <br>
 * <br>
 * java com.verisign.epp.interfaces.EPPOrgTst -Dthreads=2 ../../epp.config <br>
 * <br>
 * The unit test is dependent on the use of <a
 * href=http://www.mcwestcorp.com/Junit.html>JUNIT 3.5</a><br>
 * <br>
 * <br>
 * <br>
 *
 * @author $Author: jim $
 * @version $Revision: 1.3 $
 */
public class EPPOrgTst extends TestCase {
	/**
	 * Handle to the Singleton EPP Application instance (
	 * <code>EPPApplicationSingle</code>)
	 */
	private static EPPApplicationSingle app = EPPApplicationSingle.getInstance();

	/** Name of configuration file to use for test (default = epp.config). */
	private static String configFileName = "epp.config";

	/** Logging category */
	private static final Logger cat = Logger.getLogger(EPPOrgTst.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/** EPP Org associated with test */
	private EPPOrg org = null;

	/** EPP Session associated with test */
	private EPPSession session = null;

	/** Current test iteration */
	private int iteration = 0;

	/**
	 * Random instance for the generation of unique objects (hosts, IP addresses,
	 * etc.).
	 */
	private Random rd = new Random(System.currentTimeMillis());

	/**
	 * Allocates an <code>EPPOrgTst</code> with a logical name. The constructor
	 * will initialize the base class <code>TestCase</code> with the logical
	 * name.
	 *
	 * @param name
	 *           Logical name of the test
	 */
	public EPPOrgTst(String name) {
		super(name);
	}

	/**
	 * JUNIT test method to implement the <code>EPPOrgTst TestCase</code>. Each
	 * sub-test will be invoked in order to satisfy testing the EPPOrg interface.
	 */
	public void testOrg() {
		int numIterations = 1;

		String iterationsStr = System.getProperty("iterations");

		if (iterationsStr != null) {
			numIterations = Integer.parseInt(iterationsStr);
		}

		for (iteration = 0; (numIterations == 0) || (iteration < numIterations); iteration++) {
			printStart("Test Suite");

			orgCheck();

			orgInfo();

			orgCreate();

			orgCreatePending();

			orgDelete();

			orgUpdate();

			printEnd("Test Suite");
		}
	}

	/**
	 * Unit test of <code>EPPOrg.sendCheck()</code> for a set of org identifiers.
	 */
	public void orgCheck() {
		printStart("orgCheck");

		EPPOrgCheckResp response;

		try {
			System.out.println("\n----------------------------------------------------------------");

			System.out.println("orgCheck: Check res1523, re1523, and 1523res org identifiers.");

			this.org.setTransId("ABC-12345");

			this.org.addOrgId("res1523");

			this.org.addOrgId("re1523");

			this.org.addOrgId("1523res");

			response = this.org.sendCheck();

			// -- Output all of the response attributes
			System.out.println("orgCheck: Org Check Response = [" + response + "]\n\n");
		}
		catch (EPPCommandException e) {
			handleException(e);
		}

		printEnd("domainCheck");
	}

	/**
	 * Unit test of <code>EPPOrg.sendInfo()</code> to get the information for a
	 * specific org identifier.
	 */
	public void orgInfo() {
		printStart("orgInfo");

		EPPOrgInfoResp response;

		try {
			System.out.println("\n----------------------------------------------------------------");

			System.out.println("orgInfo: Info res1523.");

			this.org.setTransId("ABC-12345");

			this.org.addOrgId("res1523");

			response = this.org.sendInfo();

			// -- Output all of the response attributes
			System.out.println("orgInfo: Org Info Response = [" + response + "]\n\n");
		}
		catch (EPPCommandException e) {
			handleException(e);
		}

		printEnd("orgInfo");
	}

	/**
	 * Unit test of <code>EPPOrg.sendCreate()</code> to create a org.
	 */
	public void orgCreate() {
		printStart("orgCreate");

		EPPOrgCreateResp response;

		try {
			System.out.println("\n----------------------------------------------------------------");

			System.out.println("orgCreate: Create res1523.");

			this.org.setTransId("ABC-12345");

			this.org.addOrgId("res1523");

			this.org.addRole(new EPPOrgRole("reseller", EPPOrgRoleStatus.ok, null));
			this.org.addRole(new EPPOrgRole("privacyproxy", EPPOrgRoleStatus.ok, null));

			this.org.setParentId("1523res");

			EPPOrgAddress theAddress = new EPPOrgAddress();
			theAddress.setStreets("124 Example Dr.", "Suite 200");
			theAddress.setCity("Dulles");
			theAddress.setStateProvince("VA");
			theAddress.setPostalCode("20166-6503");
			theAddress.setCountry("US");

			this.org.addPostalInfo(
			      new EPPOrgPostalDefinition(EPPOrgPostalDefinition.Type.INT, "Example Organization Inc.", theAddress));
			this.org.setVoice("+1.7035555555");
			this.org.setVoiceExt("1234");
			this.org.setFax("+1.7035555556");
			this.org.setEmail("contact@organization.example");
			this.org.setUrl("http://organization.example");
			this.org.addContact(new EPPOrgContact("sh8013", EPPOrgContact.Type.ADMINISTRATIVE));
			this.org.addContact(new EPPOrgContact("sh8013", EPPOrgContact.Type.BILLING));

			response = this.org.sendCreate();

			// -- Output all of the response attributes
			System.out.println("orgCreate: Org Create Response = [" + response + "]\n\n");
		}
		catch (EPPCommandException e) {
			handleException(e);
		}

		printEnd("orgCreate");
	}

	/**
	 * Unit test of <code>EPPOrg.sendCreate()</code> to create a pending org,
	 * which should include a pending action poll message.
	 */
	public void orgCreatePending() {
		printStart("orgCreatePending");

		EPPOrgCreateResp response;

		try {
			System.out.println("orgCreatePending: Clearing poll queue");

			this.session.setPollOp(EPPSession.OP_REQ);
			EPPResponse thePollResponse = this.session.sendPoll();

			while (thePollResponse.getResult().getCode() == EPPResult.SUCCESS_POLL_MSG) {
				this.session.setPollOp(EPPSession.OP_ACK);
				this.session.setMsgID(thePollResponse.getMsgQueue().getId());
				thePollResponse = this.session.sendPoll();
				this.session.setPollOp(EPPSession.OP_REQ);
				thePollResponse = this.session.sendPoll();
			}

			System.out.println("\n----------------------------------------------------------------");

			System.out.println("orgCreatePending: Create PENDING.");

			this.org.setTransId("ABC-12345");

			this.org.addOrgId("PENDING");

			this.org.addRole(new EPPOrgRole("reseller", EPPOrgRoleStatus.ok, null));
			this.org.addRole(new EPPOrgRole("privacyproxy", EPPOrgRoleStatus.ok, null));

			this.org.setParentId("1523res");

			EPPOrgAddress theAddress = new EPPOrgAddress();
			theAddress.setStreets("124 Example Dr.", "Suite 200");
			theAddress.setCity("Dulles");
			theAddress.setStateProvince("VA");
			theAddress.setPostalCode("20166-6503");
			theAddress.setCountry("US");

			this.org.addPostalInfo(
			      new EPPOrgPostalDefinition(EPPOrgPostalDefinition.Type.INT, "Example Organization Inc.", theAddress));
			this.org.setVoice("+1.7035555555");
			this.org.setVoiceExt("1234");
			this.org.setFax("+1.7035555556");
			this.org.setEmail("contact@organization.example");
			this.org.setUrl("http://organization.example");
			this.org.addContact(new EPPOrgContact("sh8013", EPPOrgContact.Type.ADMINISTRATIVE));
			this.org.addContact(new EPPOrgContact("sh8013", EPPOrgContact.Type.BILLING));

			response = this.org.sendCreate();

			Assert.assertEquals(EPPResult.SUCCESS_PENDING, response.getResult().getCode());

			// -- Output all of the response attributes
			System.out.println("orgCreatePending: Org Create Response = [" + response + "]\n\n");

			System.out.println("orgCreatePending: Get pending action poll message");

			this.session.setPollOp(EPPSession.OP_REQ);
			thePollResponse = this.session.sendPoll();

			Assert.assertTrue(thePollResponse instanceof com.verisign.epp.codec.org.EPPOrgPendActionMsg);

			System.out.println("orgCreatePending: Org Pending Action Response = [" + thePollResponse + "]\n\n");

			this.session.setPollOp(EPPSession.OP_ACK);
			this.session.setMsgID(thePollResponse.getMsgQueue().getId());
			thePollResponse = this.session.sendPoll();
		}
		catch (EPPCommandException e) {
			handleException(e);
		}

		printEnd("orgCreatePending");
	}

	/**
	 * Unit test of <code>EPPOrg.sendDelete()</code> to delete a specific org
	 * identifier.
	 */
	public void orgDelete() {
		printStart("orgDelete");

		EPPResponse response;

		try {
			System.out.println("\n----------------------------------------------------------------");

			System.out.println("orgDelete: Delete res1523.");

			this.org.setTransId("ABC-12345");

			this.org.addOrgId("res1523");

			response = this.org.sendDelete();

			// -- Output all of the response attributes
			System.out.println("orgDelete: Org Delete Response = [" + response + "]\n\n");
		}
		catch (EPPCommandException e) {
			handleException(e);
		}

		printEnd("orgDelete");
	}

	/**
	 * Unit test of <code>EPPOrg.sendUpdate()</code> to create a org.
	 */
	public void orgUpdate() {
		printStart("orgUpdate");

		EPPResponse response;

		try {
			System.out.println("\n----------------------------------------------------------------");

			System.out.println("orgUpdate: Update res1523.");

			this.org.setTransId("ABC-12345");

			this.org.addOrgId("res1523");

			// Add section
			this.org.addAddContact(new EPPOrgContact("sh8013", EPPOrgContact.Type.TECHNICAL));
			this.org.addAddRole(new EPPOrgRole("privacyproxy", EPPOrgRoleStatus.clientLinkProhibited, null));
			this.org.addAddStatus(EPPOrgStatus.clientLinkProhibited);

			// Delete section
			this.org.addAddContact(new EPPOrgContact("sh8014", EPPOrgContact.Type.BILLING));
			this.org.addAddRole(new EPPOrgRole("reseller"));
			this.org.addRemStatus(EPPOrgStatus.clientDeleteProhibited);

			this.org.setParentId("1523res");

			EPPOrgAddress theAddress = new EPPOrgAddress();
			theAddress.setStreets("124 Example Dr.", "Suite 200");
			theAddress.setCity("Dulles");
			theAddress.setStateProvince("VA");
			theAddress.setPostalCode("20166-6503");
			theAddress.setCountry("US");

			EPPOrgPostalDefinition thePostalDef = new EPPOrgPostalDefinition(EPPOrgPostalDefinition.Type.INT);
			thePostalDef.setAddress(theAddress);
			this.org.addPostalInfo(thePostalDef);

			this.org.setVoice("+1.7034444444");

			response = this.org.sendUpdate();

			// -- Output all of the response attributes
			System.out.println("orgUpdate: Org Update Response = [" + response + "]\n\n");
		}
		catch (EPPCommandException e) {
			handleException(e);
		}

		printEnd("orgUpdate");
	}

	/**
	 * Unit test of <code>EPPSession.initSession</code>. The session attribute is
	 * initialized with the attributes defined in the EPP sample files.
	 */
	private void initSession() {
		printStart("initSession");

		// Set attributes for initSession
		session.setClientID(Environment.getProperty("EPP.Test.clientId", "ClientX"));
		session.setPassword(Environment.getProperty("EPP.Test.password", "foo-BAR2"));
		session.setTransId("ABC-12345-XYZ");
		session.setVersion("1.0");
		session.setLang("en");

		// Initialize the session
		try {
			session.initSession();
		}

		catch (EPPCommandException e) {
			EPPResponse response = session.getResponse();

			// Is a server specified error?
			if ((response != null) && (!response.isSuccess())) {
				Assert.fail("Server Error : " + response);
			}
			else {
				e.printStackTrace();

				Assert.fail("initSession Error : " + e);
			}
		}

		printEnd("initSession");
	}

	/**
	 * Unit test of <code>EPPSession.endSession</code>. The session with the EPP
	 * Server will be terminated.
	 */
	private void endSession() {
		printStart("endSession");

		session.setTransId("ABC-12345-XYZ");

		// End the session
		try {
			session.endSession();
		}

		catch (EPPCommandException e) {
			EPPResponse response = session.getResponse();

			// Is a server specified error?
			if ((response != null) && (!response.isSuccess())) {
				Assert.fail("Server Error : " + response);
			}

			else // Other error
			{
				e.printStackTrace();

				Assert.fail("initSession Error : " + e);
			}
		}

		printEnd("endSession");
	}

	/**
	 * JUNIT <code>setUp</code> method, which sets the default client Id to
	 * "theRegistrar".
	 */
	protected void setUp() {
		try {
			String theSessionClassName = System.getProperty("EPP.SessionClass");

			if (theSessionClassName != null) {
				try {
					Class theSessionClass = Class.forName(theSessionClassName);

					if (!EPPSession.class.isAssignableFrom((theSessionClass))) {
						Assert.fail(theSessionClassName + " is not a subclass of EPPSession");
					}

					session = (EPPSession) theSessionClass.getDeclaredConstructor().newInstance();
				}
				catch (Exception ex) {
					Assert.fail("Exception instantiating EPP.SessionClass value " + theSessionClassName + ": " + ex);
				}
			}
			else {
				session = new EPPSession();
			}

		}

		catch (Exception e) {
			e.printStackTrace();

			Assert.fail("Error initializing the session: " + e);
		}

		initSession();

		org = new EPPOrg(session);
	}

	/**
	 * JUNIT <code>tearDown</code>, which currently does nothing.
	 */
	protected void tearDown() {
		endSession();
	}

	/**
	 * JUNIT <code>suite</code> static method, which returns the tests associated
	 * with <code>EPPOrgTst</code>.
	 *
	 * @return Test suite
	 */
	public static Test suite() {
		TestSuite suite = new TestSuite(EPPOrgTst.class);

		String theConfigFileName = System.getProperty("EPP.ConfigFile");
		if (theConfigFileName != null)
			configFileName = theConfigFileName;

		try {
			app.initialize(configFileName);
		}

		catch (EPPCommandException e) {
			e.printStackTrace();

			Assert.fail("Error initializing the EPP Application: " + e);
		}

		return suite;
	}

	/**
	 * Handle an <code>EPPCommandException</code>, which can be either a server
	 * generated error or a general exception. If the exception was caused by a
	 * server error, "Server Error :&lt;Response XML&gt;" will be specified. If
	 * the exception was caused by a general algorithm error, "General Error
	 * :&lt;Exception Description&gt;" will be specified.
	 *
	 * @param aException
	 *           Exception thrown during test
	 */
	public void handleException(Exception aException) {
		EPPResponse response = session.getResponse();

		aException.printStackTrace();

		// Is a server specified error?
		if ((response != null) && (!response.isSuccess())) {
			Assert.fail("Server Error : " + response);
		}

		else {
			Assert.fail("General Error : " + aException);
		}
	}

	/**
	 * Unit test main, which accepts the following system property options: <br>
	 * 
	 * <ul>
	 * <li>iterations Number of unit test iterations to run</li>
	 * <li>validate Turn XML validation on (<code>true</code>) or off (
	 * <code>false</code>). If validate is not specified, validation will be off.
	 * </li>
	 * </ul>
	 * 
	 *
	 * @param args
	 *           Command line arguments
	 */
	public static void main(String[] args) {
		// Override the default configuration file name?
		if (args.length > 0) {
			configFileName = args[0];
		}

		// Number of Threads
		int numThreads = 1;

		String threadsStr = System.getProperty("threads");

		if (threadsStr != null) {
			numThreads = Integer.parseInt(threadsStr);

			// Run test suite in multiple threads?
		}

		if (numThreads > 1) {
			// Spawn each thread passing in the Test Suite
			for (int i = 0; i < numThreads; i++) {
				TestThread thread = new TestThread("EPPSessionTst Thread " + i, EPPOrgTst.suite());

				thread.start();
			}
		}

		else { // Single threaded mode.
			junit.textui.TestRunner.run(EPPOrgTst.suite());
		}

		try {
			app.endApplication();
		}

		catch (EPPCommandException e) {
			e.printStackTrace();

			Assert.fail("Error ending the EPP Application: " + e);
		}
	}

	/**
	 * Print the start of a test with the <code>Thread</code> name if the current
	 * thread is a <code>TestThread</code>.
	 *
	 * @param aTest
	 *           name for the test
	 */
	private void printStart(String aTest) {
		if (Thread.currentThread() instanceof TestThread) {
			System.out.print(Thread.currentThread().getName() + ", iteration " + iteration + ": ");

			cat.info(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aTest + " Start");
		}

		System.out.println("Start of " + aTest);

		System.out.println("****************************************************************\n");
	}

	/**
	 * Print the end of a test with the <code>Thread</code> name if the current
	 * thread is a <code>TestThread</code>.
	 *
	 * @param aTest
	 *           name for the test
	 */
	private void printEnd(String aTest) {
		System.out.println("****************************************************************");

		if (Thread.currentThread() instanceof TestThread) {
			System.out.print(Thread.currentThread().getName() + ", iteration " + iteration + ": ");

			cat.info(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aTest + " End");
		}

		System.out.println("End of " + aTest);

		System.out.println("\n");
	}

	/**
	 * Print message
	 *
	 * @param aMsg
	 *           message to print
	 */
	private void printMsg(String aMsg) {
		if (Thread.currentThread() instanceof TestThread) {
			System.out.println(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);

			cat.info(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);
		}

		else {
			System.out.println(aMsg);

			cat.info(aMsg);
		}
	}

	/**
	 * Print error message
	 *
	 * @param aMsg
	 *           errpr message to print
	 */
	private void printError(String aMsg) {
		if (Thread.currentThread() instanceof TestThread) {
			System.err.println(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);

			cat.error(Thread.currentThread().getName() + ", iteration " + iteration + ": " + aMsg);
		}

		else {
			System.err.println(aMsg);

			cat.error(aMsg);
		}
	}
}
