/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.orgext;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * <code>EPPOrgExtInfData</code> is used in the extension to the info response
 * to define the org identifiers.
 */
public class EPPOrgExtInfData implements EPPCodecComponent {

	/**
	 * Log4j category for logging
	 */
	private static Logger cat = Logger.getLogger(EPPOrgExtInfData.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for <code>EPPOrgExtInfData</code>.
	 */
	public static final String ELM_LOCALNAME = "infData";

	/**
	 * XML root tag for <code>EPPOrgExtInfData</code>.
	 */
	public static final String ELM_NAME = EPPOrgExtFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	private List<EPPOrgExtId> orgIds = new ArrayList<EPPOrgExtId>();

	/**
	 * Default constructor for <code>EPPOrgExtInfData</code>.
	 */
	public EPPOrgExtInfData() {
	}

	/**
	 * Constructor for <code>EPPOrgExtInfData</code> that takes a single
	 * organization id.
	 * 
	 * @param aOrgId
	 *           Organization identifier
	 */
	public EPPOrgExtInfData(EPPOrgExtId aOrgId) {
		this.addOrgId(aOrgId);
	}

	/**
	 * Constructor for <code>EPPOrgExtInfData</code> that takes a list of
	 * organization ids.
	 * 
	 * @param aOrgIds
	 *           Organization identifiers
	 */
	public EPPOrgExtInfData(List<EPPOrgExtId> aOrgIds) {
		this.setOrgIds(aOrgIds);
	}

	/**
	 * Are there any organization ids?
	 * 
	 * @return <code>true</code> if there organization ids; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasOrgIds() {
		return !this.orgIds.isEmpty();
	}

	/**
	 * Returns the list of organization ids. An empty list indicates that there
	 * are no organization ids.
	 * 
	 * @return List of organization ids.
	 */
	public List<EPPOrgExtId> getOrgIds() {
		return this.orgIds;
	}

	/**
	 * Sets the organization ids.
	 * 
	 * @param aOrgIds
	 *           The list of organization ids. Set to <code>null</code> or an
	 *           empty list to indicate that there are no organization ids.
	 */
	public void setOrgIds(List<EPPOrgExtId> aOrgIds) {
		if (aOrgIds == null) {
			this.orgIds = new ArrayList<EPPOrgExtId>();
		}
		else {
			this.orgIds = aOrgIds;
		}
	}

	/**
	 * Add a organization id to the list of organization ids.
	 * 
	 * @param aOrgId
	 *           organization id to add to the list of organization ids.
	 */
	public void addOrgId(EPPOrgExtId aOrgId) {
		this.orgIds.add(aOrgId);
	}

	/**
	 * encode instance into a DOM element tree. A DOM Document is passed as an
	 * argument and functions as a factory for DOM objects. The root element
	 * associated with the instance is created and each instance attribute is
	 * appended as a child node.
	 * 
	 * @param aDocument
	 *           DOM Document, which acts is an Element factory
	 * 
	 * @return Element Root element associated with the object
	 * 
	 * @exception EPPEncodeException
	 *               Error encoding <code>EPPOrgExtInfData</code>
	 */
	public Element encode(Document aDocument) throws EPPEncodeException {

		// Create root element
		Element root = aDocument.createElementNS(EPPOrgExtFactory.NS, ELM_NAME);

		// Org Identifiers
		EPPUtil.encodeCompList(aDocument, root, this.orgIds);

		return root;
	}

	/**
	 * decode a DOM element tree to initialize the instance attributes. The
	 * <code>aElement</code> argument represents the root DOM element and is used
	 * to traverse the DOM nodes for instance attribute values.
	 * 
	 * @param aElement
	 *           <code>Element</code> to decode
	 * 
	 * @exception EPPDecodeException
	 *               Error decoding <code>Element</code>
	 */
	public void decode(Element aElement) throws EPPDecodeException {

		// OrgIds
		this.orgIds = EPPUtil.decodeCompList(aElement, EPPOrgExtFactory.NS, EPPOrgExtId.ELM_NAME, EPPOrgExtId.class);
	}

	/**
	 * clone an <code>EPPCodecComponent</code>.
	 * 
	 * @return clone of concrete <code>EPPOrgExt</code>
	 * 
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPOrgExtInfData clone = (EPPOrgExtInfData) super.clone();

		// OrgIds
		clone.orgIds = new ArrayList<EPPOrgExtId>(this.orgIds.size());
		for (EPPOrgExtId orgId : this.orgIds)
			clone.orgIds.add((EPPOrgExtId) orgId.clone());

		return clone;
	}

	/**
	 * Implementation of <code>Object.toString</code>, which will result in an
	 * indented XML <code>String</code> representation of the concrete
	 * <code>EPPCodecComponent</code>.
	 * 
	 * @return Indented XML <code>String</code> if successful; <code>ERROR</code>
	 *         otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Compare an instance of <code>EPPOrgExt</code> with this instance.
	 * 
	 * @param aObject
	 *           Object to compare with.
	 * 
	 * @return <code>true</code> if equal; <code>false</code> otherwise.
	 */
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPOrgExtInfData)) {
			return false;
		}

		EPPOrgExtInfData other = (EPPOrgExtInfData) aObject;

		// OrgIds
		if (!EqualityUtil.equals(this.orgIds, other.orgIds)) {
			cat.error("EPPOrgExtInfData.equals(): orgIds not equal");
			return false;
		}

		return true;
	}

	/**
	 * Returns the XML namespace associated with the
	 * <code>EPPCodecComponent</code>.
	 * 
	 * @return XML namespace for the <code>EPPCodecComponent</code>.
	 */
	@Override
	public String getNamespace() {
		return EPPOrgExtFactory.NS;
	}

}