/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/

package com.verisign.epp.codec.orgext;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPOrgExtCreate} is used in the extension to the create command
 * to define the org identifiers.
 */
public class EPPOrgExtCreate implements EPPCodecComponent {

	/**
	 * Log4j category for logging
	 */
	private static Logger cat = Logger.getLogger(EPPOrgExtCreate.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for {@code EPPOrgExtCreate}.
	 */
	public static final String ELM_LOCALNAME = "create";

	/**
	 * XML root tag for {@code EPPOrgExtCreate}.
	 */
	public static final String ELM_NAME = EPPOrgExtFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * Organization identifiers list to create.
	 */
	private List<EPPOrgExtId> orgIds = new ArrayList<EPPOrgExtId>();

	/**
	 * Default constructor for {@code EPPOrgExtCreate}.
	 */
	public EPPOrgExtCreate() {
	}

	/**
	 * Constructor for {@code EPPOrgExtCreate} that takes a single
	 * organization id.
	 * 
	 * @param aOrgId
	 *           Organization identifier
	 */
	public EPPOrgExtCreate(EPPOrgExtId aOrgId) {
		this.addOrgId(aOrgId);
	}

	/**
	 * Constructor for {@code EPPOrgExtCreate} that takes a list of
	 * organization ids.
	 * 
	 * @param aOrgIds
	 *           Organization identifiers
	 */
	public EPPOrgExtCreate(List<EPPOrgExtId> aOrgIds) {
		this.setOrgIds(aOrgIds);
	}

	/**
	 * Are there any organization ids?
	 * 
	 * @return {@code true} if there organization ids; {@code false}
	 *         otherwise.
	 */
	public boolean hasOrgIds() {
		return !this.orgIds.isEmpty();
	}

	/**
	 * Returns the list of organization ids. An empty list indicates that there
	 * are no organization ids.
	 * 
	 * @return List of organization ids.
	 */
	public List<EPPOrgExtId> getOrgIds() {
		return this.orgIds;
	}

	/**
	 * Sets the organization ids.
	 * 
	 * @param aOrgIds
	 *           The list of organization ids. Set to {@code null} or an
	 *           empty list to indicate that there are no organization ids.
	 */
	public void setOrgIds(List<EPPOrgExtId> aOrgIds) {
		if (aOrgIds == null) {
			this.orgIds = new ArrayList<EPPOrgExtId>();
		}
		else {
			this.orgIds = aOrgIds;
		}
	}

	/**
	 * Add a organization id to the list of organization ids.
	 * 
	 * @param aOrgId
	 *           organization id to add to the list of organization ids.
	 */
	public void addOrgId(EPPOrgExtId aOrgId) {
		this.orgIds.add(aOrgId);
	}

	/**
	 * encode instance into a DOM element tree. A DOM Document is passed as an
	 * argument and functions as a factory for DOM objects. The root element
	 * associated with the instance is created and each instance attribute is
	 * appended as a child node.
	 * 
	 * @param aDocument
	 *           DOM Document, which acts is an Element factory
	 * 
	 * @return Element Root element associated with the object
	 * 
	 * @exception EPPEncodeException
	 *               Error encoding {@code EPPOrgExtCreate}
	 */
	public Element encode(Document aDocument) throws EPPEncodeException {

		// Check required attributes
		if (!this.hasOrgIds()) {
			throw new EPPEncodeException("Empty orgIds in EPPOrgExtCreate");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPOrgExtFactory.NS, ELM_NAME);

		// Org Identifiers
		EPPUtil.encodeCompList(aDocument, root, this.orgIds);

		return root;
	}

	/**
	 * decode a DOM element tree to initialize the instance attributes. The
	 * {@code aElement} argument represents the root DOM element and is used
	 * to traverse the DOM nodes for instance attribute values.
	 * 
	 * @param aElement
	 *           {@code Element} to decode
	 * 
	 * @exception EPPDecodeException
	 *               Error decoding {@code Element}
	 */
	public void decode(Element aElement) throws EPPDecodeException {

		// OrgIds
		this.orgIds = EPPUtil.decodeCompList(aElement, EPPOrgExtFactory.NS, EPPOrgExtId.ELM_NAME, EPPOrgExtId.class);
	}

	/**
	 * clone an {@code EPPCodecComponent}.
	 * 
	 * @return clone of concrete {@code EPPOrgExt}
	 * 
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPOrgExtCreate clone = (EPPOrgExtCreate) super.clone();

		// OrgIds
		clone.orgIds = new ArrayList<EPPOrgExtId>(this.orgIds.size());
		for (EPPOrgExtId orgId : this.orgIds)
			clone.orgIds.add((EPPOrgExtId) orgId.clone());

		return clone;
	}

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 * 
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Compare an instance of {@code EPPOrgExt} with this instance.
	 * 
	 * @param aObject
	 *           Object to compare with.
	 * 
	 * @return {@code true} if equal; {@code false} otherwise.
	 */
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPOrgExtCreate)) {
			return false;
		}

		EPPOrgExtCreate other = (EPPOrgExtCreate) aObject;

		// OrgIds
		if (!EqualityUtil.equals(this.orgIds, other.orgIds)) {
			cat.error("EPPOrgExtCreate.equals(): orgIds not equal");
			return false;
		}

		return true;
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 * 
	 * @return XML namespace for the {@code EPPCodecComponent}. 
	 */
	@Override
	public String getNamespace() {
		return EPPOrgExtFactory.NS;
	}
	
}
