/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.org;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPInfoCmd;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * Command used to retrieve information associated with an org object.
 * 
 * @see com.verisign.epp.codec.org.EPPOrgInfoResp
 */
public class EPPOrgInfoCmd extends EPPInfoCmd {

	/**
	 * Log4j category for logging
	 */
	private static Logger cat = Logger.getLogger(EPPOrgInfoCmd.class
			.getName(), EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for <code>EPPOrgInfoCmd</code>.
	 */
	public static final String ELM_LOCALNAME = "info";

	/**
	 * XML root tag for <code>EPPOrgInfoCmd</code>.
	 */
	public static final String ELM_NAME = EPPOrgMapFactory.NS_PREFIX + ":"
			+ ELM_LOCALNAME;

	/**
	 * XML local name for the org identifier element
	 */
	private static final String ELM_ORG_ID = "id";

	/**
	 * The org identifier of the org.
	 */
	private String orgId;

	/**
	 * <code>EPPOrgInfoCmd</code> default constructor.
	 */
	public EPPOrgInfoCmd() {
	}

	/**
	 * <code>EPPOrgInfoCmd</code> constructor that only takes the client
	 * transaction identifier
	 * 
	 * @param aTransId
	 *            Transaction Id associated with command.
	 */
	public EPPOrgInfoCmd(String aTransId) {
		super(aTransId);
	}

	/**
	 * <code>EPPOrgInfoCmd</code> constructor that takes the transaction
	 * identifier and the org identifier.
	 * 
	 * @param aTransId
	 *            Transaction Id associated with command. Set to
	 *            <code>null</code> if a client transaction identifier is not
	 *            desired.
	 * @param aOrgId
	 *            Org identifier
	 */
	public EPPOrgInfoCmd(String aTransId, String aOrgId) {
		super(aTransId);
		this.orgId = aOrgId;
	}

	/**
	 * Gets the EPP command Namespace associated with
	 * <code>EPPOrgInfoCmd</code>.
	 * 
	 * @return <code>EPPOrgMapFactory.NS</code>
	 */
	public String getNamespace() {
		return EPPOrgMapFactory.NS;
	}

	/**
	 * Gets the org identifier.
	 * 
	 * @return The org identifier if defined;<code>null</code> otherwise.
	 */
	public String getOrgId() {
		return this.orgId;
	}

	/**
	 * Sets the org identifier.
	 * 
	 * @param aOrgId
	 *            The org identifier
	 */
	public void setOrgId(String aOrgId) {
		this.orgId = aOrgId;
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * <code>EPPOrgInfoCmd</code> instance.
	 * 
	 * @param aDocument
	 *            DOM Document that is being built. Used as an Element factory.
	 * 
	 * @return Element Root DOM Element representing the
	 *         <code>EPPOrgInfoCmd</code> instance.
	 * 
	 * @exception EPPEncodeException
	 *                Unable to encode <code>EPPOrgInfoCmd</code> instance.
	 */
	protected Element doEncode(Document aDocument) throws EPPEncodeException {

		// Check required attributes
		if (this.orgId == null) {
			throw new EPPEncodeException(
					"Undefined orgId in EPPOrgInfoCmd");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPOrgMapFactory.NS,
				ELM_NAME);

		// Org Identifier
		EPPUtil.encodeString(aDocument, root, this.orgId,
				EPPOrgMapFactory.NS, EPPOrgMapFactory.NS_PREFIX + ":"
						+ ELM_ORG_ID);

		return root;
	}

	/**
	 * Decode the <code>EPPOrgInfoCmd</code> attributes from the aElement
	 * DOM Element tree.
	 * 
	 * @param aElement
	 *            Root DOM Element to decode <code>EPPOrgInfoCmd</code>
	 *            from.
	 * 
	 * @exception EPPDecodeException
	 *                Unable to decode aElement
	 */
	protected void doDecode(Element aElement) throws EPPDecodeException {

		// Org Identifier
		this.orgId = EPPUtil.decodeString(aElement,
				EPPOrgMapFactory.NS, ELM_ORG_ID);
	}

	/**
	 * Compare an instance of <code>EPPOrgInfoCmd</code> with this
	 * instance.
	 * 
	 * @param aObject
	 *            Object to compare with.
	 * 
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise.
	 */
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPOrgInfoCmd)) {
			return false;
		}

		EPPOrgInfoCmd other = (EPPOrgInfoCmd) aObject;

		// Org Identifier
		if (!EqualityUtil.equals(this.orgId, other.orgId)) {
			cat.error("EPPOrgInfoCmd.equals(): orgId not equal");
			return false;
		}

		return true;
	}

	/**
	 * Clone <code>EPPOrgInfoCmd</code>.
	 * 
	 * @return Deep copy clone of <code>EPPOrgInfoCmd</code>
	 * 
	 * @exception CloneNotSupportedException
	 *                standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPOrgInfoCmd clone = (EPPOrgInfoCmd) super.clone();

		return clone;
	}

	/**
	 * Implementation of <code>Object.toString</code>, which will result in an
	 * indented XML <code>String</code> representation of the concrete
	 * <code>EPPCodecComponent</code>.
	 * 
	 * @return Indented XML <code>String</code> if successful;
	 *         <code>ERROR</code> otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

}
