/***********************************************************
Copyright (C) 2018 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
 ***********************************************************/
package com.verisign.epp.codec.org;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPCreateCmd;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * Command used to create an org object.
 * 
 * @see com.verisign.epp.codec.org.EPPOrgCreateResp
 */
public class EPPOrgCreateCmd extends EPPCreateCmd {

	/**
	 * Log4j category for logging
	 */
	private static Logger cat = Logger.getLogger(EPPOrgCreateCmd.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * XML local name for <code>EPPOrgCreateCmd</code>.
	 */
	public static final String ELM_LOCALNAME = "create";

	/**
	 * XML root tag for <code>EPPOrgCreateCmd</code>.
	 */
	public static final String ELM_NAME = EPPOrgMapFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * XML local name for the org identifier element
	 */
	private static final String ELM_ORG_ID = "id";

	/**
	 * XML local name for the status (<code>status</code>) element.
	 */
	private final static String ELM_STATUS = "status";

	/**
	 * XML local name for the OPTIONAL parent identifier (<code>parentId</code>)
	 * element.
	 */
	private final static String ELM_PARENT_ID = "parentId";

	/**
	 * XML local name for the OPTIONAL voice number (<code>voice</code>) element.
	 */
	private final static String ELM_VOICE = "voice";

	/**
	 * XML local name for the OPTIONAL fax number (<code>fax</code>) element.
	 */
	private final static String ELM_FAX = "fax";

	/**
	 * XML local name for the email (<code>email</code>) element.
	 */
	private final static String ELM_EMAIL = "email";

	/**
	 * XML local name for the OPTIONAL URL (<code>url</code>) element.
	 */
	private final static String ELM_URL = "url";

	/**
	 * XML extension "x" attribute value for voice or fax elements.
	 */
	private final static String ATTR_EXT = "x";

	/**
	 * The org identifier of the org of a sponsoring registrar.
	 */
	private String orgId;

	/**
	 * Organization roles
	 */
	private List<EPPOrgRole> roles = new ArrayList<EPPOrgRole>();

	/**
	 * Organization statuses
	 */
	private List<EPPOrgStatus> statuses = new ArrayList<EPPOrgStatus>();

	/**
	 * OPTIONAL Parent Identifier
	 */
	private String parentId;

	/**
	 * One or two postal information elements, represented by the
	 * {@link EPPOrgPostalDefinition} class.
	 */
	private List<EPPOrgPostalDefinition> postalInfo = new ArrayList<EPPOrgPostalDefinition>();

	/**
	 * OPTIONAL Voice number
	 */
	private String voice;

	/**
	 * OPTIONAL Voice Extension
	 */
	private String voiceExt;

	/**
	 * OPTIONAL FAX number
	 */
	private String fax;

	/**
	 * OPTIONAL FAX Extension
	 */
	private String faxExt;

	/**
	 * Email Address
	 */
	private String email;

	/**
	 * URL
	 */
	private String url;

	/**
	 * Org contacts
	 */
	private List<EPPOrgContact> contacts = new ArrayList<EPPOrgContact>();

	/**
	 * <code>EPPOrgCreateCmd</code> default constructor with the default form of
	 * <code>Form.LIST_FORM</code>.
	 */
	public EPPOrgCreateCmd() {
	}

	/**
	 * <code>EPPOrgCreateCmd</code> constructor that takes the client transaction
	 * identifier.
	 * 
	 * @param aTransId
	 *           Client transaction identifier.
	 */
	public EPPOrgCreateCmd(String aTransId) {
		super(aTransId);
	}

	/**
	 * Gets the org identifier.
	 * 
	 * @return The org identifier if defined;<code>null</code> otherwise.
	 */
	public String getOrgId() {
		return this.orgId;
	}

	/**
	 * Sets the org identifier.
	 * 
	 * @param aOrgId
	 *           The org identifier
	 */
	public void setOrgId(String aOrgId) {
		this.orgId = aOrgId;
	}

	/**
	 * Is there at least one role?
	 * 
	 * @return <code>true</code> if there at least one role; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasRole() {
		return !this.roles.isEmpty();
	}

	/**
	 * Gets the roles of the org.
	 * 
	 * @return List of roles of the org.
	 */
	public List<EPPOrgRole> getRoles() {
		return this.roles;
	}

	/**
	 * Sets the roles of the org.
	 * 
	 * @param aRoles
	 *           Organization roles
	 */
	public void setRoles(List<EPPOrgRole> aRoles) {
		if (aRoles == null) {
			this.roles = new ArrayList<EPPOrgRole>();
		}
		else {
			this.roles = aRoles;
		}
	}

	/**
	 * Add a role to list of org roles.
	 * 
	 * @param aRole
	 *           Org role to add.
	 */
	public void addRole(EPPOrgRole aRole) {
		this.roles.add(aRole);
	}

	/**
	 * Is there at least one status?
	 * 
	 * @return <code>true</code> if there at least one status; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasStatus() {
		return !this.statuses.isEmpty();
	}

	/**
	 * Gets the statuses of the org.
	 * 
	 * @return List of statuses of the org.
	 */
	public List<EPPOrgStatus> getStatuses() {
		return this.statuses;
	}

	/**
	 * Sets the statuses of the org.
	 * 
	 * @param aStatuses
	 *           Organization statuses
	 */
	public void setStatuses(List<EPPOrgStatus> aStatuses) {
		if (aStatuses == null) {
			this.statuses = new ArrayList<EPPOrgStatus>();
		}
		else {
			this.statuses = aStatuses;
		}
	}

	/**
	 * Add a status to list of org statuses.
	 * 
	 * @param aStatus
	 *           Org status to add.
	 */
	public void addStatus(EPPOrgStatus aStatus) {
		this.statuses.add(aStatus);
	}

	/**
	 * Is the parentId defined?
	 * 
	 * @return <code>true</code> if the parentId is defined; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasParentId() {
		return (this.parentId != null ? true : false);
	}

	/**
	 * @return the parentId
	 */
	public String getParentId() {
		return this.parentId;
	}

	/**
	 * @param aParentId
	 *           the parentId to set
	 */
	public void setParentId(String aParentId) {
		this.parentId = aParentId;
	}

	/**
	 * Is there any postal information set?
	 * 
	 * @return <code>true</code> if there is at least one
	 *         {@link EPPOrgPostalDefinition} set in the postal information;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasPostalInfo() {
		if (this.postalInfo != null && !this.postalInfo.isEmpty()) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Adds a postal definition to the postal information for the org.
	 * 
	 * @param aPostalInfo
	 *           Postal definition to add to the postal information.
	 */
	public void addPostalInfo(EPPOrgPostalDefinition aPostalInfo) {
		this.postalInfo.add(aPostalInfo);
	}

	/**
	 * Gets the postal information for the org. There can be one or two
	 * {@link EPPOrgPostalDefinition} objects in the postal information list.
	 * 
	 * @return Postal information for the org
	 */
	public List<EPPOrgPostalDefinition> getPostalInfo() {
		return this.postalInfo;
	}

	/**
	 * Sets the postal information for the org. There can be one or two
	 * {@link EPPOrgPostalDefinition} objects in the postal information list.
	 * 
	 * @param aPostalInfo
	 *           Postal information for the org.
	 */
	public void setPostalInfo(List<EPPOrgPostalDefinition> aPostalInfo) {
		if (aPostalInfo == null) {
			this.postalInfo = new ArrayList<EPPOrgPostalDefinition>();
		}
		else {
			this.postalInfo = aPostalInfo;
		}
	}

	/**
	 * @return the voice
	 */
	public String getVoice() {
		return this.voice;
	}

	/**
	 * @param aVoice
	 *           the voice to set
	 */
	public void setVoice(String aVoice) {
		this.voice = aVoice;
	}

	/**
	 * @return the voiceExt
	 */
	public String getVoiceExt() {
		return this.voiceExt;
	}

	/**
	 * @param aVoiceExt
	 *           the voiceExt to set
	 */
	public void setVoiceExt(String aVoiceExt) {
		this.voiceExt = aVoiceExt;
	}

	/**
	 * @return the fax
	 */
	public String getFax() {
		return this.fax;
	}

	/**
	 * @param aFax
	 *           the fax to set
	 */
	public void setFax(String aFax) {
		this.fax = aFax;
	}

	/**
	 * @return the faxExt
	 */
	public String getFaxExt() {
		return this.faxExt;
	}

	/**
	 * @param aFaxExt
	 *           the faxExt to set
	 */
	public void setFaxExt(String aFaxExt) {
		this.faxExt = aFaxExt;
	}

	/**
	 * @return the email
	 */
	public String getEmail() {
		return this.email;
	}

	/**
	 * @param aEmail
	 *           the email to set
	 */
	public void setEmail(String aEmail) {
		this.email = aEmail;
	}

	/**
	 * Gets the URL of the website of the org.
	 * 
	 * @return URL of the website of the org.
	 */
	public String getUrl() {
		return this.url;
	}

	/**
	 * Sets the URL of the website of the org.
	 * 
	 * @param aUrl
	 *           URL of the website of the org.
	 */
	public void setUrl(String aUrl) {
		this.url = aUrl;
	}

	/**
	 * Is there any contacts set?
	 * 
	 * @return <code>true</code> if there is at least one {@link EPPOrgContact}
	 *         set; <code>false</code> otherwise.
	 */
	public boolean hasContacts() {
		if (this.contacts != null && !this.contacts.isEmpty()) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Adds a contact to the list of contacts.
	 * 
	 * @param aContact
	 *           Contact add to the list of contacts.
	 */
	public void addContact(EPPOrgContact aContact) {
		this.contacts.add(aContact);
	}

	/**
	 * Gets the list of org contacts.
	 * 
	 * @return The list of org contacts.
	 */
	public List<EPPOrgContact> getContacts() {
		return this.contacts;
	}

	/**
	 * Sets the list of org contacts.
	 * 
	 * @param aContacts
	 *           List of org contacts. If <code>null</code> will set to an empty
	 *           list.
	 */
	public void setContacts(List<EPPOrgContact> aContacts) {
		if (aContacts == null) {
			this.contacts = new ArrayList<EPPOrgContact>();
		}
		else {
			this.contacts = aContacts;
		}
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * <code>EPPOrgCreateCmd</code> instance.
	 * 
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 * 
	 * @return Element Root DOM Element representing the EPPOrgCreateCmd
	 *         instance.
	 * 
	 * @exception EPPEncodeException
	 *               Unable to encode EPPOrgCreateCmd instance.
	 */
	protected Element doEncode(Document aDocument) throws EPPEncodeException {

		Element theElm = null;
		Text theVal = null;

		// Check required attributes
		if (this.orgId == null) {
			throw new EPPEncodeException("Undefined orgId in EPPOrgCreateCmd");
		}
		if (this.roles.isEmpty()) {
			throw new EPPEncodeException("No roles set in EPPOrgInfoResp");
		}

		// Create root element
		Element root = aDocument.createElementNS(EPPOrgMapFactory.NS, ELM_NAME);

		// Org Identifier
		EPPUtil.encodeString(aDocument, root, this.orgId, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_ORG_ID);

		// Roles
		EPPUtil.encodeCompList(aDocument, root, this.roles);

		// Statuses
		EPPUtil.encodeList(aDocument, root, this.statuses, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_STATUS);

		// Parent Id
		EPPUtil.encodeString(aDocument, root, this.parentId, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_PARENT_ID);

		// Postal Information
		EPPUtil.encodeCompList(aDocument, root, this.postalInfo);

		// Voice and Voice Extension
		if (this.voice != null) {
			theElm = aDocument.createElementNS(EPPOrgMapFactory.NS, EPPOrgMapFactory.NS_PREFIX + ":" + ELM_VOICE);
			theVal = aDocument.createTextNode(this.voice);
			theElm.appendChild(theVal);

			if (this.voiceExt != null) {
				theElm.setAttribute(ATTR_EXT, this.voiceExt);
			}

			root.appendChild(theElm);
		}

		// Fax and Fax Extension
		if (this.fax != null) {
			theElm = aDocument.createElementNS(EPPOrgMapFactory.NS, EPPOrgMapFactory.NS_PREFIX + ":" + ELM_FAX);
			theVal = aDocument.createTextNode(this.fax);
			theElm.appendChild(theVal);

			if (this.faxExt != null) {
				theElm.setAttribute(ATTR_EXT, this.faxExt);
			}

			root.appendChild(theElm);
		}

		// Email
		EPPUtil.encodeString(aDocument, root, this.email, EPPOrgMapFactory.NS,
		      EPPOrgMapFactory.NS_PREFIX + ":" + ELM_EMAIL);

		// URL
		EPPUtil.encodeString(aDocument, root, this.url, EPPOrgMapFactory.NS, EPPOrgMapFactory.NS_PREFIX + ":" + ELM_URL);

		// Contacts
		EPPUtil.encodeCompList(aDocument, root, this.contacts);

		return root;
	}

	/**
	 * Decode the <code>EPPOrgCreateCmd</code> attributes from the aElement DOM
	 * Element tree.
	 * 
	 * @param aElement
	 *           Root DOM Element to decode <code>EPPOrgCreateCmd</code> from.
	 * 
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	protected void doDecode(Element aElement) throws EPPDecodeException {

		Element theElm = null;

		// Org Identifier
		this.orgId = EPPUtil.decodeString(aElement, EPPOrgMapFactory.NS, ELM_ORG_ID);

		// Roles
		this.roles = EPPUtil.decodeCompList(aElement, EPPOrgMapFactory.NS, EPPOrgRole.ELM_NAME, EPPOrgRole.class);

		// Statuses
		this.statuses = EPPUtil.decodeEnumList(aElement, EPPOrgMapFactory.NS, ELM_STATUS, EPPOrgStatus.class);

		// Parent Id
		this.parentId = EPPUtil.decodeString(aElement, EPPOrgMapFactory.NS, ELM_PARENT_ID);

		// Postal Information
		this.postalInfo = EPPUtil.decodeCompList(aElement, EPPOrgMapFactory.NS, EPPOrgPostalDefinition.ELM_NAME,
		      EPPOrgPostalDefinition.class);

		// Voice and Voice Extension
		theElm = EPPUtil.getElementByTagNameNS(aElement, EPPOrgMapFactory.NS, ELM_VOICE);
		if (theElm != null) {
			this.voice = EPPUtil.getTextContent(theElm);
			this.voiceExt = EPPUtil.decodeStringAttr(theElm, ATTR_EXT);
		}
		else {
			this.voice = null;
			this.voiceExt = null;
		}

		// Fax and Fax Extension
		theElm = EPPUtil.getElementByTagNameNS(aElement, EPPOrgMapFactory.NS, ELM_FAX);
		if (theElm != null) {
			this.fax = EPPUtil.getTextContent(theElm);
			this.faxExt = EPPUtil.decodeStringAttr(theElm, ATTR_EXT);
		}
		else {
			this.fax = null;
			this.faxExt = null;
		}

		// Email
		this.email = EPPUtil.decodeString(aElement, EPPOrgMapFactory.NS, ELM_EMAIL);

		// URL
		this.url = EPPUtil.decodeString(aElement, EPPOrgMapFactory.NS, ELM_URL);

		// Contacts
		this.contacts = EPPUtil.decodeCompList(aElement, EPPOrgMapFactory.NS, EPPOrgContact.ELM_NAME,
		      EPPOrgContact.class);
	}

	/**
	 * Clone <code>EPPOrgCreateCmd</code>.
	 * 
	 * @return clone of <code>EPPOrgCreateCmd</code>
	 * 
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPOrgCreateCmd clone = (EPPOrgCreateCmd) super.clone();

		// Roles
		clone.roles = (List) ((ArrayList) this.roles).clone();

		if (this.hasStatus()) {
			// Statuses
			clone.statuses = (List) ((ArrayList) this.statuses).clone();
		}

		if (this.hasPostalInfo()) {
			clone.postalInfo = (List) ((ArrayList) this.postalInfo).clone();
		}

		if (this.hasContacts()) {
			clone.contacts = (List) ((ArrayList) this.contacts).clone();
		}

		return clone;
	}

	/**
	 * Gets the EPP response type associated with <code>EPPOrgCreateCmd</code>.
	 * 
	 * @return <code>EPPOrgCreateCmd.ELM_NAME</code>
	 */
	public String getType() {
		return ELM_NAME;
	}

	/**
	 * Gets the EPP command namespace associated with
	 * <code>EPPOrgCreateCmd</code>.
	 * 
	 * @return <code>EPPOrgMapFactory.NS</code>
	 */
	public String getNamespace() {
		return EPPOrgMapFactory.NS;
	}

	/**
	 * Compare an instance of <code>EPPOrgCreateCmd</code> with this instance.
	 * 
	 * @param aObject
	 *           Object to compare with.
	 * 
	 * @return {@code true} if this object is the same as the aObject argument;
	 *         {@code false} otherwise
	 */
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPOrgCreateCmd)) {
			return false;
		}

		EPPOrgCreateCmd other = (EPPOrgCreateCmd) aObject;

		// Org Identifier
		if (!EqualityUtil.equals(this.orgId, other.orgId)) {
			cat.error("EPPOrgCreateCmd.equals(): orgId not equal");
			return false;
		}

		// Roles
		if (!EPPUtil.equalLists(this.roles, other.roles)) {
			cat.error("EPPOrgCreateCmd.equals(): roles not equal");
			return false;
		}

		// Statuses
		if (!EPPUtil.equalLists(this.statuses, other.statuses)) {
			cat.error("EPPOrgCreateCmd.equals(): statuses not equal");
			return false;
		}

		// Parent Id
		if (!EqualityUtil.equals(this.parentId, other.parentId)) {
			cat.error("EPPOrgCreateCmd.equals(): parentId not equal");
			return false;
		}

		// Parent Id
		if (!EqualityUtil.equals(this.parentId, other.parentId)) {
			cat.error("EPPOrgCreateCmd.equals(): parentId not equal");
			return false;
		}

		// Postal Information
		if (!EPPUtil.equalLists(this.postalInfo, other.postalInfo)) {
			cat.error("EPPOrgCreateCmd.equals(): postalInfo not equal");
			return false;
		}

		// Voice
		if (!EqualityUtil.equals(this.voice, other.voice)) {
			cat.error("EPPOrgCreateCmd.equals(): voice not equal");
			return false;
		}

		// Voice Ext
		if (!EqualityUtil.equals(this.voiceExt, other.voiceExt)) {
			cat.error("EPPOrgCreateCmd.equals(): voiceExt not equal");
			return false;
		}

		// Fax
		if (!EqualityUtil.equals(this.fax, other.fax)) {
			cat.error("EPPOrgCreateCmd.equals(): fax not equal");
			return false;
		}

		// Fax Ext
		if (!EqualityUtil.equals(this.faxExt, other.faxExt)) {
			cat.error("EPPOrgCreateCmd.equals(): faxExt not equal, this.faxExt = " + this.faxExt + ", other.faxExt = "
			      + other.faxExt);
			return false;
		}

		// Email
		if (!EqualityUtil.equals(this.email, other.email)) {
			cat.error("EPPOrgCreateCmd.equals(): email not equal");
			return false;
		}

		// Url
		if (!EqualityUtil.equals(this.url, other.url)) {
			cat.error("EPPOrgCreateCmd.equals(): url not equal");
			return false;
		}

		// Contacts
		if (!EPPUtil.equalLists(this.contacts, other.contacts)) {
			cat.error("EPPOrgCreateCmd.equals(): contacts not equal");
			return false;
		}

		return true;
	}

	/**
	 * Implementation of <code>Object.toString</code>, which will result in an
	 * indented XML <code>String</code> representation of the concrete
	 * <code>EPPCodecComponent</code>.
	 * 
	 * @return Indented XML <code>String</code> if successful; <code>ERROR</code>
	 *         otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

}
