/***********************************************************
Copyright (C) 2017 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 ***********************************************************/
package com.verisign.epp.codec.fee.v23;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * The command along with the optional phase, sub-phase, and custom name that
 * the fee is associated with.
 */
public class EPPFeeCommand implements EPPCodecComponent {

	/**
	 * Log4j category for logging
	 */
	private static Logger cat = Logger.getLogger(EPPFeeCommand.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/**
	 * Constant for the phase local name
	 */
	public static final String ELM_LOCALNAME = "command";

	/**
	 * Constant for the phase qualified name (prefix and local name)
	 */
	public static final String ELM_NAME = EPPFeeExtFactory.NS_PREFIX + ":" + ELM_LOCALNAME;

	/**
	 * OPTIONAL phase attribute name that defines the phase of the command.
	 */
	public static final String ATTR_PHASE = "phase";

	/**
	 * OPTIONAL subphase attribute name that defines the sub-phase of the
	 * command.
	 */
	public static final String ATTR_SUBPHASE = "subphase";

	/**
	 * name attribute that is set with the Command value.
	 */
	public static final String ATTR_NAME = "name";

	/**
	 * OPTIONAL customName attribute name used when the command is set to
	 * {@link Command#CUSTOM}.
	 */
	public static final String ATTR_CUSTOMNAME = "customName";

	/**
	 * OPTIONAL "phase" attribute.
	 */
	private Phase phase;

	/**
	 * OPTIONAL "subphase" attribute.
	 */
	private String subphase;

	/**
	 * Command value based on the {@link Command} enumeration. This value is
	 * required.
	 */
	private Command command;

	/**
	 * OPTIONAL customName attribute used when the command is set to
	 * {@link Command#CUSTOM}.
	 */
	private String customName;

	/**
	 * OPTIONAL period of the fee
	 */
	private EPPFeePeriod period;

	/**
	 * Default constructor. 
	 */
	public EPPFeeCommand() {
	}

	/**
	 * Create <code>EPPFeeCommand</code> instance with a defined command value.
	 * 
	 * @param aCommand
	 *           Command value.
	 */
	public EPPFeeCommand(Command aCommand) {
		this.command = aCommand;
	}

	/**
	 * Create <code>EPPFeeCommand</code> instance with a defined command value and period.
	 * 
	 * @param aCommand
	 *           Command value.
	 * @param aPeriod
	 * 			 Command period
	 */
	public EPPFeeCommand(Command aCommand, EPPFeePeriod aPeriod) {
		this.command = aCommand;
		this.period = aPeriod;
	}
	
	/**
	 * Create <code>EPPFeeCommand</code> instance with a defined command value
	 * and custom name value.
	 * 
	 * @param aCommand
	 *           Command value, which should be {@link Command#CUSTOM}.
	 * @param aCustomName
	 *           Custom name of the command.
	 */
	public EPPFeeCommand(Command aCommand, String aCustomName) {
		this.command = aCommand;
		this.customName = aCustomName;
	}

	/**
	 * Create <code>EPPFeeCommand</code> instance with a defined command and
	 * phase value.
	 * 
	 * @param aCommand
	 *           Command value.
	 * @param aPhase
	 *           Phase value.
	 */
	public EPPFeeCommand(Command aCommand, Phase aPhase) {
		this.command = aCommand;
		this.phase = aPhase;
	}

	/**
	 * Create <code>EPPFeeCommand</code> instance with a defined command, phase,
	 * and sub-phase value.
	 * 
	 * @param aCommand
	 *           Command value.
	 * @param aPhase
	 *           Phase value.
	 * @param aSubPhase
	 *           Sub-phase value
	 */
	public EPPFeeCommand(Command aCommand, Phase aPhase, String aSubPhase) {
		this.command = aCommand;
		this.phase = aPhase;
		this.subphase = aSubPhase;
	}

	/**
	 * Create <code>EPPFeeCommand</code> instance with a defined command value
	 * and custom name value.
	 * 
	 * @param aCommand
	 *           Command value, which should be {@link Command#CUSTOM}.
	 * @param aCustomName
	 *           Custom name of the command.
	 * @param aPhase
	 *           Phase value.
	 * @param aSubPhase
	 *           Sub-phase value
	 * @param aPeriod
	 *           Command period
	 */
	public EPPFeeCommand(Command aCommand, String aCustomName, Phase aPhase, String aSubPhase, EPPFeePeriod aPeriod) {
		this.command = aCommand;
		this.customName = aCustomName;
		this.phase = aPhase;
		this.subphase = aSubPhase;
		this.period = aPeriod;
	}
	
	/**
	 * Instantiate instance using an existing <code>EPPFeeCommand</code> instance.  This is equivalent to cloning the <code>EPPFeeCommand</code> instance.  
	 * 
	 * @param aCommand instance to initialize the <code>EPPFeeCommand</code> instance with.
	 */
	public EPPFeeCommand(EPPFeeCommand aCommand) {
		this.command = aCommand.getCommand();
		this.customName = aCommand.getCustomName();
		this.phase = aCommand.getPhase();
		this.subphase = aCommand.getSubPhase();

		if (this.period != null) {
			try {
				this.period = (EPPFeePeriod) aCommand.period.clone();
			}
			catch (CloneNotSupportedException e) {
			}
		}
	}

	/**
	 * Is the command defined?
	 * 
	 * @return <code>true</code> if the command is defined; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasCommand() {
		return (this.command != null ? true : false);
	}

	/**
	 * Gets the command value.
	 * 
	 * @return Command value if defined; <code>null</code> otherwise.
	 */
	public Command getCommand() {
		return this.command;
	}

	/**
	 * Sets the command value.
	 * 
	 * @param aCommand
	 *           Command value.
	 */
	public void setCommand(Command aCommand) {
		this.command = aCommand;
	}

	/**
	 * Is the custom name defined?
	 * 
	 * @return <code>true</code> if the custom name is defined;
	 *         <code>false</code> otherwise.
	 */
	public boolean hasCustomName() {
		return (this.customName != null ? true : false);
	}

	/**
	 * Gets the custom name value.
	 * 
	 * @return Custom name value if defined; <code>null</code> otherwise.
	 */
	public String getCustomName() {
		return this.customName;
	}

	/**
	 * Sets the custom name value when the command is set to
	 * {@link Command#CUSTOM}.
	 * 
	 * @param aCustomName
	 *           Custom name value.
	 */
	public void setCustomName(String aCustomName) {
		this.customName = aCustomName;
	}

	/**
	 * Is the phase defined?
	 * 
	 * @return <code>true</code> if the phase is defined; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasPhase() {
		return (this.phase != null ? true : false);
	}

	/**
	 * Gets the OPTIONAL phase value.
	 * 
	 * @return Phase value if defined; <code>null</code> otherwise.
	 */
	public Phase getPhase() {
		return this.phase;
	}

	/**
	 * Sets the OPTIONAL phase value.
	 * 
	 * @param aPhase
	 *           Phase value
	 */
	public void setPhase(Phase aPhase) {
		this.phase = aPhase;
	}

	/**
	 * Is the sub-phase defined?
	 * 
	 * @return <code>true</code> if the phase is defined; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasSubPhase() {
		return (this.subphase != null ? true : false);
	}

	/**
	 * Gets the OPTIONAL sub-phase value.
	 * 
	 * @return Sub-phase value if defined; <code>null</code> otherwise.
	 */
	public String getSubPhase() {
		return this.subphase;
	}

	/**
	 * Sets the OPTIONAL sub-phase value.
	 * 
	 * @param aSubPhase
	 *           Sub-phase value.
	 */
	public void setSubPhase(String aSubPhase) {
		this.subphase = aSubPhase;
	}

	/**
	 * Is the period defined?
	 * 
	 * @return <code>true</code> if the period is defined; <code>false</code>
	 *         otherwise.
	 */
	public boolean hasPeriod() {
		return (this.period != null ? true : false);
	}

	/**
	 * Gets the period value.
	 * 
	 * @return Period if defined; <code>null</code> otherwise.
	 */
	public EPPFeePeriod getPeriod() {
		return this.period;
	}

	/**
	 * Sets the optional period value.
	 * 
	 * @param aPeriod
	 *           Period value. Set to <code>null</code> to specify no period.
	 */
	public void setPeriod(EPPFeePeriod aPeriod) {
		this.period = aPeriod;
	}

	/**
	 * Clone <code>EPPFeeCommand</code> instance.
	 *
	 * @return clone of <code>EPPFeeCommand</code>
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	public Object clone() throws CloneNotSupportedException {
		EPPFeeCommand clone = null;

		clone = (EPPFeeCommand) super.clone();

		clone.phase = this.phase;
		clone.subphase = this.subphase;
		clone.command = this.command;
		clone.customName = this.customName;

		// Period
		if (this.period != null) {
			clone.period = (EPPFeePeriod) this.period.clone();
		}
		return clone;
	}

	/**
	 * Decode the <code>EPPFeeCommand</code> element aElement DOM Element tree.
	 *
	 * @param aElement
	 *           - Root DOM Element to decode <code>EPPFeeCommand</code> from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement
	 */
	public void decode(Element aElement) throws EPPDecodeException {
		// Phase
		String thePhase = aElement.getAttribute(ATTR_PHASE);
		if (thePhase != null && !thePhase.isEmpty()) {
			this.phase = Phase.getPhase(thePhase);
		}
		else {
			this.phase = null;
		}

		// Sub-phase
		String theSubPhase = aElement.getAttribute(ATTR_SUBPHASE);
		if (theSubPhase != null && !theSubPhase.isEmpty()) {
			this.subphase = theSubPhase;
		}
		else {
			this.subphase = null;
		}

		// Command
		String theName = aElement.getAttribute(ATTR_NAME);
		if (theName != null && !theName.isEmpty()) {
			this.command = Command.getCommand(theName);
		}
		else {
			this.command = null;
		}

		// Custom Name
		String theCustomName = aElement.getAttribute(ATTR_CUSTOMNAME);
		if (theCustomName != null && !theCustomName.isEmpty()) {
			this.customName = theCustomName;
		}
		else {
			this.customName = null;
		}

		// Period
		this.period = (EPPFeePeriod) EPPUtil.decodeComp(aElement, EPPFeeExtFactory.NS, EPPFeePeriod.ELM_NAME,
		      EPPFeePeriod.class);
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * <code>EPPFeeCommand</code> instance.
	 *
	 * @param aDocument
	 *           - DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element - Root DOM Element representing the
	 *         <code>EPPFeeCommand</code> instance.
	 *
	 * @exception EPPEncodeException
	 *               - Unable to encode <code>EPPFeeCommand</code> instance.
	 */
	public Element encode(Document aDocument) throws EPPEncodeException {

		if (aDocument == null) {
			throw new EPPEncodeException("aDocument is null" + " on in EPPFeeCommand.encode(Document)");
		}
		if (this.command == null) {
			throw new EPPEncodeException("command is null" + " on in EPPFeeCommand.encode(Document)");
		}

		Element root = aDocument.createElementNS(EPPFeeExtFactory.NS, ELM_NAME);

		// Phase
		if (this.hasPhase()) {
			root.setAttribute(ATTR_PHASE, this.phase.toString());
		}

		// Sub-phase
		if (this.hasSubPhase()) {
			root.setAttribute(ATTR_SUBPHASE, this.subphase);
		}

		// Command
		root.setAttribute(ATTR_NAME, this.command.toString());

		// Custom Name
		if (this.hasCustomName()) {
			root.setAttribute(ATTR_CUSTOMNAME, this.customName);
		}

		// Period
		EPPUtil.encodeComp(aDocument, root, this.period);

		return root;
	}

	/**
	 * Implements a deep <code>EPPFeeCommand</code> compare.
	 *
	 * @param aObject
	 *           <code>EPPFeeCommand</code> instance to compare with
	 *
	 * @return <code>true</code> if equal; <code>false</code> otherwise
	 */
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPFeeCommand)) {
			cat.error("EPPFeeCommand.equals(): " + aObject.getClass().getName() + "! instanceof "
			      + EPPFeeCommand.class.getName());
			return false;
		}

		EPPFeeCommand other = (EPPFeeCommand) aObject;

		// Phase
		if (!EqualityUtil.equals(this.phase, other.phase)) {
			cat.error("EPPFeeCommand.equals(): phase not equal");
			return false;
		}

		// Sub-phase
		if (!EqualityUtil.equals(this.subphase, other.subphase)) {
			cat.error("EPPFeeCommand.equals(): subphase not equal");
			return false;
		}

		// Command
		if (!EqualityUtil.equals(this.command, other.command)) {
			cat.error("EPPFeeCommand.equals(): command not equal");
			return false;
		}

		// Custom Name
		if (!EqualityUtil.equals(this.customName, other.customName)) {
			cat.error("EPPFeeCommand.equals(): customName not equal");
			return false;
		}

		// Period
		if (!EqualityUtil.equals(this.period, other.period)) {
			cat.error("EPPFeeCommand.equals(): period not equal");
			return false;
		}

		return true;
	}

	/**
	 * Implementation of <code>Object.toString</code>, which will result in an
	 * indented XML <code>String</code> representation of the concrete
	 * <code>EPPCodecComponent</code>.
	 *
	 * @return Indented XML <code>String</code> if successful; <code>ERROR</code>
	 *         otherwise.
	 */
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Returns the XML namespace associated with the <code>EPPCodecComponent</code>.
	 * 
	 * @return XML namespace for the <code>EPPCodecComponent</code>. 
	 */
	@Override
	public String getNamespace() {
		return EPPFeeExtFactory.NS;
	}

}
