/***********************************************************

Copyright (C) 2004 VeriSign, Inc.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-0107  USA

http://www.verisign.com/nds/naming/namestore/techdocs.html
***********************************************************/
package com.verisign.epp.codec.emailFwd;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.verisign.epp.codec.gen.EPPCodecComponent;
import com.verisign.epp.codec.gen.EPPDecodeException;
import com.verisign.epp.codec.gen.EPPEncodeException;
import com.verisign.epp.codec.gen.EPPUtil;
import com.verisign.epp.util.EPPCatFactory;
import com.verisign.epp.util.EqualityUtil;

/**
 * {@code EPPEmailFwdCheckResult} represents the result of an individual
 * emailFwd name ping. The attributes of {@code EPPEmailFwdCheckResult}
 * include the emailFwd name and a boolean value indicating if the emailFwd name
 * is already available. {@code emailFwd reason} must be set before
 * invoking {@code encode} if the available flag is set to
 * {@code false}.
 * 
 * @see com.verisign.epp.codec.emailFwd.EPPEmailFwdCheckResp
 */
public class EPPEmailFwdCheckResult implements EPPCodecComponent {
	/**
	 * Log4j category for logging
	 */
	private static Logger cat = Logger.getLogger(EPPEmailFwdCheckResult.class.getName(),
	      EPPCatFactory.getInstance().getFactory());

	/** XML attribute name for the {@code available} attribute. */
	private final static String ATTR_AVAIL = "avail";

	/** XML attribute name for the {@code emailFwd:reason} attribute. */
	private final static String ATTR_LANG = "lang";

	/** XML Element Name for the {@code names} attribute. */
	private final static String ELM_EMAILFWD_NAME = "emailFwd:name";

	/** XML Element Name for the {@code reason} attribute. */
	private final static String ELM_EMAILFWD_REASON = "emailFwd:reason";

	/** XML root tag name for {@code EPPEmailFwdCheckResult}. */
	final static String ELM_NAME = "emailFwd:cd";

	/** Default XML attribute value for emailFwd reason language. */
	private final static String VALUE_LANG = "en";

	/** Is the EmailFwd Name (name) available? */
	private boolean available;

	/** XML attribute value for the {@code lang} attribute. */
	private String language = "en";

	/** EmailFwd Name associated with result. */
	private String name;

	/** EmailFwd Reason to check. This is a {@code String}. */
	private String reason;

	/**
	 * Default constructor for {@code EPPEmailFwdCheckResult}. the defaults
	 * include the following: <br>
	 * <br>
	 *
	 * <ul>
	 * <li>name is set to {@code null}</li>
	 * <li>available is set to {@code true}</li>
	 * </ul>
	 *
	 * <br>
	 * The name must be set before invoking {@code encode}.
	 */
	public EPPEmailFwdCheckResult() {
		this.name = null;
		this.available = true;
	}

	/**
	 * Constructor for {@code EPPEmailFwdCheckResult} that includes the
	 * emailFwd name and the is available flag.
	 *
	 * @param aName
	 *           EmailFwd name associated with result
	 * @param aIsAvailable
	 *           Is the emailFwd name available?
	 */
	public EPPEmailFwdCheckResult(String aName, boolean aIsAvailable) {
		this.name = aName;
		this.available = aIsAvailable;
	}

	/**
	 * Clone {@code EPPEmailFwdCheckResult}.
	 *
	 * @return clone of {@code EPPEmailFwdCheckResult}
	 *
	 * @exception CloneNotSupportedException
	 *               standard Object.clone exception
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		EPPEmailFwdCheckResult clone = null;

		clone = (EPPEmailFwdCheckResult) super.clone();

		return clone;
	}

	/**
	 * Decode the {@code EPPEmailFwdCheckResult} attributes from the
	 * aElement DOM Element tree.
	 *
	 * @param aElement
	 *           Root DOM Element to decode {@code EPPEmailFwdCheckResult}
	 *           from.
	 *
	 * @exception EPPDecodeException
	 *               Unable to decode aElement.
	 */
	@Override
	public void decode(Element aElement) throws EPPDecodeException {
		// Available
		Element currElm = EPPUtil.getElementByTagNameNS(aElement, EPPEmailFwdMapFactory.NS, ELM_EMAILFWD_NAME);
		this.available = EPPUtil.decodeBooleanAttr(currElm, ATTR_AVAIL);

		// EmailFwd Name
		this.name = currElm.getFirstChild().getNodeValue();

		// EmailFwd Reason
		currElm = EPPUtil.getElementByTagNameNS(aElement, EPPEmailFwdMapFactory.NS, ELM_EMAILFWD_REASON);

		if (currElm != null) {
			this.reason = currElm.getFirstChild().getNodeValue();

			// Language
			String lang = currElm.getAttribute(ATTR_LANG);

			if (lang.length() > 0) {
				if (!lang.equals(VALUE_LANG)) {
					setLanguage(lang);
				}
			}
		}
	}

	/**
	 * Encode a DOM Element tree from the attributes of the
	 * {@code EPPEmailFwdCheckResult} instance.
	 *
	 * @param aDocument
	 *           DOM Document that is being built. Used as an Element factory.
	 *
	 * @return Element Root DOM Element representing the
	 *         {@code EPPEmailFwdCheckResult} instance.
	 *
	 * @exception EPPEncodeException
	 *               Unable to encode {@code EPPEmailFwdCheckResult}
	 *               instance.
	 */
	@Override
	public Element encode(Document aDocument) throws EPPEncodeException {
		// Validate state
		if (this.name == null) {
			throw new EPPEncodeException("name required attribute is not set");
		}

		// Name
		Element root = aDocument.createElementNS(EPPEmailFwdMapFactory.NS, ELM_NAME);

		// EmailFwd Name
		Element nameElm = aDocument.createElementNS(EPPEmailFwdMapFactory.NS, ELM_EMAILFWD_NAME);
		root.appendChild(nameElm);

		// Available
		EPPUtil.encodeBooleanAttr(nameElm, ATTR_AVAIL, this.available);

		// Name
		Text textNode = aDocument.createTextNode(this.name);
		nameElm.appendChild(textNode);

		// EmailFwd Reason
		if (this.reason != null) {
			Element reasonElm = aDocument.createElementNS(EPPEmailFwdMapFactory.NS, ELM_EMAILFWD_REASON);
			root.appendChild(reasonElm);

			// Language
			if (!this.language.equals(VALUE_LANG)) {
				reasonElm.setAttribute(ATTR_LANG, this.language);
			}

			// EmailFwd Reason
			Text aReason = aDocument.createTextNode(this.reason);
			reasonElm.appendChild(aReason);
		}

		return root;
	}

	/**
	 * Compare an instance of {@code EPPEmailFwdPingResp} with this
	 * instance.
	 *
	 * @param aObject
	 *           Object to compare with.
	 *
	 * @return {@code true} if the objects are equal; {@code false}
	 *         otherwise.
	 */
	@Override
	public boolean equals(Object aObject) {
		if (!(aObject instanceof EPPEmailFwdCheckResult)) {
			return false;
		}

		EPPEmailFwdCheckResult theComp = (EPPEmailFwdCheckResult) aObject;

		// Name
		if (!EqualityUtil.equals(this.name, theComp.name)) {
			cat.error("EPPEmailFwdCheckResult.equals(): name not equal");
			return false;
		}

		// Available
		if (!EqualityUtil.equals(this.available, theComp.available)) {
			cat.error("EPPEmailFwdCheckResult.equals(): available not equal");
			return false;
		}

		// Reason
		if (!EqualityUtil.equals(this.reason, theComp.reason)) {
			cat.error("EPPEmailFwdCheckResult.equals(): reason not equal");
			return false;
		}

		// Language
		if (!EqualityUtil.equals(this.language, theComp.language)) {
			cat.error("EPPEmailFwdCheckResult.equals(): language not equal");
			return false;
		}
		
		return true;
	}

	/**
	 * Gets emailFwd reason.
	 *
	 * @return String of emailFwd reason if defined; {@code null} otherwise.
	 */
	public String getEmailFwdReason() {
		return this.reason;
	}

	/**
	 * Sets emailFwd reason to check.
	 *
	 * @return String of emailFwd reason language if defined; {@code nuill}
	 *         otherwise.
	 */
	public String getLanguage() {
		return this.language;
	}

	/**
	 * Gets the emailFwd name associated with the result.
	 *
	 * @return EmailFwd name associated with the result if defined;
	 *         {@code null} otherwise.
	 */
	public String getName() {
		return this.name;
	}

	/**
	 * Gets if the emailFwd associated with {@code EPPEmailFwdCheckResult}
	 * is available.
	 *
	 * @return Is the emailFwd available?
	 */
	public boolean isAvailable() {
		return this.available;
	}

	/**
	 * Sets emailFwd reason.
	 *
	 * @param aReason
	 *           EmailFwd Reason.
	 */
	public void setEmailFwdReason(String aReason) {
		this.reason = aReason;
	}

	/**
	 * Sets if the emailFwd associated with {@code EPPEmailFwdCheckResult}
	 * is available.
	 *
	 * @param aIsAvailable
	 *           Is the emailFwd available?
	 */
	public void setIsAvailable(boolean aIsAvailable) {
		this.available = aIsAvailable;
	}

	/**
	 * Sets language attribute.
	 *
	 * @param aLang
	 *           Sets emailFwd reason language attribute.
	 */
	public void setLanguage(String aLang) {
		this.language = aLang;
	}

	/**
	 * Sets the emailFwd name associated with the result.
	 *
	 * @param aName
	 *           EmailFwd Name associated with the result.
	 */
	public void setName(String aName) {
		this.name = aName;
	}

	// End EPPEmailFwdCheckResult.clone()

	/**
	 * Implementation of {@code Object.toString}, which will result in an
	 * indented XML {@code String} representation of the concrete
	 * {@code EPPCodecComponent}.
	 *
	 * @return Indented XML {@code String} if successful; {@code ERROR}
	 *         otherwise.
	 */
	@Override
	public String toString() {
		return EPPUtil.toString(this);
	}

	/**
	 * Returns the XML namespace associated with the {@code EPPCodecComponent}.
	 * 
	 * @return XML namespace for the {@code EPPCodecComponent}. 
	 */
	@Override
	public String getNamespace() {
		return EPPEmailFwdMapFactory.NS;
	}

}
